"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FunctionArgsValidationError = exports.ChatFunctionClient = void 0;
var _lodash = require("lodash");
var _filter_function_definitions = require("../../../common/utils/filter_function_definitions");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
/* eslint-disable max-classes-per-file*/

class FunctionArgsValidationError extends Error {
  constructor(errors) {
    super('Function arguments are invalid');
    this.errors = errors;
  }
}
exports.FunctionArgsValidationError = FunctionArgsValidationError;
class ChatFunctionClient {
  constructor(contextRegistry, functionRegistry, validators) {
    this.contextRegistry = contextRegistry;
    this.functionRegistry = functionRegistry;
    this.validators = validators;
  }
  validate(name, parameters) {
    const validator = this.validators.get(name);
    const result = validator(parameters);
    if (!result) {
      throw new FunctionArgsValidationError(validator.errors);
    }
  }
  getContexts() {
    return Array.from(this.contextRegistry.values());
  }
  getFunctions({
    contexts,
    filter
  } = {}) {
    const allFunctions = Array.from(this.functionRegistry.values());
    const functionsByName = (0, _lodash.keyBy)(allFunctions, definition => definition.definition.name);
    const matchingDefinitions = (0, _filter_function_definitions.filterFunctionDefinitions)({
      contexts,
      filter,
      definitions: allFunctions.map(fn => fn.definition)
    });
    return matchingDefinitions.map(definition => functionsByName[definition.name]);
  }
  hasFunction(name) {
    return this.functionRegistry.has(name);
  }
  async executeFunction({
    name,
    args,
    messages,
    signal,
    connectorId
  }) {
    const fn = this.functionRegistry.get(name);
    if (!fn) {
      throw new Error(`Function ${name} not found`);
    }
    const parsedArguments = args ? JSON.parse(args) : {};
    this.validate(name, parsedArguments);
    return await fn.respond({
      arguments: parsedArguments,
      messages,
      connectorId
    }, signal);
  }
}
exports.ChatFunctionClient = ChatFunctionClient;