"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deleteFilterLists = deleteFilterLists;
var _dependency_cache = require("../../../../util/dependency_cache");
var _i18n = require("@kbn/i18n");
var _ml_api_service = require("../../../../services/ml_api_service");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function deleteFilterLists(filterListsToDelete) {
  if (filterListsToDelete === undefined || filterListsToDelete.length === 0) {
    return;
  }
  const toastNotifications = (0, _dependency_cache.getToastNotifications)();

  // Delete each of the specified filter lists in turn, waiting for each response
  // before deleting the next to minimize load on the cluster.
  toastNotifications.add(_i18n.i18n.translate('xpack.ml.settings.filterLists.deleteFilterLists.deletingNotificationMessage', {
    defaultMessage: 'Deleting {filterListsToDeleteLength, plural, one {{filterListToDeleteId}} other {# filter lists}}',
    values: {
      filterListsToDeleteLength: filterListsToDelete.length,
      filterListToDeleteId: !!filterListsToDelete.length && filterListsToDelete[0].filter_id
    }
  }));
  for (const filterList of filterListsToDelete) {
    const filterId = filterList.filter_id;
    try {
      await _ml_api_service.ml.filters.deleteFilter(filterId);
    } catch (resp) {
      console.log('Error deleting filter list:', resp);
      toastNotifications.addDanger(_i18n.i18n.translate('xpack.ml.settings.filterLists.deleteFilterLists.deletingErrorMessage', {
        defaultMessage: 'An error occurred deleting filter list {filterListId}{respMessage}',
        values: {
          filterListId: filterList.filter_id,
          respMessage: resp.message ? ` : ${resp.message}` : ''
        }
      }));
    }
  }
  toastNotifications.addSuccess(_i18n.i18n.translate('xpack.ml.settings.filterLists.deleteFilterLists.filtersSuccessfullyDeletedNotificationMessage', {
    defaultMessage: '{filterListsToDeleteLength, plural, one {{filterListToDeleteId}} other {# filter lists}} deleted',
    values: {
      filterListsToDeleteLength: filterListsToDelete.length,
      filterListToDeleteId: !!filterListsToDelete.length && filterListsToDelete[0].filter_id
    }
  }));
}