"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SavedMap = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = _interopRequireDefault(require("lodash"));
var _analytics = require("@kbn/analytics");
var _i18n = require("@kbn/i18n");
var _constants = require("../../../../common/constants");
var _store = require("../../../reducers/store");
var _map_selectors = require("../../../selectors/map_selectors");
var _actions = require("../../../actions");
var _ui_selectors = require("../../../selectors/ui_selectors");
var _map_attribute_service = require("../../../map_attribute_service");
var _kibana_services = require("../../../kibana_services");
var _copy_persistent_state = require("../../../reducers/copy_persistent_state");
var _get_breadcrumbs = require("./get_breadcrumbs");
var _ui = require("../../../reducers/ui");
var _create_basemap_layer_descriptor = require("../../../classes/layers/create_basemap_layer_descriptor");
var _licensed_features = require("../../../licensed_features");
var _ui_actions = require("../../../actions/ui_actions");
var _telemetry = require("../../../../common/telemetry");
var _index_pattern_util = require("../../../index_pattern_util");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function setMapSettingsFromEncodedState(settings) {
  const decodedCustomIcons = settings.customIcons ?
  // base64 decode svg string
  settings.customIcons.map(icon => {
    return {
      ...icon,
      svg: Buffer.from(icon.svg, 'base64').toString('utf-8')
    };
  }) : [];
  return (0, _actions.setMapSettings)({
    ...settings,
    customIcons: decodedCustomIcons
  });
}
class SavedMap {
  constructor({
    defaultLayers = [],
    mapEmbeddableInput,
    embeddableId,
    onSaveCallback,
    originatingApp,
    stateTransfer,
    originatingPath,
    defaultLayerWizard
  }) {
    (0, _defineProperty2.default)(this, "_attributes", null);
    (0, _defineProperty2.default)(this, "_sharingSavedObjectProps", null);
    (0, _defineProperty2.default)(this, "_defaultLayers", void 0);
    (0, _defineProperty2.default)(this, "_embeddableId", void 0);
    (0, _defineProperty2.default)(this, "_initialLayerListConfig", []);
    (0, _defineProperty2.default)(this, "_mapEmbeddableInput", void 0);
    (0, _defineProperty2.default)(this, "_onSaveCallback", void 0);
    (0, _defineProperty2.default)(this, "_originatingApp", void 0);
    (0, _defineProperty2.default)(this, "_originatingPath", void 0);
    (0, _defineProperty2.default)(this, "_stateTransfer", void 0);
    (0, _defineProperty2.default)(this, "_store", void 0);
    (0, _defineProperty2.default)(this, "_tags", []);
    (0, _defineProperty2.default)(this, "_defaultLayerWizard", void 0);
    (0, _defineProperty2.default)(this, "hasUnsavedChanges", () => {
      if (!this._attributes) {
        throw new Error('Invalid usage, must await whenReady before calling hasUnsavedChanges');
      }
      const savedLayerList = this._attributes.layerListJSON ? JSON.parse(this._attributes.layerListJSON) : null;
      const layerListConfigOnly = (0, _map_selectors.getLayerListConfigOnly)(this._store.getState());
      return !savedLayerList ? !_lodash.default.isEqual(layerListConfigOnly, this._initialLayerListConfig) :
      // savedMap stores layerList as a JSON string using JSON.stringify.
      // JSON.stringify removes undefined properties from objects.
      // savedMap.getLayerList converts the JSON string back into Javascript array of objects.
      // Need to perform the same process for layerListConfigOnly to compare apples to apples
      // and avoid undefined properties in layerListConfigOnly triggering unsaved changes.
      !_lodash.default.isEqual(JSON.parse(JSON.stringify(layerListConfigOnly)), savedLayerList);
    });
    (0, _defineProperty2.default)(this, "getAppNameFromId", appId => {
      return this._getStateTransfer().getAppNameFromId(appId);
    });
    this._defaultLayers = defaultLayers;
    this._mapEmbeddableInput = mapEmbeddableInput;
    this._embeddableId = embeddableId;
    this._onSaveCallback = onSaveCallback;
    this._originatingApp = originatingApp;
    this._originatingPath = originatingPath;
    this._stateTransfer = stateTransfer;
    this._store = (0, _store.createMapStore)();
    this._defaultLayerWizard = defaultLayerWizard || '';
  }
  getStore() {
    return this._store;
  }
  async whenReady() {
    var _this$_attributes, _this$_attributes2, _this$_attributes3, _this$_attributes4, _this$_attributes5;
    await (0, _licensed_features.whenLicenseInitialized)();
    if (!this._mapEmbeddableInput) {
      this._attributes = {
        title: '',
        description: ''
      };
    } else {
      const {
        attributes: doc,
        metaInfo
      } = await (0, _map_attribute_service.getMapAttributeService)().unwrapAttributes(this._mapEmbeddableInput);
      const {
        references,
        ...savedObjectAttributes
      } = doc;
      this._attributes = savedObjectAttributes;
      if (metaInfo !== null && metaInfo !== void 0 && metaInfo.sharingSavedObjectProps) {
        this._sharingSavedObjectProps = metaInfo.sharingSavedObjectProps;
      }
      const savedObjectsTagging = (0, _kibana_services.getSavedObjectsTagging)();
      if (savedObjectsTagging && references && references.length) {
        this._tags = savedObjectsTagging.ui.getTagIdsFromReferences(references);
      }
    }
    this._reportUsage();
    if ((_this$_attributes = this._attributes) !== null && _this$_attributes !== void 0 && _this$_attributes.mapStateJSON) {
      try {
        const mapState = JSON.parse(this._attributes.mapStateJSON);
        if (mapState.adHocDataViews && mapState.adHocDataViews.length > 0) {
          const dataViewService = (0, _kibana_services.getIndexPatternService)();
          const promises = mapState.adHocDataViews.map(spec => {
            return dataViewService.create(spec);
          });
          await Promise.all(promises);
        }
      } catch (e) {
        // ignore malformed mapStateJSON, not a critical error for viewing map - map will just use defaults
      }
    }
    if (this._mapEmbeddableInput && this._mapEmbeddableInput.mapSettings !== undefined) {
      this._store.dispatch(setMapSettingsFromEncodedState(this._mapEmbeddableInput.mapSettings));
    } else if ((_this$_attributes2 = this._attributes) !== null && _this$_attributes2 !== void 0 && _this$_attributes2.mapStateJSON) {
      try {
        const mapState = JSON.parse(this._attributes.mapStateJSON);
        if (mapState.settings) {
          this._store.dispatch(setMapSettingsFromEncodedState(mapState.settings));
        }
      } catch (e) {
        // ignore malformed mapStateJSON, not a critical error for viewing map - map will just use defaults
      }
    }
    let isLayerTOCOpen = _ui.DEFAULT_IS_LAYER_TOC_OPEN;
    if (this._mapEmbeddableInput && this._mapEmbeddableInput.isLayerTOCOpen !== undefined) {
      isLayerTOCOpen = this._mapEmbeddableInput.isLayerTOCOpen;
    } else if ((_this$_attributes3 = this._attributes) !== null && _this$_attributes3 !== void 0 && _this$_attributes3.uiStateJSON) {
      try {
        const uiState = JSON.parse(this._attributes.uiStateJSON);
        if ('isLayerTOCOpen' in uiState) {
          isLayerTOCOpen = uiState.isLayerTOCOpen;
        }
      } catch (e) {
        // ignore malformed uiStateJSON, not a critical error for viewing map - map will just use defaults
      }
    }
    this._store.dispatch((0, _actions.setIsLayerTOCOpen)(isLayerTOCOpen));
    let openTOCDetails = [];
    if (this._mapEmbeddableInput && this._mapEmbeddableInput.openTOCDetails !== undefined) {
      openTOCDetails = this._mapEmbeddableInput.openTOCDetails;
    } else if ((_this$_attributes4 = this._attributes) !== null && _this$_attributes4 !== void 0 && _this$_attributes4.uiStateJSON) {
      try {
        const uiState = JSON.parse(this._attributes.uiStateJSON);
        if ('openTOCDetails' in uiState) {
          openTOCDetails = uiState.openTOCDetails;
        }
      } catch (e) {
        // ignore malformed uiStateJSON, not a critical error for viewing map - map will just use defaults
      }
    }
    this._store.dispatch((0, _actions.setOpenTOCDetails)(openTOCDetails));
    if (this._mapEmbeddableInput && this._mapEmbeddableInput.mapCenter !== undefined) {
      this._store.dispatch((0, _actions.setGotoWithCenter)({
        lat: this._mapEmbeddableInput.mapCenter.lat,
        lon: this._mapEmbeddableInput.mapCenter.lon,
        zoom: this._mapEmbeddableInput.mapCenter.zoom
      }));
    } else if ((_this$_attributes5 = this._attributes) !== null && _this$_attributes5 !== void 0 && _this$_attributes5.mapStateJSON) {
      try {
        const mapState = JSON.parse(this._attributes.mapStateJSON);
        this._store.dispatch((0, _actions.setGotoWithCenter)({
          lat: mapState.center.lat,
          lon: mapState.center.lon,
          zoom: mapState.zoom
        }));
      } catch (e) {
        // ignore malformed mapStateJSON, not a critical error for viewing map - map will just use defaults
      }
    }
    let layerList = [];
    if (this._attributes.layerListJSON) {
      try {
        layerList = JSON.parse(this._attributes.layerListJSON);
      } catch (e) {
        throw new Error('Malformed saved object: unable to parse layerListJSON');
      }
    } else {
      const basemapLayerDescriptor = (0, _create_basemap_layer_descriptor.createBasemapLayerDescriptor)();
      if (basemapLayerDescriptor) {
        layerList.push(basemapLayerDescriptor);
      }
      if (this._defaultLayers.length) {
        layerList.push(...this._defaultLayers);
      }
    }
    this._store.dispatch((0, _actions.replaceLayerList)(layerList));
    if (this._mapEmbeddableInput && this._mapEmbeddableInput.hiddenLayers !== undefined) {
      this._store.dispatch((0, _actions.setHiddenLayers)(this._mapEmbeddableInput.hiddenLayers));
    }
    this._initialLayerListConfig = (0, _copy_persistent_state.copyPersistentState)(layerList);
    if (this._defaultLayerWizard) {
      this._store.dispatch((0, _ui_actions.setAutoOpenLayerWizardId)(this._defaultLayerWizard));
    }
  }
  _getStateTransfer() {
    if (!this._stateTransfer) {
      throw new Error('stateTransfer not provided in constructor');
    }
    return this._stateTransfer;
  }
  _getPageTitle() {
    if (!this._mapEmbeddableInput) {
      return _i18n.i18n.translate('xpack.maps.breadcrumbsCreate', {
        defaultMessage: 'Create'
      });
    }
    return this.isByValue() ? _i18n.i18n.translate('xpack.maps.breadcrumbsEditByValue', {
      defaultMessage: 'Edit map'
    }) : this._attributes.title;
  }
  _reportUsage() {
    const usageCollector = (0, _kibana_services.getUsageCollection)();
    if (!usageCollector || !this._attributes) {
      return;
    }
    const mapSettingsStatsCollector = new _telemetry.MapSettingsCollector(this._attributes);
    const layerStatsCollector = new _telemetry.LayerStatsCollector(this._attributes);
    const uiCounterEvents = {
      layer: layerStatsCollector.getLayerCounts(),
      scaling: layerStatsCollector.getScalingCounts(),
      resolution: layerStatsCollector.getResolutionCounts(),
      join: layerStatsCollector.getJoinCounts(),
      ems_basemap: layerStatsCollector.getBasemapCounts(),
      settings: {
        custom_icons_count: mapSettingsStatsCollector.getCustomIconsCount()
      }
    };
    for (const [eventType, eventTypeMetrics] of Object.entries(uiCounterEvents)) {
      for (const [eventName, count] of Object.entries(eventTypeMetrics)) {
        usageCollector.reportUiCounter(_constants.APP_ID, _analytics.METRIC_TYPE.LOADED, `${eventType}_${eventName}`, count);
      }
    }
  }
  setBreadcrumbs(history) {
    if (!this._attributes) {
      throw new Error('Invalid usage, must await whenReady before calling hasUnsavedChanges');
    }
    if ((0, _kibana_services.getServerless)()) {
      // TODO: https://github.com/elastic/kibana/issues/163488
      // for now, serverless breadcrumbs only set the title,
      // the rest of the breadcrumbs are handled by the serverless navigation
      // the serverless navigation is not yet aware of the byValue/originatingApp context
      (0, _kibana_services.getServerless)().setBreadcrumbs({
        text: this._getPageTitle()
      });
    } else {
      const breadcrumbs = (0, _get_breadcrumbs.getBreadcrumbs)({
        pageTitle: this._getPageTitle(),
        isByValue: this.isByValue(),
        getHasUnsavedChanges: this.hasUnsavedChanges,
        originatingApp: this._originatingApp,
        getAppNameFromId: this._getStateTransfer().getAppNameFromId,
        history
      });
      (0, _kibana_services.getCoreChrome)().setBreadcrumbs(breadcrumbs);
    }
  }
  getSavedObjectId() {
    return this._mapEmbeddableInput && 'savedObjectId' in this._mapEmbeddableInput ? this._mapEmbeddableInput.savedObjectId : undefined;
  }
  getOriginatingApp() {
    return this._originatingApp;
  }
  getOriginatingAppName() {
    return this._originatingApp ? this.getAppNameFromId(this._originatingApp) : undefined;
  }
  hasOriginatingApp() {
    return !!this._originatingApp;
  }
  getOriginatingPath() {
    return this._originatingPath;
  }
  getTags() {
    return this._tags;
  }
  hasSaveAndReturnConfig() {
    const hasOriginatingApp = this.hasOriginatingApp();
    const isNewMap = !this.getSavedObjectId();
    return (0, _kibana_services.getIsAllowByValueEmbeddables)() ? hasOriginatingApp : !isNewMap && hasOriginatingApp;
  }
  getTitle() {
    if (!this._attributes) {
      throw new Error('Invalid usage, must await getTitle before calling getAttributes');
    }
    return this._attributes.title !== undefined ? this._attributes.title : '';
  }
  getAttributes() {
    if (!this._attributes) {
      throw new Error('Invalid usage, must await whenReady before calling getAttributes');
    }
    return this._attributes;
  }
  getAutoFitToBounds() {
    var _this$_mapEmbeddableI, _this$_mapEmbeddableI2;
    if (this._mapEmbeddableInput && ((_this$_mapEmbeddableI = this._mapEmbeddableInput) === null || _this$_mapEmbeddableI === void 0 ? void 0 : (_this$_mapEmbeddableI2 = _this$_mapEmbeddableI.mapSettings) === null || _this$_mapEmbeddableI2 === void 0 ? void 0 : _this$_mapEmbeddableI2.autoFitToDataBounds) !== undefined) {
      return this._mapEmbeddableInput.mapSettings.autoFitToDataBounds;
    }
    if (!this._attributes || !this._attributes.mapStateJSON) {
      return false;
    }
    try {
      const mapState = JSON.parse(this._attributes.mapStateJSON);
      if ((mapState === null || mapState === void 0 ? void 0 : mapState.settings.autoFitToDataBounds) !== undefined) {
        return mapState.settings.autoFitToDataBounds;
      }
    } catch (e) {
      // ignore malformed mapStateJSON, not a critical error for viewing map - map will just use defaults
    }
    return false;
  }
  getSharingSavedObjectProps() {
    return this._sharingSavedObjectProps;
  }
  isByValue() {
    const hasSavedObjectId = !!this.getSavedObjectId();
    return (0, _kibana_services.getIsAllowByValueEmbeddables)() && !!this._originatingApp && !hasSavedObjectId;
  }
  async save({
    newDescription,
    newTitle,
    newCopyOnSave,
    returnToOrigin,
    newTags,
    saveByReference,
    dashboardId,
    history
  }) {
    if (!this._attributes) {
      throw new Error('Invalid usage, must await whenReady before calling save');
    }
    const prevTitle = this._attributes.title;
    const prevDescription = this._attributes.description;
    const prevTags = this._tags;
    this._attributes.title = newTitle;
    this._attributes.description = newDescription;
    if (newTags) {
      this._tags = newTags;
    }
    await this._syncAttributesWithStore();
    let updatedMapEmbeddableInput;
    try {
      const savedObjectsTagging = (0, _kibana_services.getSavedObjectsTagging)();
      // Attribute service deviates from Saved Object client by including references as a child to attributes in stead of a sibling
      const attributes = savedObjectsTagging && newTags ? {
        ...this._attributes,
        references: savedObjectsTagging.ui.updateTagsReferences([], newTags)
      } : this._attributes;
      updatedMapEmbeddableInput = await (0, _map_attribute_service.getMapAttributeService)().wrapAttributes(attributes, saveByReference, newCopyOnSave ? undefined : this._mapEmbeddableInput);
    } catch (e) {
      // Error toast displayed by wrapAttributes
      this._attributes.title = prevTitle;
      this._attributes.description = prevDescription;
      this._tags = prevTags;
      return;
    }
    if (returnToOrigin) {
      if (!this._originatingApp) {
        (0, _kibana_services.getToasts)().addDanger({
          title: _i18n.i18n.translate('xpack.maps.topNav.saveErrorTitle', {
            defaultMessage: `Error saving '{title}'`,
            values: {
              title: newTitle
            }
          }),
          text: _i18n.i18n.translate('xpack.maps.topNav.saveErrorText', {
            defaultMessage: 'Unable to return to app without an originating app'
          })
        });
        return;
      }
      await this._getStateTransfer().navigateToWithEmbeddablePackage(this._originatingApp, {
        state: {
          embeddableId: newCopyOnSave ? undefined : this._embeddableId,
          type: _constants.MAP_SAVED_OBJECT_TYPE,
          input: updatedMapEmbeddableInput
        },
        path: this._originatingPath
      });
      return;
    } else if (dashboardId) {
      await this._getStateTransfer().navigateToWithEmbeddablePackage('dashboards', {
        state: {
          type: _constants.MAP_SAVED_OBJECT_TYPE,
          input: updatedMapEmbeddableInput
        },
        path: dashboardId === 'new' ? '#/create' : `#/view/${dashboardId}`
      });
      return;
    }
    this._mapEmbeddableInput = updatedMapEmbeddableInput;
    // break connection to originating application
    this._originatingApp = undefined;

    // remove editor state so the connection is still broken after reload
    this._getStateTransfer().clearEditorState(_constants.APP_ID);
    (0, _kibana_services.getToasts)().addSuccess({
      title: _i18n.i18n.translate('xpack.maps.topNav.saveSuccessMessage', {
        defaultMessage: `Saved '{title}'`,
        values: {
          title: newTitle
        }
      })
    });
    (0, _kibana_services.getCoreChrome)().docTitle.change(newTitle);
    this.setBreadcrumbs(history);
    history.push(`/${_constants.MAP_PATH}/${this.getSavedObjectId()}${window.location.hash}`);
    if (this._onSaveCallback) {
      this._onSaveCallback();
    }
    return;
  }
  async _syncAttributesWithStore() {
    const state = this._store.getState();
    const layerList = (0, _map_selectors.getLayerListRaw)(state);
    const layerListConfigOnly = (0, _copy_persistent_state.copyPersistentState)(layerList);
    this._attributes.layerListJSON = JSON.stringify(layerListConfigOnly);
    const mapSettings = (0, _map_selectors.getMapSettings)(state);
    this._attributes.mapStateJSON = JSON.stringify({
      adHocDataViews: await this._getAdHocDataViews(),
      zoom: (0, _map_selectors.getMapZoom)(state),
      center: (0, _map_selectors.getMapCenter)(state),
      timeFilters: (0, _map_selectors.getTimeFilters)(state),
      refreshConfig: {
        isPaused: (0, _kibana_services.getTimeFilter)().getRefreshInterval().pause,
        interval: (0, _kibana_services.getTimeFilter)().getRefreshInterval().value
      },
      query: (0, _map_selectors.getQuery)(state),
      filters: (0, _map_selectors.getFilters)(state),
      settings: {
        ...mapSettings,
        // base64 encode custom icons to avoid svg strings breaking saved object stringification/parsing.
        customIcons: mapSettings.customIcons.map(icon => {
          return {
            ...icon,
            svg: Buffer.from(icon.svg).toString('base64')
          };
        })
      }
    });
    this._attributes.uiStateJSON = JSON.stringify({
      isLayerTOCOpen: (0, _ui_selectors.getIsLayerTOCOpen)(state),
      openTOCDetails: (0, _ui_selectors.getOpenTOCDetails)(state)
    });
  }
  async _getAdHocDataViews() {
    const dataViewIds = [];
    (0, _map_selectors.getLayerList)(this._store.getState()).forEach(layer => {
      dataViewIds.push(...layer.getIndexPatternIds());
    });
    const dataViews = await (0, _index_pattern_util.getIndexPatternsFromIds)(_lodash.default.uniq(dataViewIds));
    return dataViews.filter(dataView => {
      return !dataView.isPersisted();
    }).map(dataView => {
      return dataView.toSpec(false);
    });
  }
}
exports.SavedMap = SavedMap;