"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.EmsVectorTileLayer = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _react = _interopRequireDefault(require("react"));
var _emsClient = require("@elastic/ems-client");
var _i18n = require("@kbn/i18n");
var _lodash = _interopRequireDefault(require("lodash"));
var _eui = require("@elastic/eui");
var _image_utils = require("./image_utils");
var _layer = require("../layer");
var _constants = require("../../../../common/constants");
var _util = require("../../../util");
var _ems_vector_tile_style = require("../../styles/ems/ems_vector_tile_style");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class EmsVectorTileLayer extends _layer.AbstractLayer {
  static createDescriptor(options) {
    const emsVectorTileLayerDescriptor = super.createDescriptor(options);
    emsVectorTileLayerDescriptor.type = _constants.LAYER_TYPE.EMS_VECTOR_TILE;
    emsVectorTileLayerDescriptor.alpha = _lodash.default.get(options, 'alpha', 1);
    emsVectorTileLayerDescriptor.locale = _lodash.default.get(options, 'locale', _constants.AUTOSELECT_EMS_LOCALE);
    emsVectorTileLayerDescriptor.style = _ems_vector_tile_style.EMSVectorTileStyle.createDescriptor();
    return emsVectorTileLayerDescriptor;
  }
  constructor({
    source,
    layerDescriptor
  }) {
    super({
      source,
      layerDescriptor
    });
    (0, _defineProperty2.default)(this, "_style", void 0);
    if (!layerDescriptor.style) {
      const defaultStyle = _ems_vector_tile_style.EMSVectorTileStyle.createDescriptor();
      this._style = new _ems_vector_tile_style.EMSVectorTileStyle(defaultStyle);
    } else {
      this._style = new _ems_vector_tile_style.EMSVectorTileStyle(layerDescriptor.style);
    }
  }
  _isTiled() {
    // Uses tiled maplibre source 'vector'
    return true;
  }
  getSource() {
    return super.getSource();
  }
  getStyleForEditing() {
    return this._style;
  }
  getStyle() {
    return this._style;
  }
  getCurrentStyle() {
    return this._style;
  }
  getLocale() {
    var _this$_descriptor$loc;
    return (_this$_descriptor$loc = this._descriptor.locale) !== null && _this$_descriptor$loc !== void 0 ? _this$_descriptor$loc : _constants.NO_EMS_LOCALE;
  }
  _canSkipSync({
    prevDataRequest,
    nextMeta
  }) {
    if (!prevDataRequest) {
      return false;
    }
    const prevMeta = prevDataRequest.getMeta();
    if (!prevMeta) {
      return false;
    }
    return prevMeta.tileLayerId === nextMeta.tileLayerId;
  }
  async syncData({
    startLoading,
    stopLoading,
    onLoadError
  }) {
    const nextMeta = {
      tileLayerId: this.getSource().getTileLayerId()
    };
    const canSkipSync = this._canSkipSync({
      prevDataRequest: this.getSourceDataRequest(),
      nextMeta
    });
    if (canSkipSync) {
      return;
    }
    const requestToken = Symbol(`layer-source-refresh:${this.getId()} - source`);
    try {
      startLoading(_constants.SOURCE_DATA_REQUEST_ID, requestToken, nextMeta);
      const styleAndSprites = await this.getSource().getVectorStyleSheetAndSpriteMeta((0, _util.isRetina)());
      const spriteSheetImageData = styleAndSprites.spriteMeta ? await this._loadSpriteSheetImageData(styleAndSprites.spriteMeta.png) : undefined;
      const data = {
        ...styleAndSprites,
        spriteSheetImageData
      };
      stopLoading(_constants.SOURCE_DATA_REQUEST_ID, requestToken, data);
    } catch (error) {
      onLoadError(_constants.SOURCE_DATA_REQUEST_ID, requestToken, error);
    }
  }
  _generateMbId(name) {
    return `${this.getId()}_${name}`;
  }
  _generateMbSourceIdPrefix() {
    const DELIMITTER = '___';
    return `${this.getId()}${DELIMITTER}${this.getSource().getTileLayerId()}${DELIMITTER}`;
  }
  _generateMbSourceId(name) {
    return `${this._generateMbSourceIdPrefix()}${name}`;
  }
  _getVectorStyle() {
    var _sourceDataRequest$ge;
    const sourceDataRequest = this.getSourceDataRequest();
    if (!sourceDataRequest) {
      return null;
    }
    return (_sourceDataRequest$ge = sourceDataRequest.getData()) === null || _sourceDataRequest$ge === void 0 ? void 0 : _sourceDataRequest$ge.vectorStyleSheet;
  }
  _getSpriteMeta() {
    var _sourceDataRequest$ge2;
    const sourceDataRequest = this.getSourceDataRequest();
    if (!sourceDataRequest) {
      return null;
    }
    return (_sourceDataRequest$ge2 = sourceDataRequest.getData()) === null || _sourceDataRequest$ge2 === void 0 ? void 0 : _sourceDataRequest$ge2.spriteMeta;
  }
  _getSpriteImageData() {
    var _sourceDataRequest$ge3;
    const sourceDataRequest = this.getSourceDataRequest();
    if (!sourceDataRequest) {
      return null;
    }
    return (_sourceDataRequest$ge3 = sourceDataRequest.getData()) === null || _sourceDataRequest$ge3 === void 0 ? void 0 : _sourceDataRequest$ge3.spriteSheetImageData;
  }
  getMbLayerIds() {
    const vectorStyle = this._getVectorStyle();
    if (!vectorStyle || !vectorStyle.layers) {
      return [];
    }
    return vectorStyle.layers.map(layer => this._generateMbId(layer.id));
  }
  getMbSourceIds() {
    const vectorStyle = this._getVectorStyle();
    if (!vectorStyle || !vectorStyle.sources) {
      return [];
    }
    const sourceIds = Object.keys(vectorStyle.sources);
    return sourceIds.map(sourceId => this._generateMbSourceId(sourceId));
  }
  ownsMbLayerId(mbLayerId) {
    return mbLayerId.startsWith(this.getId());
  }
  ownsMbSourceId(mbSourceId) {
    return mbSourceId.startsWith(this.getId());
  }
  _makeNamespacedImageId(imageId) {
    const prefix = this.getSource().getSpriteNamespacePrefix() + '/';
    return prefix + imageId;
  }
  _requiresPrevSourceCleanup(mbMap) {
    const sourceIdPrefix = this._generateMbSourceIdPrefix();
    const mbStyle = mbMap.getStyle();
    if (!mbStyle.sources) {
      return false;
    }
    return Object.keys(mbStyle.sources).some(mbSourceId => {
      const doesMbSourceBelongToLayer = this.ownsMbSourceId(mbSourceId);
      const doesMbSourceBelongToSource = mbSourceId.startsWith(sourceIdPrefix);
      return doesMbSourceBelongToLayer && !doesMbSourceBelongToSource;
    });
  }
  _getImageData(img) {
    const canvas = window.document.createElement('canvas');
    const context = canvas.getContext('2d');
    if (!context) {
      throw new Error('failed to create canvas 2d context');
    }
    canvas.width = img.width;
    canvas.height = img.height;
    context.drawImage(img, 0, 0, img.width, img.height);
    return context.getImageData(0, 0, img.width, img.height);
  }
  _isCrossOriginUrl(url) {
    const a = window.document.createElement('a');
    a.href = url;
    return a.protocol !== window.document.location.protocol || a.host !== window.document.location.host || a.port !== window.document.location.port;
  }
  _loadSpriteSheetImageData(imgUrl) {
    return new Promise((resolve, reject) => {
      const image = new Image();
      if (this._isCrossOriginUrl(imgUrl)) {
        image.crossOrigin = 'Anonymous';
      }
      image.onload = event => {
        resolve(this._getImageData(image));
      };
      image.onerror = e => {
        reject(e);
      };
      image.src = imgUrl;
    });
  }
  _addSpriteSheetToMapFromImageData(json, imgData, mbMap) {
    for (const imageId in json) {
      if (!(json.hasOwnProperty(imageId) && !mbMap.hasImage(imageId))) {
        continue;
      }
      const {
        width,
        height,
        x,
        y,
        sdf,
        pixelRatio
      } = json[imageId];
      if (typeof width !== 'number' || typeof height !== 'number') {
        continue;
      }
      const data = new _image_utils.RGBAImage({
        width,
        height
      });
      _image_utils.RGBAImage.copy(imgData, data, {
        x,
        y
      }, {
        x: 0,
        y: 0
      }, {
        width,
        height
      });
      mbMap.addImage(imageId, data, {
        pixelRatio,
        sdf
      });
    }
  }
  syncLayerWithMB(mbMap) {
    const vectorStyle = this._getVectorStyle();
    if (!vectorStyle) {
      return;
    }
    this._removeStaleMbSourcesAndLayers(mbMap);
    let initialBootstrapCompleted = false;
    const sourceIds = vectorStyle.sources ? Object.keys(vectorStyle.sources) : [];
    sourceIds.forEach(sourceId => {
      if (initialBootstrapCompleted) {
        return;
      }
      const mbSourceId = this._generateMbSourceId(sourceId);
      const mbSource = mbMap.getSource(mbSourceId);
      if (mbSource) {
        // if a single source is present, the layer already has bootstrapped with the mbMap
        initialBootstrapCompleted = true;
        return;
      }
      mbMap.addSource(mbSourceId, vectorStyle.sources[sourceId]);
    });
    if (!initialBootstrapCompleted) {
      // sync spritesheet
      const spriteMeta = this._getSpriteMeta();
      if (!spriteMeta) {
        return;
      }
      const newJson = {};
      for (const imageId in spriteMeta.json) {
        if (spriteMeta.json.hasOwnProperty(imageId)) {
          const namespacedImageId = this._makeNamespacedImageId(imageId);
          newJson[namespacedImageId] = spriteMeta.json[imageId];
        }
      }
      const imageData = this._getSpriteImageData();
      if (!imageData) {
        return;
      }
      this._addSpriteSheetToMapFromImageData(newJson, imageData, mbMap);
      // sync layers
      const layers = vectorStyle.layers ? vectorStyle.layers : [];
      layers.forEach(layer => {
        const mbLayerId = this._generateMbId(layer.id);
        const mbLayer = mbMap.getLayer(mbLayerId);
        if (mbLayer) {
          return;
        }
        const newLayerObject = {
          ...layer,
          source: this._generateMbSourceId('source' in layer && typeof layer.source === 'string' ? layer.source : undefined),
          id: mbLayerId
        };
        if (newLayerObject.type === 'symbol' && newLayerObject.layout && typeof newLayerObject.layout['icon-image'] === 'string') {
          newLayerObject.layout['icon-image'] = this._makeNamespacedImageId(newLayerObject.layout['icon-image']);
        }
        if (newLayerObject.type === 'fill' && newLayerObject.paint && typeof newLayerObject.paint['fill-pattern'] === 'string') {
          newLayerObject.paint['fill-pattern'] = this._makeNamespacedImageId(newLayerObject.paint['fill-pattern']);
        }
        mbMap.addLayer(newLayerObject);
      });
    }
    this._setTileLayerProperties(mbMap);
  }
  _getOpacityProps(layerType) {
    if (layerType === 'fill') {
      return ['fill-opacity'];
    }
    if (layerType === 'line') {
      return ['line-opacity'];
    }
    if (layerType === 'circle') {
      return ['circle-opacity'];
    }
    if (layerType === 'background') {
      return ['background-opacity'];
    }
    if (layerType === 'symbol') {
      return ['icon-opacity', 'text-opacity'];
    }
    return [];
  }
  _setColorFilter(mbMap, mbLayer, mbLayerId) {
    const color = this.getCurrentStyle().getColor();
    const colorOperation = _emsClient.TMSService.colorOperationDefaults.find(({
      style
    }) => {
      return style === this.getSource().getTileLayerId();
    });
    if (!colorOperation) return;
    const {
      operation,
      percentage
    } = colorOperation;
    const properties = _emsClient.TMSService.transformColorProperties(mbLayer, color, operation, percentage);
    for (const {
      property,
      color: newColor
    } of properties) {
      mbMap.setPaintProperty(mbLayerId, property, newColor);
    }
  }
  _setOpacityForType(mbMap, mbLayer, mbLayerId) {
    this._getOpacityProps(mbLayer.type).forEach(opacityProp => {
      const mbPaint = mbLayer.paint;
      if (mbPaint && typeof mbPaint[opacityProp] === 'number') {
        const newOpacity = mbPaint[opacityProp] * this.getAlpha();
        mbMap.setPaintProperty(mbLayerId, opacityProp, newOpacity);
      } else {
        mbMap.setPaintProperty(mbLayerId, opacityProp, this.getAlpha());
      }
    });
  }
  _setLanguage(mbMap, mbLayer, mbLayerId) {
    const locale = this.getLocale();
    if (locale === null || locale === _constants.NO_EMS_LOCALE) {
      var _mbLayer$layout;
      if (mbLayer.type !== 'symbol') return;
      const textProperty = (_mbLayer$layout = mbLayer.layout) === null || _mbLayer$layout === void 0 ? void 0 : _mbLayer$layout['text-field'];
      if (mbLayer.layout && textProperty) {
        mbMap.setLayoutProperty(mbLayerId, 'text-field', textProperty);
      }
      return;
    }
    const textProperty = locale === _constants.AUTOSELECT_EMS_LOCALE ? _emsClient.TMSService.transformLanguageProperty(mbLayer, _i18n.i18n.getLocale()) : _emsClient.TMSService.transformLanguageProperty(mbLayer, locale);
    if (textProperty !== undefined) {
      mbMap.setLayoutProperty(mbLayerId, 'text-field', textProperty);
    }
  }
  _setLayerZoomRange(mbMap, mbLayer, mbLayerId) {
    let minZoom = this.getMinZoom();
    if (typeof mbLayer.minzoom === 'number') {
      minZoom = Math.max(minZoom, mbLayer.minzoom);
    }
    let maxZoom = this.getMaxZoom();
    if (typeof mbLayer.maxzoom === 'number') {
      maxZoom = Math.min(maxZoom, mbLayer.maxzoom);
    }
    mbMap.setLayerZoomRange(mbLayerId, minZoom, maxZoom);
  }
  _setTileLayerProperties(mbMap) {
    const vectorStyle = this._getVectorStyle();
    if (!vectorStyle || !vectorStyle.layers) {
      return;
    }
    vectorStyle.layers.forEach(mbLayer => {
      const mbLayerId = this._generateMbId(mbLayer.id);
      this.syncVisibilityWithMb(mbMap, mbLayerId);
      this._setLayerZoomRange(mbMap, mbLayer, mbLayerId);
      this._setOpacityForType(mbMap, mbLayer, mbLayerId);
      this._setColorFilter(mbMap, mbLayer, mbLayerId);
      this._setLanguage(mbMap, mbLayer, mbLayerId);
    });
  }
  areLabelsOnTop() {
    return !!this._descriptor.areLabelsOnTop;
  }
  supportsLabelsOnTop() {
    return true;
  }
  supportsLabelLocales() {
    return true;
  }
  async getLicensedFeatures() {
    return this._source.getLicensedFeatures();
  }
  getLayerTypeIconName() {
    return 'grid';
  }
  getLayerIcon() {
    return {
      icon: /*#__PURE__*/_react.default.createElement(_eui.EuiIcon, {
        size: "m",
        type: "grid"
      }),
      tooltipContent: _i18n.i18n.translate('xpack.maps.emsVectorTileLayer.layerDescription', {
        defaultMessage: `Reference map provided by Elastic Maps Service (EMS).`
      })
    };
  }
  isBasemap(order) {
    return order === 0;
  }
}
exports.EmsVectorTileLayer = EmsVectorTileLayer;