"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.suggestions = suggestions;
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
var _constants = require("../../../common/constants");
var _visualizations = require("../../../common/visualizations");
var _partition_charts_meta = require("./partition_charts_meta");
var _ = require("../..");
var _utils = require("../../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function hasIntervalScale(columns) {
  return columns.some(col => col.operation.scale === 'interval');
}
function shouldReject({
  table,
  keptLayerIds,
  state
}) {
  return keptLayerIds.length > 1 || keptLayerIds.length && table.layerId !== keptLayerIds[0] || table.changeType === 'reorder' || table.columns.some(col => col.operation.isStaticValue);
}
function getNewShape(groups, subVisualizationId) {
  var _newShape;
  if (subVisualizationId) {
    return subVisualizationId;
  }
  let newShape;
  if (groups.length !== 1 && !subVisualizationId) {
    newShape = _constants.PieChartTypes.PIE;
  }
  return (_newShape = newShape) !== null && _newShape !== void 0 ? _newShape : _constants.PieChartTypes.DONUT;
}
function hasCustomSuggestionsExists(shape) {
  const shapes = [_constants.PieChartTypes.TREEMAP, _constants.PieChartTypes.WAFFLE, _constants.PieChartTypes.MOSAIC];
  return shape ? shapes.includes(shape) : false;
}
const maximumGroupLength = Math.max(...Object.values(_partition_charts_meta.PartitionChartsMeta).map(({
  maxBuckets
}) => maxBuckets));
function suggestions({
  table,
  state,
  keptLayerIds,
  mainPalette,
  subVisualizationId
}) {
  if (shouldReject({
    table,
    state,
    keptLayerIds
  })) {
    return [];
  }
  const isActive = Boolean(state);
  const [groups, metrics] = (0, _lodash.partition)(
  // filter out all metrics which are not number based
  table.columns.filter(col => col.operation.isBucketed || col.operation.dataType === 'number'), col => col.operation.isBucketed);
  if (groups.length === 0 && metrics.length === 0) {
    return [];
  }
  if (metrics.length > 1 && !isActive || groups.length > maximumGroupLength) {
    return [];
  }
  if (metrics.length > 1 && !(state !== null && state !== void 0 && state.layers[0].allowMultipleMetrics)) {
    return [];
  }
  const incompleteConfiguration = metrics.length === 0 || groups.length === 0;
  if (incompleteConfiguration && state && !subVisualizationId) {
    // reject incomplete configurations if the sub visualization isn't specifically requested
    // this allows to switch chart types via switcher with incomplete configurations, but won't
    // cause incomplete suggestions getting auto applied on dropped fields
    return [];
  }
  const metricColumnIds = metrics.map(({
    columnId
  }) => columnId);
  const results = [];

  // Histograms are not good for pi. But we should not hide suggestion on switching between partition charts.
  const shouldHideSuggestion = state !== null && state !== void 0 && state.shape && (0, _visualizations.isPartitionShape)(state.shape) ? false : hasIntervalScale(table.columns);
  if (groups.length <= _partition_charts_meta.PartitionChartsMeta.pie.maxBuckets && !hasCustomSuggestionsExists(subVisualizationId)) {
    const newShape = getNewShape(groups, subVisualizationId);
    const baseSuggestion = {
      title: _i18n.i18n.translate('xpack.lens.pie.suggestionLabel', {
        defaultMessage: '{chartName}',
        values: {
          chartName: _partition_charts_meta.PartitionChartsMeta[newShape].label
        },
        description: 'chartName is already translated'
      }),
      score: state && !hasCustomSuggestionsExists(state.shape) ? 0.6 : 0.4,
      state: {
        shape: newShape,
        palette: (mainPalette === null || mainPalette === void 0 ? void 0 : mainPalette.type) === 'legacyPalette' ? mainPalette.value : state === null || state === void 0 ? void 0 : state.palette,
        layers: [state !== null && state !== void 0 && state.layers[0] ? {
          ...state.layers[0],
          layerId: table.layerId,
          primaryGroups: groups.map(col => col.columnId),
          metrics: metricColumnIds,
          layerType: _.layerTypes.DATA,
          colorMapping: !mainPalette ? (0, _utils.getColorMappingDefaults)() : (mainPalette === null || mainPalette === void 0 ? void 0 : mainPalette.type) === 'colorMapping' ? mainPalette.value : state.layers[0].colorMapping
        } : {
          layerId: table.layerId,
          primaryGroups: groups.map(col => col.columnId),
          metrics: metricColumnIds,
          numberDisplay: _constants.NumberDisplay.PERCENT,
          categoryDisplay: _constants.CategoryDisplay.DEFAULT,
          legendDisplay: _constants.LegendDisplay.DEFAULT,
          nestedLegend: false,
          layerType: _.layerTypes.DATA,
          colorMapping: !mainPalette ? (0, _utils.getColorMappingDefaults)() : (mainPalette === null || mainPalette === void 0 ? void 0 : mainPalette.type) === 'colorMapping' ? mainPalette.value : undefined
        }]
      },
      previewIcon: _partition_charts_meta.PartitionChartsMeta[newShape].icon,
      // dont show suggestions for same type
      hide: table.changeType === 'reduced' || hasIntervalScale(groups) || state && !hasCustomSuggestionsExists(state.shape)
    };
    results.push(baseSuggestion);
    results.push({
      ...baseSuggestion,
      title: _i18n.i18n.translate('xpack.lens.pie.suggestionLabel', {
        defaultMessage: '{chartName}',
        values: {
          chartName: _partition_charts_meta.PartitionChartsMeta[newShape === _constants.PieChartTypes.PIE ? _constants.PieChartTypes.DONUT : _constants.PieChartTypes.PIE].label
        },
        description: 'chartName is already translated'
      }),
      score: 0.1,
      state: {
        ...baseSuggestion.state,
        shape: newShape === _constants.PieChartTypes.PIE ? _constants.PieChartTypes.DONUT : _constants.PieChartTypes.PIE
      },
      hide: true
    });
  }
  if (groups.length <= _partition_charts_meta.PartitionChartsMeta.treemap.maxBuckets && (!subVisualizationId || subVisualizationId === _constants.PieChartTypes.TREEMAP)) {
    results.push({
      title: _i18n.i18n.translate('xpack.lens.pie.treemapSuggestionLabel', {
        defaultMessage: 'Treemap'
      }),
      // Use a higher score when currently active, to prevent chart type switching
      // on the user unintentionally
      score: (state === null || state === void 0 ? void 0 : state.shape) === _constants.PieChartTypes.TREEMAP ? 0.7 : 0.5,
      state: {
        shape: _constants.PieChartTypes.TREEMAP,
        palette: (mainPalette === null || mainPalette === void 0 ? void 0 : mainPalette.type) === 'legacyPalette' ? mainPalette.value : state === null || state === void 0 ? void 0 : state.palette,
        layers: [state !== null && state !== void 0 && state.layers[0] ? {
          ...state.layers[0],
          layerId: table.layerId,
          primaryGroups: groups.map(col => col.columnId),
          metrics: metricColumnIds,
          categoryDisplay: state.layers[0].categoryDisplay === _constants.CategoryDisplay.INSIDE ? _constants.CategoryDisplay.DEFAULT : state.layers[0].categoryDisplay,
          layerType: _.layerTypes.DATA,
          colorMapping: (mainPalette === null || mainPalette === void 0 ? void 0 : mainPalette.type) === 'colorMapping' ? mainPalette.value : state.layers[0].colorMapping
        } : {
          layerId: table.layerId,
          primaryGroups: groups.map(col => col.columnId),
          metrics: metricColumnIds,
          numberDisplay: _constants.NumberDisplay.PERCENT,
          categoryDisplay: _constants.CategoryDisplay.DEFAULT,
          legendDisplay: _constants.LegendDisplay.DEFAULT,
          nestedLegend: false,
          layerType: _.layerTypes.DATA,
          colorMapping: (mainPalette === null || mainPalette === void 0 ? void 0 : mainPalette.type) === 'colorMapping' ? mainPalette.value : undefined
        }]
      },
      previewIcon: _partition_charts_meta.PartitionChartsMeta.treemap.icon,
      // hide treemap suggestions from bottom bar, but keep them for chart switcher
      hide: table.changeType === 'reduced' || !state || hasIntervalScale(groups) || state && state.shape === _constants.PieChartTypes.TREEMAP
    });
  }
  if (groups.length <= _partition_charts_meta.PartitionChartsMeta.mosaic.maxBuckets && (!subVisualizationId || subVisualizationId === _constants.PieChartTypes.MOSAIC)) {
    results.push({
      title: _i18n.i18n.translate('xpack.lens.pie.mosaicSuggestionLabel', {
        defaultMessage: 'Mosaic'
      }),
      score: (state === null || state === void 0 ? void 0 : state.shape) === _constants.PieChartTypes.MOSAIC ? 0.7 : 0.5,
      state: {
        shape: _constants.PieChartTypes.MOSAIC,
        palette: (mainPalette === null || mainPalette === void 0 ? void 0 : mainPalette.type) === 'legacyPalette' ? mainPalette.value : state === null || state === void 0 ? void 0 : state.palette,
        layers: [state !== null && state !== void 0 && state.layers[0] ? {
          ...state.layers[0],
          layerId: table.layerId,
          primaryGroups: groups[0] ? [groups[0].columnId] : [],
          secondaryGroups: groups[1] ? [groups[1].columnId] : [],
          metrics: metricColumnIds,
          categoryDisplay: _constants.CategoryDisplay.DEFAULT,
          layerType: _.layerTypes.DATA,
          allowMultipleMetrics: false,
          colorMapping: (mainPalette === null || mainPalette === void 0 ? void 0 : mainPalette.type) === 'colorMapping' ? mainPalette.value : state.layers[0].colorMapping
        } : {
          layerId: table.layerId,
          primaryGroups: groups[0] ? [groups[0].columnId] : [],
          secondaryGroups: groups[1] ? [groups[1].columnId] : [],
          metrics: metricColumnIds,
          numberDisplay: _constants.NumberDisplay.PERCENT,
          categoryDisplay: _constants.CategoryDisplay.DEFAULT,
          legendDisplay: _constants.LegendDisplay.DEFAULT,
          nestedLegend: false,
          layerType: _.layerTypes.DATA,
          allowMultipleMetrics: false,
          colorMapping: (mainPalette === null || mainPalette === void 0 ? void 0 : mainPalette.type) === 'colorMapping' ? mainPalette.value : undefined
        }]
      },
      previewIcon: _partition_charts_meta.PartitionChartsMeta.mosaic.icon,
      hide: groups.length !== 2 || table.changeType === 'reduced' || hasIntervalScale(groups) || state && state.shape === 'mosaic'
    });
  }
  if (groups.length <= _partition_charts_meta.PartitionChartsMeta.waffle.maxBuckets && (!subVisualizationId || subVisualizationId === _constants.PieChartTypes.WAFFLE)) {
    results.push({
      title: _i18n.i18n.translate('xpack.lens.pie.waffleSuggestionLabel', {
        defaultMessage: 'Waffle'
      }),
      score: (state === null || state === void 0 ? void 0 : state.shape) === _constants.PieChartTypes.WAFFLE ? 0.7 : 0.4,
      state: {
        shape: _constants.PieChartTypes.WAFFLE,
        palette: (mainPalette === null || mainPalette === void 0 ? void 0 : mainPalette.type) === 'legacyPalette' ? mainPalette.value : state === null || state === void 0 ? void 0 : state.palette,
        layers: [state !== null && state !== void 0 && state.layers[0] ? {
          ...state.layers[0],
          layerId: table.layerId,
          primaryGroups: groups.map(col => col.columnId),
          metrics: metricColumnIds,
          secondaryGroups: [],
          categoryDisplay: _constants.CategoryDisplay.DEFAULT,
          layerType: _.layerTypes.DATA,
          colorMapping: (mainPalette === null || mainPalette === void 0 ? void 0 : mainPalette.type) === 'colorMapping' ? mainPalette.value : state.layers[0].colorMapping
        } : {
          layerId: table.layerId,
          primaryGroups: groups.map(col => col.columnId),
          metrics: metricColumnIds,
          numberDisplay: _constants.NumberDisplay.PERCENT,
          categoryDisplay: _constants.CategoryDisplay.DEFAULT,
          legendDisplay: _constants.LegendDisplay.DEFAULT,
          nestedLegend: false,
          layerType: _.layerTypes.DATA,
          colorMapping: (mainPalette === null || mainPalette === void 0 ? void 0 : mainPalette.type) === 'colorMapping' ? mainPalette.value : undefined
        }]
      },
      previewIcon: _partition_charts_meta.PartitionChartsMeta.waffle.icon,
      hide: groups.length !== 1 || table.changeType === 'reduced' || hasIntervalScale(groups) || state && state.shape === 'waffle'
    });
  }
  return [...results].map(suggestion => ({
    ...suggestion,
    score: shouldHideSuggestion ? 0 : suggestion.score + 0.05 * groups.length + 0.01 * metricColumnIds.length
  })).sort((a, b) => b.score - a.score).map(suggestion => ({
    ...suggestion,
    hide: shouldHideSuggestion || incompleteConfiguration || suggestion.hide,
    incomplete: incompleteConfiguration
  }));
}