"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildXY = buildXY;
var _columns = require("../columns");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const ACCESSOR = 'metric_formula_accessor';
function buildVisualizationState(config) {
  var _config$legend, _config$legend2;
  return {
    legend: {
      isVisible: ((_config$legend = config.legend) === null || _config$legend === void 0 ? void 0 : _config$legend.show) || true,
      position: ((_config$legend2 = config.legend) === null || _config$legend2 === void 0 ? void 0 : _config$legend2.position) || 'left'
    },
    preferredSeriesType: 'line',
    valueLabels: 'hide',
    fittingFunction: 'None',
    axisTitlesVisibilitySettings: {
      x: true,
      yLeft: true,
      yRight: true
    },
    tickLabelsVisibilitySettings: {
      x: true,
      yLeft: true,
      yRight: true
    },
    labelsOrientation: {
      x: 0,
      yLeft: 0,
      yRight: 0
    },
    gridlinesVisibilitySettings: {
      x: true,
      yLeft: true,
      yRight: true
    },
    layers: config.layers.map((layer, i) => {
      switch (layer.type) {
        case 'annotation':
          return {
            layerId: `layer_${i}`,
            layerType: 'annotations',
            annotations: layer.events.map((e, eventNr) => {
              if ('datetime' in e) {
                return {
                  type: 'manual',
                  id: `annotation_${eventNr}`,
                  icon: e.icon || 'triangle',
                  color: e.color || 'blue',
                  label: e.name,
                  key: {
                    type: 'point_in_time',
                    timestamp: e.datetime
                  }
                };
              } else {
                return {
                  id: `event${eventNr}`,
                  type: 'query',
                  icon: e.icon || 'triangle',
                  color: e.color || 'blue',
                  label: e.name,
                  key: {
                    type: 'point_in_time'
                  },
                  filter: {
                    type: 'kibana_query',
                    query: e.filter,
                    language: 'kuery'
                  },
                  ...(e.field ? {
                    timeField: e.field
                  } : {})
                };
              }
            }),
            ignoreGlobalFilters: true
          };
        case 'reference':
          return {
            layerId: `layer_${i}`,
            layerType: 'referenceLine',
            accessors: [`${ACCESSOR}${i}`],
            yConfig: [{
              forAccessor: `${ACCESSOR}${i}`,
              axisMode: 'left'
            }]
          };
        case 'series':
          return {
            layerId: `layer_${i}`,
            layerType: 'data',
            xAccessor: `${ACCESSOR}${i}_x`,
            ...(layer.breakdown ? {
              splitAccessor: `${ACCESSOR}${i}_y}`
            } : {}),
            accessors: [`${ACCESSOR}${i}`],
            seriesType: layer.seriesType || 'line'
          };
      }
    })
  };
}
function getValueColumns(layer, i) {
  if (layer.breakdown && typeof layer.breakdown !== 'string') {
    throw new Error('breakdown must be a field name when not using index source');
  }
  if (typeof layer.xAxis !== 'string') {
    throw new Error('xAxis must be a field name when not using index source');
  }
  return [...(layer.breakdown ? [(0, _columns.getValueColumn)(`${ACCESSOR}${i}_breakdown`, layer.breakdown)] : []), (0, _columns.getValueColumn)(`${ACCESSOR}${i}_x`, layer.xAxis), (0, _columns.getValueColumn)(`${ACCESSOR}${i}`, layer.value, 'number')];
}
function buildFormulaLayer(layer, i, dataView, formulaAPI) {
  if (layer.type === 'series') {
    const resultLayer = {
      ...(0, _columns.getFormulaColumn)(`${ACCESSOR}${i}`, {
        value: layer.value
      }, dataView, formulaAPI)
    };
    if (layer.xAxis) {
      const columnName = `${ACCESSOR}${i}_x`;
      const breakdownColumn = (0, _columns.getBreakdownColumn)({
        options: layer.xAxis,
        dataView
      });
      (0, _utils.addLayerColumn)(resultLayer, columnName, breakdownColumn, true);
    }
    if (layer.breakdown) {
      const columnName = `${ACCESSOR}${i}_y`;
      const breakdownColumn = (0, _columns.getBreakdownColumn)({
        options: layer.breakdown,
        dataView
      });
      (0, _utils.addLayerColumn)(resultLayer, columnName, breakdownColumn, true);
    }
    return resultLayer;
  } else if (layer.type === 'annotation') {
    // nothing ?
  } else if (layer.type === 'reference') {
    return {
      ...(0, _columns.getFormulaColumn)(`${ACCESSOR}${i}`, {
        value: layer.value
      }, dataView, formulaAPI)
    };
  }
  return {
    columns: {},
    columnOrder: []
  };
}
async function buildXY(config, {
  dataViewsAPI,
  formulaAPI
}) {
  const dataviews = {};
  const _buildFormulaLayer = (cfg, i, dataView) => buildFormulaLayer(cfg, i, dataView, formulaAPI);
  const datasourceStates = await (0, _utils.buildDatasourceStates)(config, dataviews, _buildFormulaLayer, getValueColumns, dataViewsAPI);
  const references = (0, _utils.buildReferences)(dataviews);
  return {
    title: config.title,
    visualizationType: 'lnsXY',
    references,
    state: {
      datasourceStates,
      internalReferences: [],
      filters: [],
      query: {
        language: 'kuery',
        query: ''
      },
      visualization: buildVisualizationState(config),
      // Getting the spec from a data view is a heavy operation, that's why the result is cached.
      adHocDataViews: (0, _utils.getAdhocDataviews)(dataviews)
    }
  };
}