"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CloudDetector = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _aws = require("./aws");
var _azure = require("./azure");
var _gcp = require("./gcp");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const SUPPORTED_SERVICES = [_aws.AWSCloudService, _azure.AzureCloudService, _gcp.GCPCloudService];
/**
 * The `CloudDetector` can be used to asynchronously detect the
 * cloud service that Kibana is running within.
 *
 * @internal
 */
class CloudDetector {
  constructor(options = {}) {
    var _options$cloudService;
    (0, _defineProperty2.default)(this, "cloudServices", void 0);
    (0, _defineProperty2.default)(this, "cloudDetails", void 0);
    /**
     * Get any cloud details that we have detected.
     */
    (0, _defineProperty2.default)(this, "getCloudDetails", () => {
      return this.cloudDetails;
    });
    /**
     * Asynchronously detect the cloud service.
     *
     * Callers are _not_ expected to await this method, which allows the
     * caller to trigger the lookup and then simply use it whenever we
     * determine it.
     */
    (0, _defineProperty2.default)(this, "detectCloudService", async abortSignal => {
      this.cloudDetails = await this.getCloudService(abortSignal);
    });
    this.cloudServices = (_options$cloudService = options.cloudServices) !== null && _options$cloudService !== void 0 ? _options$cloudService : SUPPORTED_SERVICES.map(Service => new Service());
  }
  /**
   * Check every cloud service until the first one reports success from detection.
   */
  async getCloudService(abortSignal) {
    // check each service until we find one that is confirmed to match;
    // order is assumed to matter
    let stopping = false;
    abortSignal === null || abortSignal === void 0 ? void 0 : abortSignal.addEventListener('abort', () => stopping = true);
    for (const service of this.cloudServices) {
      if (stopping) break;
      try {
        const serviceResponse = await service.checkIfService(abortSignal);
        if (serviceResponse !== null && serviceResponse !== void 0 && serviceResponse.isConfirmed()) {
          return serviceResponse.toJSON();
        }
      } catch (ignoredError) {
        // ignored until we make wider use of this in the UI
      }
    }

    // explicitly null to differentiate from not having populated the field yet
    return null;
  }
}
exports.CloudDetector = CloudDetector;