"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.bootstrap = bootstrap;
var _coreLoggingServerInternal = require("@kbn/core-logging-server-internal");
var _config = require("./config");
var _kibana = require("./kibana");
var _server = require("./server");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

async function bootstrap() {
  const loggingSystem = new _coreLoggingServerInternal.LoggingSystem();
  const logger = loggingSystem.asLoggerFactory();
  const configService = (0, _config.getConfigService)({
    logger
  });
  const configDescriptors = [_coreLoggingServerInternal.config, _kibana.config, _server.config];
  for (const {
    path,
    schema
  } of configDescriptors) {
    configService.setSchema(path, schema);
  }
  await configService.validate();
  await loggingSystem.upgrade(configService.atPathSync('logging'));
  const log = logger.get('root');
  let server;
  let serverStart;
  try {
    server = new _server.Server({
      config: configService,
      logger
    });
    serverStart = await server.start();
  } catch (e) {
    log.error(`Failed to start Server: ${e}`);
    process.exit(1);
  }
  let kibanaService;
  try {
    kibanaService = new _kibana.KibanaService({
      config: configService,
      logger
    });
    await kibanaService.start({
      server: serverStart
    });
  } catch (e) {
    log.error(`Failed to start Kibana service: ${e}`);
    process.exit(1);
  }
  const attemptGracefulShutdown = async (exitCode = 0) => {
    await server.stop();
    kibanaService.stop();
    await loggingSystem.stop();
    process.exit(exitCode);
  };
  process.on('unhandledRejection', async err => {
    log.error(err);
    await attemptGracefulShutdown(1);
  });
  process.on('SIGINT', async () => await attemptGracefulShutdown());
  process.on('SIGTERM', async () => await attemptGracefulShutdown());
}