"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PackagePolicyServiceImpl = exports.DATA_STREAM_ALLOWED_INDEX_PRIVILEGES = void 0;
exports._applyIndexPrivileges = _applyIndexPrivileges;
exports._compilePackagePolicyInputs = _compilePackagePolicyInputs;
exports._validateRestrictedFieldsNotModifiedOrThrow = _validateRestrictedFieldsNotModifiedOrThrow;
exports.getInputsWithStreamIds = getInputsWithStreamIds;
exports.packagePolicyService = void 0;
exports.preconfigurePackageInputs = preconfigurePackageInputs;
exports.sendUpdatePackagePolicyTelemetryEvent = sendUpdatePackagePolicyTelemetryEvent;
exports.updatePackageInputs = updatePackageInputs;
var _classPrivateFieldGet2 = _interopRequireDefault(require("@babel/runtime/helpers/classPrivateFieldGet"));
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
var _lt = _interopRequireDefault(require("semver/functions/lt"));
var _std = require("@kbn/std");
var _server = require("@kbn/core/server");
var _uuid = require("uuid");
var _jsYaml = require("js-yaml");
var _constants = require("@kbn/spaces-plugin/common/constants");
var _pMap = _interopRequireDefault(require("p-map"));
var _http_authorization_header = require("../../common/http_authorization_header");
var _services = require("../../common/services");
var _constants2 = require("../../common/constants");
var _constants3 = require("../constants");
var _errors = require("../errors");
var _types = require("../types");
var _create_so_find_iterable = require("./utils/create_so_find_iterable");
var _security = require("./security");
var _agent_policies = require("./agent_policies");
var _agent_policy = require("./agent_policy");
var _packages = require("./epm/packages");
var _assets = require("./epm/packages/assets");
var _agent = require("./epm/agent/agent");
var _saved_object = require("./saved_object");
var _ = require(".");
var _cleanup = require("./epm/packages/cleanup");
var _upgrade_sender = require("./upgrade_sender");
var _package_policies = require("./package_policies");
var _update = require("./epm/packages/update");
var _install = require("./epm/packages/install");
var _audit_logging = require("./audit_logging");
var _secrets = require("./secrets");
function _classPrivateFieldInitSpec(obj, privateMap, value) { _checkPrivateRedeclaration(obj, privateMap); privateMap.set(obj, value); }
function _checkPrivateRedeclaration(obj, privateCollection) { if (privateCollection.has(obj)) { throw new TypeError("Cannot initialize the same private elements twice on an object"); } } /*
                                                                                                                                                                                            * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
                                                                                                                                                                                            * or more contributor license agreements. Licensed under the Elastic License
                                                                                                                                                                                            * 2.0; you may not use this file except in compliance with the Elastic License
                                                                                                                                                                                            * 2.0.
                                                                                                                                                                                            */ /* eslint-disable max-classes-per-file */
const SAVED_OBJECT_TYPE = _constants3.PACKAGE_POLICY_SAVED_OBJECT_TYPE;
const DATA_STREAM_ALLOWED_INDEX_PRIVILEGES = new Set(['auto_configure', 'create_doc', 'maintenance', 'monitor', 'read', 'read_cross_cluster']);
exports.DATA_STREAM_ALLOWED_INDEX_PRIVILEGES = DATA_STREAM_ALLOWED_INDEX_PRIVILEGES;
class PackagePolicyClientImpl {
  async create(soClient, esClient, packagePolicy, options = {}, context, request) {
    var _enrichedPackagePolic, _enrichedPackagePolic2, _secretReferences, _options$user$usernam, _options$user, _options$user$usernam2, _options$user2, _options$bumpRevision;
    const packagePolicyId = (options === null || options === void 0 ? void 0 : options.id) || (0, _uuid.v4)();
    let authorizationHeader = options.authorizationHeader;
    if (!authorizationHeader && request) {
      authorizationHeader = _http_authorization_header.HTTPAuthorizationHeader.parseFromRequest(request);
    }
    _audit_logging.auditLoggingService.writeCustomSoAuditLog({
      action: 'create',
      id: packagePolicyId,
      savedObjectType: _constants3.PACKAGE_POLICY_SAVED_OBJECT_TYPE
    });
    const logger = _.appContextService.getLogger();
    let secretReferences;
    let enrichedPackagePolicy = await packagePolicyService.runExternalCallbacks('packagePolicyCreate', packagePolicy, soClient, esClient, context, request);
    const agentPolicy = await _agent_policy.agentPolicyService.get(soClient, enrichedPackagePolicy.policy_id, true);
    if (agentPolicy && ((_enrichedPackagePolic = enrichedPackagePolicy.package) === null || _enrichedPackagePolic === void 0 ? void 0 : _enrichedPackagePolic.name) === _constants2.FLEET_APM_PACKAGE) {
      const dataOutput = await (0, _agent_policies.getDataOutputForAgentPolicy)(soClient, agentPolicy);
      if (dataOutput.type === _constants2.outputType.Logstash) {
        throw new _errors.FleetError('You cannot add APM to a policy using a logstash output');
      }
    }
    await validateIsNotHostedPolicy(soClient, enrichedPackagePolicy.policy_id, options === null || options === void 0 ? void 0 : options.force);

    // trailing whitespace causes issues creating API keys
    enrichedPackagePolicy.name = enrichedPackagePolicy.name.trim();
    if (!(options !== null && options !== void 0 && options.skipUniqueNameVerification)) {
      await requireUniqueName(soClient, enrichedPackagePolicy);
    }
    let elasticsearchPrivileges;
    let inputs = getInputsWithStreamIds(enrichedPackagePolicy, packagePolicyId);

    // Make sure the associated package is installed
    if ((_enrichedPackagePolic2 = enrichedPackagePolicy.package) !== null && _enrichedPackagePolic2 !== void 0 && _enrichedPackagePolic2.name) {
      var _options$packageInfo, _pkgInfo$elasticsearc;
      if (!(options !== null && options !== void 0 && options.skipEnsureInstalled)) {
        await (0, _packages.ensureInstalledPackage)({
          esClient,
          spaceId: (options === null || options === void 0 ? void 0 : options.spaceId) || _constants.DEFAULT_SPACE_ID,
          savedObjectsClient: soClient,
          pkgName: enrichedPackagePolicy.package.name,
          pkgVersion: enrichedPackagePolicy.package.version,
          force: options === null || options === void 0 ? void 0 : options.force,
          authorizationHeader
        });
      }

      // Handle component template/mappings updates for experimental features, e.g. synthetic source
      await (0, _package_policies.handleExperimentalDatastreamFeatureOptIn)({
        soClient,
        esClient,
        packagePolicy: enrichedPackagePolicy
      });
      const pkgInfo = (_options$packageInfo = options === null || options === void 0 ? void 0 : options.packageInfo) !== null && _options$packageInfo !== void 0 ? _options$packageInfo : await (0, _packages.getPackageInfo)({
        savedObjectsClient: soClient,
        pkgName: enrichedPackagePolicy.package.name,
        pkgVersion: enrichedPackagePolicy.package.version,
        prerelease: true
      });

      // Check if it is a limited package, and if so, check that the corresponding agent policy does not
      // already contain a package policy for this package
      if ((0, _services.isPackageLimited)(pkgInfo)) {
        if (agentPolicy && (0, _services.doesAgentPolicyAlreadyIncludePackage)(agentPolicy, pkgInfo.name)) {
          throw new _errors.FleetError(`Unable to create integration policy. Integration '${pkgInfo.name}' already exists on this agent policy.`);
        }
      }
      validatePackagePolicyOrThrow(enrichedPackagePolicy, pkgInfo);
      if (await (0, _secrets.isSecretStorageEnabled)(esClient, soClient)) {
        const secretsRes = await (0, _secrets.extractAndWriteSecrets)({
          packagePolicy: {
            ...enrichedPackagePolicy,
            inputs
          },
          packageInfo: pkgInfo,
          esClient
        });
        enrichedPackagePolicy = secretsRes.packagePolicy;
        secretReferences = secretsRes.secretReferences;
        inputs = enrichedPackagePolicy.inputs;
      }
      inputs = await _compilePackagePolicyInputs(pkgInfo, enrichedPackagePolicy.vars || {}, inputs);
      elasticsearchPrivileges = (_pkgInfo$elasticsearc = pkgInfo.elasticsearch) === null || _pkgInfo$elasticsearc === void 0 ? void 0 : _pkgInfo$elasticsearc.privileges;
      if (pkgInfo.type === 'input') {
        await (0, _install.installAssetsForInputPackagePolicy)({
          soClient,
          esClient,
          pkgInfo,
          packagePolicy: enrichedPackagePolicy,
          force: !!(options !== null && options !== void 0 && options.force),
          logger
        });
      }
    }
    const isoDate = new Date().toISOString();
    const newSo = await soClient.create(SAVED_OBJECT_TYPE, {
      ...enrichedPackagePolicy,
      ...(enrichedPackagePolicy.package ? {
        package: (0, _lodash.omit)(enrichedPackagePolicy.package, 'experimental_data_stream_features')
      } : {}),
      inputs,
      ...(elasticsearchPrivileges && {
        elasticsearch: {
          privileges: elasticsearchPrivileges
        }
      }),
      ...(((_secretReferences = secretReferences) === null || _secretReferences === void 0 ? void 0 : _secretReferences.length) && {
        secret_references: secretReferences
      }),
      revision: 1,
      created_at: isoDate,
      created_by: (_options$user$usernam = options === null || options === void 0 ? void 0 : (_options$user = options.user) === null || _options$user === void 0 ? void 0 : _options$user.username) !== null && _options$user$usernam !== void 0 ? _options$user$usernam : 'system',
      updated_at: isoDate,
      updated_by: (_options$user$usernam2 = options === null || options === void 0 ? void 0 : (_options$user2 = options.user) === null || _options$user2 === void 0 ? void 0 : _options$user2.username) !== null && _options$user$usernam2 !== void 0 ? _options$user$usernam2 : 'system'
    }, {
      ...options,
      id: packagePolicyId
    });
    if ((_options$bumpRevision = options === null || options === void 0 ? void 0 : options.bumpRevision) !== null && _options$bumpRevision !== void 0 ? _options$bumpRevision : true) {
      await _agent_policy.agentPolicyService.bumpRevision(soClient, esClient, enrichedPackagePolicy.policy_id, {
        user: options === null || options === void 0 ? void 0 : options.user
      });
    }
    const createdPackagePolicy = {
      id: newSo.id,
      version: newSo.version,
      ...newSo.attributes
    };
    return packagePolicyService.runExternalCallbacks('packagePolicyPostCreate', createdPackagePolicy, soClient, esClient);
  }
  async bulkCreate(soClient, esClient, packagePolicies, options) {
    var _options$bumpRevision2;
    for (const packagePolicy of packagePolicies) {
      if (!packagePolicy.id) {
        packagePolicy.id = _server.SavedObjectsUtils.generateId();
      }
      _audit_logging.auditLoggingService.writeCustomSoAuditLog({
        action: 'create',
        id: packagePolicy.id,
        savedObjectType: _constants3.PACKAGE_POLICY_SAVED_OBJECT_TYPE
      });
    }
    const agentPolicyIds = new Set(packagePolicies.map(pkgPolicy => pkgPolicy.policy_id));
    for (const agentPolicyId of agentPolicyIds) {
      await validateIsNotHostedPolicy(soClient, agentPolicyId, options === null || options === void 0 ? void 0 : options.force);
    }
    const packageInfos = await getPackageInfoForPackagePolicies(packagePolicies, soClient);
    const isoDate = new Date().toISOString();
    const policiesToCreate = [];
    const failedPolicies = [];
    const logger = _.appContextService.getLogger();
    const packagePoliciesWithIds = packagePolicies.map(p => {
      if (!p.id) {
        p.id = _server.SavedObjectsUtils.generateId();
      }
      return p;
    });
    await (0, _pMap.default)(packagePoliciesWithIds, async packagePolicy => {
      try {
        var _packagePolicy$id, _options$user$usernam3, _options$user3, _options$user$usernam4, _options$user4;
        const packagePolicyId = (_packagePolicy$id = packagePolicy.id) !== null && _packagePolicy$id !== void 0 ? _packagePolicy$id : (0, _uuid.v4)();
        const agentPolicyId = packagePolicy.policy_id;
        let inputs = getInputsWithStreamIds(packagePolicy, packagePolicyId);
        const {
          id,
          ...pkgPolicyWithoutId
        } = packagePolicy;
        let elasticsearch;
        if (packagePolicy.package) {
          const pkgInfo = packageInfos.get(`${packagePolicy.package.name}-${packagePolicy.package.version}`);
          inputs = pkgInfo ? await _compilePackagePolicyInputs(pkgInfo, packagePolicy.vars || {}, inputs) : inputs;
          elasticsearch = pkgInfo === null || pkgInfo === void 0 ? void 0 : pkgInfo.elasticsearch;
        }
        policiesToCreate.push({
          type: SAVED_OBJECT_TYPE,
          id: packagePolicyId,
          attributes: {
            ...pkgPolicyWithoutId,
            ...(packagePolicy.package ? {
              package: (0, _lodash.omit)(packagePolicy.package, 'experimental_data_stream_features')
            } : {}),
            inputs,
            elasticsearch,
            policy_id: agentPolicyId,
            revision: 1,
            created_at: isoDate,
            created_by: (_options$user$usernam3 = options === null || options === void 0 ? void 0 : (_options$user3 = options.user) === null || _options$user3 === void 0 ? void 0 : _options$user3.username) !== null && _options$user$usernam3 !== void 0 ? _options$user$usernam3 : 'system',
            updated_at: isoDate,
            updated_by: (_options$user$usernam4 = options === null || options === void 0 ? void 0 : (_options$user4 = options.user) === null || _options$user4 === void 0 ? void 0 : _options$user4.username) !== null && _options$user$usernam4 !== void 0 ? _options$user$usernam4 : 'system'
          }
        });
      } catch (error) {
        failedPolicies.push({
          packagePolicy,
          error
        });
        logger.error(error);
      }
    });
    const {
      saved_objects: createdObjects
    } = await soClient.bulkCreate(policiesToCreate);

    // Filter out invalid SOs
    const newSos = createdObjects.filter(so => !so.error && so.attributes);
    packagePoliciesWithIds.forEach(packagePolicy => {
      const hasCreatedSO = newSos.find(so => so.id === packagePolicy.id);
      const hasFailed = failedPolicies.some(({
        packagePolicy: failedPackagePolicy
      }) => failedPackagePolicy.id === packagePolicy.id);
      if (hasCreatedSO !== null && hasCreatedSO !== void 0 && hasCreatedSO.error && !hasFailed) {
        var _hasCreatedSO$error;
        failedPolicies.push({
          packagePolicy,
          error: (_hasCreatedSO$error = hasCreatedSO === null || hasCreatedSO === void 0 ? void 0 : hasCreatedSO.error) !== null && _hasCreatedSO$error !== void 0 ? _hasCreatedSO$error : new Error('Failed to create package policy.')
        });
      }
    });

    // Assign it to the given agent policy

    if ((_options$bumpRevision2 = options === null || options === void 0 ? void 0 : options.bumpRevision) !== null && _options$bumpRevision2 !== void 0 ? _options$bumpRevision2 : true) {
      for (const agentPolicyIdT of agentPolicyIds) {
        await _agent_policy.agentPolicyService.bumpRevision(soClient, esClient, agentPolicyIdT, {
          user: options === null || options === void 0 ? void 0 : options.user
        });
      }
    }
    return {
      created: newSos.map(newSo => ({
        id: newSo.id,
        version: newSo.version,
        ...newSo.attributes
      })),
      failed: failedPolicies
    };
  }

  /** Purpose of this function is to take a package policy and compile the inputs
   This is primarily used by the Synthetics UI to display the inputs which are passed to agent
   Purpose is to debug the inputs which are passed to the agent and also compared them to the config
   which is passed to public service locations */
  async inspect(soClient, packagePolicy) {
    if (!packagePolicy.id) {
      packagePolicy.id = _server.SavedObjectsUtils.generateId();
    }
    const packageInfos = await getPackageInfoForPackagePolicies([packagePolicy], soClient);
    const agentPolicyId = packagePolicy.policy_id;
    let inputs = getInputsWithStreamIds(packagePolicy, packagePolicy.id);
    const {
      id,
      ...pkgPolicyWithoutId
    } = packagePolicy;
    let elasticsearch;
    if (packagePolicy.package) {
      const pkgInfo = packageInfos.get(`${packagePolicy.package.name}-${packagePolicy.package.version}`);
      inputs = pkgInfo ? await _compilePackagePolicyInputs(pkgInfo, packagePolicy.vars || {}, inputs) : inputs;
      elasticsearch = pkgInfo === null || pkgInfo === void 0 ? void 0 : pkgInfo.elasticsearch;
    }
    return {
      id: packagePolicy.id,
      ...pkgPolicyWithoutId,
      ...(packagePolicy.package ? {
        package: (0, _lodash.omit)(packagePolicy.package, 'experimental_data_stream_features')
      } : {}),
      inputs,
      elasticsearch,
      policy_id: agentPolicyId
    };
  }
  async get(soClient, id) {
    var _packagePolicySO$attr;
    const packagePolicySO = await soClient.get(SAVED_OBJECT_TYPE, id);
    if (!packagePolicySO) {
      return null;
    }
    if (packagePolicySO.error) {
      throw new Error(packagePolicySO.error.message);
    }
    let experimentalFeatures;
    if ((_packagePolicySO$attr = packagePolicySO.attributes.package) !== null && _packagePolicySO$attr !== void 0 && _packagePolicySO$attr.name) {
      var _packagePolicySO$attr2;
      const installation = await soClient.get(_constants2.PACKAGES_SAVED_OBJECT_TYPE, (_packagePolicySO$attr2 = packagePolicySO.attributes.package) === null || _packagePolicySO$attr2 === void 0 ? void 0 : _packagePolicySO$attr2.name);
      if (installation && !installation.error) {
        var _installation$attribu;
        experimentalFeatures = (_installation$attribu = installation.attributes) === null || _installation$attribu === void 0 ? void 0 : _installation$attribu.experimental_data_stream_features;
      }
    }
    const response = {
      id: packagePolicySO.id,
      version: packagePolicySO.version,
      ...packagePolicySO.attributes
    };

    // If possible, return the experimental features map for the package policy's `package` field
    if (experimentalFeatures && response.package) {
      response.package.experimental_data_stream_features = experimentalFeatures;
    }
    _audit_logging.auditLoggingService.writeCustomSoAuditLog({
      action: 'get',
      id,
      savedObjectType: _constants3.PACKAGE_POLICY_SAVED_OBJECT_TYPE
    });
    return response;
  }
  async findAllForAgentPolicy(soClient, agentPolicyId) {
    const packagePolicySO = await soClient.find({
      type: SAVED_OBJECT_TYPE,
      filter: `${SAVED_OBJECT_TYPE}.attributes.policy_id:${(0, _saved_object.escapeSearchQueryPhrase)(agentPolicyId)}`,
      perPage: _constants2.SO_SEARCH_LIMIT
    });
    if (!packagePolicySO) {
      return [];
    }
    const packagePolicies = packagePolicySO.saved_objects.map(so => ({
      id: so.id,
      version: so.version,
      ...so.attributes
    }));
    for (const packagePolicy of packagePolicies) {
      _audit_logging.auditLoggingService.writeCustomSoAuditLog({
        action: 'find',
        id: packagePolicy.id,
        savedObjectType: _constants3.PACKAGE_POLICY_SAVED_OBJECT_TYPE
      });
    }
    return packagePolicies;
  }
  async getByIDs(soClient, ids, options = {}) {
    const packagePolicySO = await soClient.bulkGet(ids.map(id => ({
      id,
      type: SAVED_OBJECT_TYPE
    })));
    if (!packagePolicySO) {
      return null;
    }
    const packagePolicies = packagePolicySO.saved_objects.map(so => {
      if (so.error) {
        if (options.ignoreMissing && so.error.statusCode === 404) {
          return null;
        } else if (so.error.statusCode === 404) {
          throw new _errors.PackagePolicyNotFoundError(`Package policy ${so.id} not found`);
        } else {
          throw new Error(so.error.message);
        }
      }
      return {
        id: so.id,
        version: so.version,
        ...so.attributes
      };
    }).filter(packagePolicy => packagePolicy !== null);
    for (const packagePolicy of packagePolicies) {
      _audit_logging.auditLoggingService.writeCustomSoAuditLog({
        action: 'get',
        id: packagePolicy.id,
        savedObjectType: _constants3.PACKAGE_POLICY_SAVED_OBJECT_TYPE
      });
    }
    return packagePolicies;
  }
  async list(soClient, options) {
    const {
      page = 1,
      perPage = 20,
      sortField = 'updated_at',
      sortOrder = 'desc',
      kuery
    } = options;
    const packagePolicies = await soClient.find({
      type: SAVED_OBJECT_TYPE,
      sortField,
      sortOrder,
      page,
      perPage,
      filter: kuery ? (0, _saved_object.normalizeKuery)(SAVED_OBJECT_TYPE, kuery) : undefined
    });
    for (const packagePolicy of (_packagePolicies$save = packagePolicies === null || packagePolicies === void 0 ? void 0 : packagePolicies.saved_objects) !== null && _packagePolicies$save !== void 0 ? _packagePolicies$save : []) {
      var _packagePolicies$save;
      _audit_logging.auditLoggingService.writeCustomSoAuditLog({
        action: 'find',
        id: packagePolicy.id,
        savedObjectType: _constants3.PACKAGE_POLICY_SAVED_OBJECT_TYPE
      });
    }
    return {
      items: packagePolicies === null || packagePolicies === void 0 ? void 0 : packagePolicies.saved_objects.map(packagePolicySO => ({
        id: packagePolicySO.id,
        version: packagePolicySO.version,
        ...packagePolicySO.attributes
      })),
      total: packagePolicies === null || packagePolicies === void 0 ? void 0 : packagePolicies.total,
      page,
      perPage
    };
  }
  async listIds(soClient, options) {
    const {
      page = 1,
      perPage = 20,
      sortField = 'updated_at',
      sortOrder = 'desc',
      kuery
    } = options;
    const packagePolicies = await soClient.find({
      type: SAVED_OBJECT_TYPE,
      sortField,
      sortOrder,
      page,
      perPage,
      fields: [],
      filter: kuery ? (0, _saved_object.normalizeKuery)(SAVED_OBJECT_TYPE, kuery) : undefined
    });
    for (const packagePolicy of packagePolicies.saved_objects) {
      _audit_logging.auditLoggingService.writeCustomSoAuditLog({
        action: 'find',
        id: packagePolicy.id,
        savedObjectType: _constants3.PACKAGE_POLICY_SAVED_OBJECT_TYPE
      });
    }
    return {
      items: packagePolicies.saved_objects.map(packagePolicySO => packagePolicySO.id),
      total: packagePolicies.total,
      page,
      perPage
    };
  }
  async update(soClient, esClient, id, packagePolicyUpdate, options) {
    var _packagePolicy$packag, _secretReferences2, _options$user$usernam5, _options$user5, _oldPackagePolicy$pac, _secretsToDelete;
    _audit_logging.auditLoggingService.writeCustomSoAuditLog({
      action: 'update',
      id,
      savedObjectType: _constants3.PACKAGE_POLICY_SAVED_OBJECT_TYPE
    });
    let enrichedPackagePolicy;
    let secretReferences;
    let secretsToDelete;
    try {
      enrichedPackagePolicy = await packagePolicyService.runExternalCallbacks('packagePolicyUpdate', packagePolicyUpdate, soClient, esClient);
    } catch (error) {
      const logger = _.appContextService.getLogger();
      logger.error(`An error occurred executing "packagePolicyUpdate" callback: ${error}`);
      logger.error(error);
      if (error.apiPassThrough) {
        throw error;
      }
      enrichedPackagePolicy = packagePolicyUpdate;
    }
    const packagePolicy = {
      ...enrichedPackagePolicy,
      name: enrichedPackagePolicy.name.trim()
    };
    const oldPackagePolicy = await this.get(soClient, id);
    if (packagePolicyUpdate.is_managed && !(options !== null && options !== void 0 && options.force)) {
      throw new _errors.PackagePolicyRestrictionRelatedError(`Cannot update package policy ${id}`);
    }
    if (!oldPackagePolicy) {
      throw new Error('Package policy not found');
    }
    if (packagePolicy.name && packagePolicy.name !== oldPackagePolicy.name && !(options !== null && options !== void 0 && options.skipUniqueNameVerification)) {
      await requireUniqueName(soClient, enrichedPackagePolicy, id);
    }

    // eslint-disable-next-line prefer-const
    let {
      version,
      ...restOfPackagePolicy
    } = packagePolicy;
    let inputs = getInputsWithStreamIds(restOfPackagePolicy, oldPackagePolicy.id);
    inputs = enforceFrozenInputs(oldPackagePolicy.inputs, inputs, options === null || options === void 0 ? void 0 : options.force);
    let elasticsearchPrivileges;
    let pkgInfo;
    if ((_packagePolicy$packag = packagePolicy.package) !== null && _packagePolicy$packag !== void 0 && _packagePolicy$packag.name) {
      var _pkgInfo$elasticsearc2;
      pkgInfo = await (0, _packages.getPackageInfo)({
        savedObjectsClient: soClient,
        pkgName: packagePolicy.package.name,
        pkgVersion: packagePolicy.package.version,
        prerelease: true
      });
      _validateRestrictedFieldsNotModifiedOrThrow({
        pkgInfo,
        oldPackagePolicy,
        packagePolicyUpdate
      });
      validatePackagePolicyOrThrow(packagePolicy, pkgInfo);
      if (await (0, _secrets.isSecretStorageEnabled)(esClient, soClient)) {
        const secretsRes = await (0, _secrets.extractAndUpdateSecrets)({
          oldPackagePolicy,
          packagePolicyUpdate: {
            ...restOfPackagePolicy,
            inputs
          },
          packageInfo: pkgInfo,
          esClient
        });
        restOfPackagePolicy = secretsRes.packagePolicyUpdate;
        secretReferences = secretsRes.secretReferences;
        secretsToDelete = secretsRes.secretsToDelete;
        inputs = restOfPackagePolicy.inputs;
      }
      inputs = await _compilePackagePolicyInputs(pkgInfo, restOfPackagePolicy.vars || {}, inputs);
      elasticsearchPrivileges = (_pkgInfo$elasticsearc2 = pkgInfo.elasticsearch) === null || _pkgInfo$elasticsearc2 === void 0 ? void 0 : _pkgInfo$elasticsearc2.privileges;
    }

    // Handle component template/mappings updates for experimental features, e.g. synthetic source
    await (0, _package_policies.handleExperimentalDatastreamFeatureOptIn)({
      soClient,
      esClient,
      packagePolicy: restOfPackagePolicy
    });
    await soClient.update(SAVED_OBJECT_TYPE, id, {
      ...restOfPackagePolicy,
      ...(restOfPackagePolicy.package ? {
        package: (0, _lodash.omit)(restOfPackagePolicy.package, 'experimental_data_stream_features')
      } : {}),
      inputs,
      ...(elasticsearchPrivileges && {
        elasticsearch: {
          privileges: elasticsearchPrivileges
        }
      }),
      ...(((_secretReferences2 = secretReferences) === null || _secretReferences2 === void 0 ? void 0 : _secretReferences2.length) && {
        secret_references: secretReferences
      }),
      revision: oldPackagePolicy.revision + 1,
      updated_at: new Date().toISOString(),
      updated_by: (_options$user$usernam5 = options === null || options === void 0 ? void 0 : (_options$user5 = options.user) === null || _options$user5 === void 0 ? void 0 : _options$user5.username) !== null && _options$user$usernam5 !== void 0 ? _options$user$usernam5 : 'system'
    }, {
      version
    });
    const newPolicy = await this.get(soClient, id);

    // if we have moved to an input package we need to create the index templates
    // for the package policy as input packages create index templates per package policy
    if (pkgInfo && pkgInfo.type === 'input' && oldPackagePolicy.package && ((_oldPackagePolicy$pac = oldPackagePolicy.package) === null || _oldPackagePolicy$pac === void 0 ? void 0 : _oldPackagePolicy$pac.version) !== pkgInfo.version) {
      if (oldPackagePolicy.package) {
        var _oldPackagePolicy$pac2, _oldPackagePolicy$pac3;
        const oldPackage = await (0, _packages.getPackageInfo)({
          savedObjectsClient: soClient,
          pkgName: (_oldPackagePolicy$pac2 = oldPackagePolicy.package) === null || _oldPackagePolicy$pac2 === void 0 ? void 0 : _oldPackagePolicy$pac2.name,
          pkgVersion: (_oldPackagePolicy$pac3 = oldPackagePolicy.package) === null || _oldPackagePolicy$pac3 === void 0 ? void 0 : _oldPackagePolicy$pac3.version,
          prerelease: true
        });
        if (oldPackage.type === 'integration') {
          await (0, _install.installAssetsForInputPackagePolicy)({
            logger: _.appContextService.getLogger(),
            soClient,
            esClient,
            pkgInfo,
            packagePolicy: newPolicy,
            force: true
          });
        }
      }
    }
    // Bump revision of associated agent policy
    const bumpPromise = _agent_policy.agentPolicyService.bumpRevision(soClient, esClient, packagePolicy.policy_id, {
      user: options === null || options === void 0 ? void 0 : options.user
    });
    const assetRemovePromise = (0, _cleanup.removeOldAssets)({
      soClient,
      pkgName: newPolicy.package.name,
      currentVersion: newPolicy.package.version
    });
    const deleteSecretsPromise = (_secretsToDelete = secretsToDelete) !== null && _secretsToDelete !== void 0 && _secretsToDelete.length ? (0, _secrets.deleteSecretsIfNotReferenced)({
      esClient,
      soClient,
      ids: secretsToDelete.map(s => s.id)
    }) : Promise.resolve();
    await Promise.all([bumpPromise, assetRemovePromise, deleteSecretsPromise]);
    sendUpdatePackagePolicyTelemetryEvent(soClient, [packagePolicyUpdate], [oldPackagePolicy]);
    return newPolicy;
  }
  async bulkUpdate(soClient, esClient, packagePolicyUpdates, options) {
    for (const packagePolicy of packagePolicyUpdates) {
      _audit_logging.auditLoggingService.writeCustomSoAuditLog({
        action: 'update',
        id: packagePolicy.id,
        savedObjectType: _constants3.PACKAGE_POLICY_SAVED_OBJECT_TYPE
      });
    }
    const oldPackagePolicies = await this.getByIDs(soClient, packagePolicyUpdates.map(p => p.id));
    if (!oldPackagePolicies || oldPackagePolicies.length === 0) {
      throw new Error('Package policy not found');
    }
    const packageInfos = await getPackageInfoForPackagePolicies(packagePolicyUpdates, soClient);
    const allSecretsToDelete = [];
    const policiesToUpdate = [];
    const failedPolicies = [];
    await (0, _pMap.default)(packagePolicyUpdates, async packagePolicyUpdate => {
      try {
        var _packagePolicy$packag2, _secretReferences3, _options$user$usernam6, _options$user6;
        const id = packagePolicyUpdate.id;
        const packagePolicy = {
          ...packagePolicyUpdate,
          name: packagePolicyUpdate.name.trim()
        };
        const oldPackagePolicy = oldPackagePolicies.find(p => p.id === id);
        if (!oldPackagePolicy) {
          throw new Error('Package policy not found');
        }
        let secretReferences;

        // id and version are not part of the saved object attributes
        // eslint-disable-next-line prefer-const
        let {
          version,
          id: _id,
          ...restOfPackagePolicy
        } = packagePolicy;
        if (packagePolicyUpdate.is_managed && !(options !== null && options !== void 0 && options.force)) {
          throw new _errors.PackagePolicyRestrictionRelatedError(`Cannot update package policy ${id}`);
        }
        let inputs = getInputsWithStreamIds(restOfPackagePolicy, oldPackagePolicy.id);
        inputs = enforceFrozenInputs(oldPackagePolicy.inputs, inputs, options === null || options === void 0 ? void 0 : options.force);
        let elasticsearchPrivileges;
        if ((_packagePolicy$packag2 = packagePolicy.package) !== null && _packagePolicy$packag2 !== void 0 && _packagePolicy$packag2.name) {
          const pkgInfo = packageInfos.get(`${packagePolicy.package.name}-${packagePolicy.package.version}`);
          if (pkgInfo) {
            var _pkgInfo$elasticsearc3;
            validatePackagePolicyOrThrow(packagePolicy, pkgInfo);
            if (await (0, _secrets.isSecretStorageEnabled)(esClient, soClient)) {
              const secretsRes = await (0, _secrets.extractAndUpdateSecrets)({
                oldPackagePolicy,
                packagePolicyUpdate: {
                  ...restOfPackagePolicy,
                  inputs
                },
                packageInfo: pkgInfo,
                esClient
              });
              restOfPackagePolicy = secretsRes.packagePolicyUpdate;
              secretReferences = secretsRes.secretReferences;
              allSecretsToDelete.push(...secretsRes.secretsToDelete);
              inputs = restOfPackagePolicy.inputs;
            }
            inputs = await _compilePackagePolicyInputs(pkgInfo, packagePolicy.vars || {}, inputs);
            elasticsearchPrivileges = (_pkgInfo$elasticsearc3 = pkgInfo.elasticsearch) === null || _pkgInfo$elasticsearc3 === void 0 ? void 0 : _pkgInfo$elasticsearc3.privileges;
          }
        }

        // Handle component template/mappings updates for experimental features, e.g. synthetic source
        await (0, _package_policies.handleExperimentalDatastreamFeatureOptIn)({
          soClient,
          esClient,
          packagePolicy
        });
        policiesToUpdate.push({
          type: SAVED_OBJECT_TYPE,
          id,
          attributes: {
            ...restOfPackagePolicy,
            ...(restOfPackagePolicy.package ? {
              package: (0, _lodash.omit)(restOfPackagePolicy.package, 'experimental_data_stream_features')
            } : {}),
            inputs,
            ...(elasticsearchPrivileges && {
              elasticsearch: {
                privileges: elasticsearchPrivileges
              }
            }),
            ...(((_secretReferences3 = secretReferences) === null || _secretReferences3 === void 0 ? void 0 : _secretReferences3.length) && {
              secret_references: secretReferences
            }),
            revision: oldPackagePolicy.revision + 1,
            updated_at: new Date().toISOString(),
            updated_by: (_options$user$usernam6 = options === null || options === void 0 ? void 0 : (_options$user6 = options.user) === null || _options$user6 === void 0 ? void 0 : _options$user6.username) !== null && _options$user$usernam6 !== void 0 ? _options$user$usernam6 : 'system'
          },
          version
        });
      } catch (error) {
        failedPolicies.push({
          packagePolicy: packagePolicyUpdate,
          error
        });
      }
    });
    const {
      saved_objects: updatedPolicies
    } = await soClient.bulkUpdate(policiesToUpdate);
    const agentPolicyIds = new Set(packagePolicyUpdates.map(p => p.policy_id));
    const bumpPromise = (0, _pMap.default)(agentPolicyIds, async agentPolicyId => {
      // Bump revision of associated agent policy
      await _agent_policy.agentPolicyService.bumpRevision(soClient, esClient, agentPolicyId, {
        user: options === null || options === void 0 ? void 0 : options.user
      });
    });
    const pkgVersions = {};
    packagePolicyUpdates.forEach(({
      package: pkg
    }) => {
      if (pkg) {
        pkgVersions[pkg.name + '-' + pkg.version] = {
          name: pkg.name,
          version: pkg.version
        };
      }
    });
    const removeAssetPromise = (0, _pMap.default)(Object.keys(pkgVersions), async pkgVersion => {
      const {
        name,
        version
      } = pkgVersions[pkgVersion];
      await (0, _cleanup.removeOldAssets)({
        soClient,
        pkgName: name,
        currentVersion: version
      });
    });
    const deleteSecretsPromise = allSecretsToDelete.length ? (0, _secrets.deleteSecretsIfNotReferenced)({
      esClient,
      soClient,
      ids: allSecretsToDelete.map(s => s.id)
    }) : Promise.resolve();
    await Promise.all([bumpPromise, removeAssetPromise, deleteSecretsPromise]);
    sendUpdatePackagePolicyTelemetryEvent(soClient, packagePolicyUpdates, oldPackagePolicies);
    updatedPolicies.forEach(policy => {
      if (policy.error) {
        const hasAlreadyFailed = failedPolicies.some(failedPolicy => failedPolicy.packagePolicy.id === policy.id);
        if (!hasAlreadyFailed) {
          failedPolicies.push({
            packagePolicy: packagePolicyUpdates.find(p => p.id === policy.id),
            error: policy.error
          });
        }
      }
    });
    const updatedPoliciesSuccess = updatedPolicies.filter(policy => !policy.error && policy.attributes).map(soPolicy => ({
      id: soPolicy.id,
      version: soPolicy.version,
      ...soPolicy.attributes
    }));
    return {
      updatedPolicies: updatedPoliciesSuccess,
      failedPolicies
    };
  }
  async delete(soClient, esClient, ids, options, context, request) {
    for (const id of ids) {
      _audit_logging.auditLoggingService.writeCustomSoAuditLog({
        action: 'delete',
        id,
        savedObjectType: _constants3.PACKAGE_POLICY_SAVED_OBJECT_TYPE
      });
    }
    const result = [];
    const logger = _.appContextService.getLogger();
    const packagePolicies = await this.getByIDs(soClient, ids, {
      ignoreMissing: true
    });
    if (!packagePolicies) {
      return [];
    }
    try {
      await packagePolicyService.runDeleteExternalCallbacks(packagePolicies, soClient, esClient, context, request);
    } catch (error) {
      logger.error(`An error occurred executing "packagePolicyDelete" callback: ${error}`);
      logger.error(error);
    }
    const uniqueAgentPolicyIds = [...new Set(packagePolicies.map(packagePolicy => packagePolicy.policy_id))];
    const hostedAgentPolicies = [];
    for (const agentPolicyId of uniqueAgentPolicyIds) {
      try {
        await validateIsNotHostedPolicy(soClient, agentPolicyId, options === null || options === void 0 ? void 0 : options.force, 'Cannot remove integrations of hosted agent policy');
      } catch (e) {
        hostedAgentPolicies.push(agentPolicyId);
      }
    }
    const idsToDelete = [];
    ids.forEach(id => {
      try {
        const packagePolicy = packagePolicies.find(p => p.id === id);
        if (!packagePolicy) {
          throw new _errors.PackagePolicyNotFoundError(`Saved object [ingest-package-policies/${id}] not found`);
        }
        if (packagePolicy.is_managed && !(options !== null && options !== void 0 && options.force)) {
          throw new _errors.PackagePolicyRestrictionRelatedError(`Cannot delete package policy ${id}`);
        }
        if (hostedAgentPolicies.includes(packagePolicy.policy_id)) {
          throw new _errors.HostedAgentPolicyRestrictionRelatedError('Cannot remove integrations of hosted agent policy');
        }
        idsToDelete.push(id);
      } catch (error) {
        result.push({
          id,
          success: false,
          ...(0, _errors.fleetErrorToResponseOptions)(error)
        });
      }
    });
    const secretsToDelete = [];
    if (idsToDelete.length > 0) {
      const {
        statuses
      } = await soClient.bulkDelete(idsToDelete.map(id => ({
        id,
        type: SAVED_OBJECT_TYPE
      })));
      statuses.forEach(({
        id,
        success,
        error
      }) => {
        const packagePolicy = packagePolicies.find(p => p.id === id);
        if (success && packagePolicy) {
          var _packagePolicy$packag3, _packagePolicy$packag4, _packagePolicy$packag5, _packagePolicy$secret;
          result.push({
            id,
            name: packagePolicy.name,
            success: true,
            package: {
              name: ((_packagePolicy$packag3 = packagePolicy.package) === null || _packagePolicy$packag3 === void 0 ? void 0 : _packagePolicy$packag3.name) || '',
              title: ((_packagePolicy$packag4 = packagePolicy.package) === null || _packagePolicy$packag4 === void 0 ? void 0 : _packagePolicy$packag4.title) || '',
              version: ((_packagePolicy$packag5 = packagePolicy.package) === null || _packagePolicy$packag5 === void 0 ? void 0 : _packagePolicy$packag5.version) || ''
            },
            policy_id: packagePolicy.policy_id
          });
          if (packagePolicy !== null && packagePolicy !== void 0 && (_packagePolicy$secret = packagePolicy.secret_references) !== null && _packagePolicy$secret !== void 0 && _packagePolicy$secret.length) {
            secretsToDelete.push(...packagePolicy.secret_references.map(s => s.id));
          }
        } else if (!success && error) {
          result.push({
            id,
            success: false,
            statusCode: error.statusCode,
            body: {
              message: error.message
            }
          });
        }
      });
    }
    if (!(options !== null && options !== void 0 && options.skipUnassignFromAgentPolicies)) {
      const uniquePolicyIdsR = [...new Set(result.filter(r => r.success && r.policy_id).map(r => r.policy_id))];
      const agentPoliciesWithEndpointPackagePolicies = result.reduce((acc, cur) => {
        var _cur$package;
        if (cur.success && cur.policy_id && ((_cur$package = cur.package) === null || _cur$package === void 0 ? void 0 : _cur$package.name) === 'endpoint') {
          return acc.add(cur.policy_id);
        }
        return acc;
      }, new Set());
      const agentPolicies = await _agent_policy.agentPolicyService.getByIDs(soClient, uniquePolicyIdsR);
      for (const policyId of uniquePolicyIdsR) {
        const agentPolicy = agentPolicies.find(p => p.id === policyId);
        if (agentPolicy) {
          // is the agent policy attached to package policy with endpoint
          await _agent_policy.agentPolicyService.bumpRevision(soClient, esClient, policyId, {
            user: options === null || options === void 0 ? void 0 : options.user,
            removeProtection: agentPoliciesWithEndpointPackagePolicies.has(policyId)
          });
        }
      }
    }
    if (secretsToDelete.length > 0) {
      await (0, _secrets.deleteSecretsIfNotReferenced)({
        esClient,
        soClient,
        ids: secretsToDelete
      });
    }
    try {
      await packagePolicyService.runPostDeleteExternalCallbacks(result, soClient, esClient, context, request);
    } catch (error) {
      logger.error(`An error occurred executing "packagePolicyPostDelete" callback: ${error}`);
      logger.error(error);
    }
    return result;
  }

  // TODO should move out, public only for unit tests
  async getUpgradePackagePolicyInfo(soClient, id, packagePolicy, pkgVersion) {
    if (!packagePolicy) {
      var _await$this$get;
      packagePolicy = (_await$this$get = await this.get(soClient, id)) !== null && _await$this$get !== void 0 ? _await$this$get : undefined;
    }
    let experimentalDataStreamFeatures = [];
    if (!pkgVersion && packagePolicy) {
      var _installedPackage$exp;
      const installedPackage = await (0, _packages.getInstallation)({
        savedObjectsClient: soClient,
        pkgName: packagePolicy.package.name
      });
      if (!installedPackage) {
        throw new _errors.FleetError(_i18n.i18n.translate('xpack.fleet.packagePolicy.packageNotInstalledError', {
          defaultMessage: 'Package {name} is not installed',
          values: {
            name: packagePolicy.package.name
          }
        }));
      }
      pkgVersion = installedPackage.version;
      experimentalDataStreamFeatures = (_installedPackage$exp = installedPackage.experimental_data_stream_features) !== null && _installedPackage$exp !== void 0 ? _installedPackage$exp : [];
    }
    let packageInfo;
    if (packagePolicy) {
      var _pkgVersion;
      packageInfo = await (0, _packages.getPackageInfo)({
        savedObjectsClient: soClient,
        pkgName: packagePolicy.package.name,
        pkgVersion: (_pkgVersion = pkgVersion) !== null && _pkgVersion !== void 0 ? _pkgVersion : '',
        prerelease: !!pkgVersion // using prerelease only if version is specified
      });
    }

    this.validateUpgradePackagePolicy(id, packageInfo, packagePolicy);
    return {
      packagePolicy: packagePolicy,
      packageInfo: packageInfo,
      experimentalDataStreamFeatures
    };
  }
  validateUpgradePackagePolicy(id, packageInfo, packagePolicy) {
    var _packagePolicy$packag6, _packageInfo$version;
    if (!packagePolicy) {
      throw new _errors.FleetError(_i18n.i18n.translate('xpack.fleet.packagePolicy.policyNotFoundError', {
        defaultMessage: 'Package policy with id {id} not found',
        values: {
          id
        }
      }));
    }
    if (!((_packagePolicy$packag6 = packagePolicy.package) !== null && _packagePolicy$packag6 !== void 0 && _packagePolicy$packag6.name)) {
      throw new _errors.FleetError(_i18n.i18n.translate('xpack.fleet.packagePolicy.packageNotFoundError', {
        defaultMessage: 'Package policy with id {id} has no named package',
        values: {
          id
        }
      }));
    }
    const isInstalledVersionLessThanPolicyVersion = (0, _lt.default)((_packageInfo$version = packageInfo === null || packageInfo === void 0 ? void 0 : packageInfo.version) !== null && _packageInfo$version !== void 0 ? _packageInfo$version : '', packagePolicy.package.version);
    if (isInstalledVersionLessThanPolicyVersion) {
      throw new _errors.PackagePolicyIneligibleForUpgradeError(_i18n.i18n.translate('xpack.fleet.packagePolicy.ineligibleForUpgradeError', {
        defaultMessage: "Package policy {id}'s package version {version} of package {name} is newer than the installed package version. Please install the latest version of {name}.",
        values: {
          id: packagePolicy.id,
          name: packagePolicy.package.name,
          version: packagePolicy.package.version
        }
      }));
    }
  }
  async upgrade(soClient, esClient, ids, options, packagePolicy, pkgVersion) {
    const result = [];
    for (const id of ids) {
      try {
        const {
          packagePolicy: currentPackagePolicy,
          packageInfo,
          experimentalDataStreamFeatures
        } = await this.getUpgradePackagePolicyInfo(soClient, id, packagePolicy, pkgVersion);
        if (currentPackagePolicy.is_managed && !(options !== null && options !== void 0 && options.force)) {
          throw new _errors.PackagePolicyRestrictionRelatedError(`Cannot upgrade package policy ${id}`);
        }
        await this.doUpgrade(soClient, esClient, id, currentPackagePolicy, result, packageInfo, experimentalDataStreamFeatures, options);
      } catch (error) {
        result.push({
          id,
          success: false,
          ...(0, _errors.fleetErrorToResponseOptions)(error)
        });
      }
    }
    return result;
  }
  async doUpgrade(soClient, esClient, id, packagePolicy, result, packageInfo, experimentalDataStreamFeatures, options) {
    const updatePackagePolicy = updatePackageInputs({
      ...(0, _lodash.omit)(packagePolicy, 'id'),
      inputs: packagePolicy.inputs,
      package: {
        ...packagePolicy.package,
        version: packageInfo.version
      }
    }, packageInfo, (0, _services.packageToPackagePolicyInputs)(packageInfo));
    updatePackagePolicy.inputs = await _compilePackagePolicyInputs(packageInfo, updatePackagePolicy.vars || {}, updatePackagePolicy.inputs);
    updatePackagePolicy.elasticsearch = packageInfo.elasticsearch;
    const updateOptions = {
      skipUniqueNameVerification: true,
      ...options
    };
    await this.update(soClient, esClient, id, updatePackagePolicy, updateOptions);

    // Persist any experimental feature opt-ins that come through the upgrade process to the Installation SO
    await (0, _update.updateDatastreamExperimentalFeatures)(soClient, packagePolicy.package.name, experimentalDataStreamFeatures);
    result.push({
      id,
      name: packagePolicy.name,
      success: true
    });
  }
  async getUpgradeDryRunDiff(soClient, id, packagePolicy, pkgVersion) {
    try {
      let packageInfo;
      let experimentalDataStreamFeatures;
      ({
        packagePolicy,
        packageInfo,
        experimentalDataStreamFeatures
      } = await this.getUpgradePackagePolicyInfo(soClient, id, packagePolicy, pkgVersion));

      // Ensure the experimental features from the Installation saved object come through on the package policy
      // during an upgrade dry run
      if (packagePolicy.package) {
        packagePolicy.package.experimental_data_stream_features = experimentalDataStreamFeatures;
      }
      return this.calculateDiff(soClient, packagePolicy, packageInfo);
    } catch (error) {
      return {
        hasErrors: true,
        ...(0, _errors.fleetErrorToResponseOptions)(error)
      };
    }
  }
  async calculateDiff(soClient, packagePolicy, packageInfo) {
    var _packagePolicy$packag7;
    const updatedPackagePolicy = updatePackageInputs({
      ...(0, _lodash.omit)(packagePolicy, 'id'),
      inputs: packagePolicy.inputs,
      package: {
        ...packagePolicy.package,
        version: packageInfo.version,
        experimental_data_stream_features: (_packagePolicy$packag7 = packagePolicy.package) === null || _packagePolicy$packag7 === void 0 ? void 0 : _packagePolicy$packag7.experimental_data_stream_features
      }
    }, packageInfo, (0, _services.packageToPackagePolicyInputs)(packageInfo), true);
    updatedPackagePolicy.inputs = await _compilePackagePolicyInputs(packageInfo, updatedPackagePolicy.vars || {}, updatedPackagePolicy.inputs);
    updatedPackagePolicy.elasticsearch = packageInfo.elasticsearch;
    const hasErrors = ('errors' in updatedPackagePolicy);
    this.sendUpgradeTelemetry(packagePolicy.package, packageInfo.version, hasErrors, updatedPackagePolicy.errors);
    return {
      name: updatedPackagePolicy.name,
      diff: [packagePolicy, updatedPackagePolicy],
      // TODO: Currently only returns the agent inputs for current package policy, not the upgraded one
      // as we only show this version in the UI
      agent_diff: [(0, _agent_policies.storedPackagePolicyToAgentInputs)(packagePolicy, packageInfo)],
      hasErrors
    };
  }
  sendUpgradeTelemetry(packagePolicyPackage, latestVersion, hasErrors, errors) {
    if (packagePolicyPackage.version !== latestVersion) {
      const upgradeTelemetry = {
        packageName: packagePolicyPackage.name,
        currentVersion: packagePolicyPackage.version,
        newVersion: latestVersion,
        status: hasErrors ? 'failure' : 'success',
        error: hasErrors ? errors : undefined,
        dryRun: true,
        eventType: 'package-policy-upgrade'
      };
      (0, _upgrade_sender.sendTelemetryEvents)(_.appContextService.getLogger(), _.appContextService.getTelemetryEventsSender(), upgradeTelemetry);
      _.appContextService.getLogger().info(`Package policy upgrade dry run ${hasErrors ? 'resulted in errors' : 'ran successfully'}`);
      _.appContextService.getLogger().debug(JSON.stringify(upgradeTelemetry));
    }
  }
  async enrichPolicyWithDefaultsFromPackage(soClient, newPolicy) {
    let newPackagePolicy = newPolicy;
    if (newPolicy.package) {
      const newPP = await this.buildPackagePolicyFromPackageWithVersion(soClient, newPolicy.package.name, newPolicy.package.version);
      if (newPP) {
        var _newPolicy$namespace, _newPolicy$descriptio, _newPolicy$enabled, _newPolicy$package, _newPolicy$policy_id, _newPolicy$inputs$;
        const inputs = newPolicy.inputs.map(input => {
          var _defaultInput$streams;
          const defaultInput = newPP.inputs.find(i => i.type === input.type && (!input.policy_template || input.policy_template === i.policy_template));
          return {
            ...defaultInput,
            enabled: input.enabled,
            type: input.type,
            // to propagate "enabled: false" to streams
            streams: defaultInput === null || defaultInput === void 0 ? void 0 : (_defaultInput$streams = defaultInput.streams) === null || _defaultInput$streams === void 0 ? void 0 : _defaultInput$streams.map(stream => ({
              ...stream,
              enabled: input.enabled
            }))
          };
        });
        let agentPolicyId;
        // fallback to first agent policy id in case no policy_id is specified, BWC with 8.0
        if (!newPolicy.policy_id) {
          const {
            items: agentPolicies
          } = await _agent_policy.agentPolicyService.list(soClient, {
            perPage: 1
          });
          if (agentPolicies.length > 0) {
            agentPolicyId = agentPolicies[0].id;
          }
        }
        newPackagePolicy = {
          ...newPP,
          name: newPolicy.name,
          namespace: (_newPolicy$namespace = newPolicy.namespace) !== null && _newPolicy$namespace !== void 0 ? _newPolicy$namespace : 'default',
          description: (_newPolicy$descriptio = newPolicy.description) !== null && _newPolicy$descriptio !== void 0 ? _newPolicy$descriptio : '',
          enabled: (_newPolicy$enabled = newPolicy.enabled) !== null && _newPolicy$enabled !== void 0 ? _newPolicy$enabled : true,
          package: {
            ...newPP.package,
            experimental_data_stream_features: (_newPolicy$package = newPolicy.package) === null || _newPolicy$package === void 0 ? void 0 : _newPolicy$package.experimental_data_stream_features
          },
          policy_id: (_newPolicy$policy_id = newPolicy.policy_id) !== null && _newPolicy$policy_id !== void 0 ? _newPolicy$policy_id : agentPolicyId,
          inputs: (_newPolicy$inputs$ = newPolicy.inputs[0]) !== null && _newPolicy$inputs$ !== void 0 && _newPolicy$inputs$.streams ? newPolicy.inputs : inputs,
          vars: newPolicy.vars || newPP.vars
        };
      }
    }
    return newPackagePolicy;
  }
  async buildPackagePolicyFromPackageWithVersion(soClient, pkgName, pkgVersion) {
    const packageInfo = await (0, _packages.getPackageInfo)({
      savedObjectsClient: soClient,
      pkgName,
      pkgVersion,
      skipArchive: true,
      prerelease: true
    });
    if (packageInfo) {
      return (0, _services.packageToPackagePolicy)(packageInfo, '');
    }
  }
  async buildPackagePolicyFromPackage(soClient, pkgName, logger) {
    const pkgInstall = await (0, _packages.getInstallation)({
      savedObjectsClient: soClient,
      pkgName,
      logger
    });
    if (pkgInstall) {
      const packageInfo = await (0, _packages.getPackageInfo)({
        savedObjectsClient: soClient,
        pkgName: pkgInstall.name,
        pkgVersion: pkgInstall.version,
        prerelease: true
      });
      if (packageInfo) {
        return (0, _services.packageToPackagePolicy)(packageInfo, '');
      }
    }
  }
  async runExternalCallbacks(externalCallbackType, packagePolicy, soClient, esClient, context, request) {
    var _appContextService$ge;
    const logger = _.appContextService.getLogger();
    const numberOfCallbacks = (_appContextService$ge = _.appContextService.getExternalCallbacks(externalCallbackType)) === null || _appContextService$ge === void 0 ? void 0 : _appContextService$ge.size;
    logger.debug(`Running ${numberOfCallbacks} external callbacks for ${externalCallbackType}`);
    try {
      if (externalCallbackType === 'packagePolicyPostDelete') {
        return await this.runPostDeleteExternalCallbacks(packagePolicy, soClient, esClient, context, request);
      } else if (externalCallbackType === 'packagePolicyDelete') {
        return await this.runDeleteExternalCallbacks(packagePolicy, soClient, esClient);
      } else {
        if (!Array.isArray(packagePolicy)) {
          let newData = packagePolicy;
          const externalCallbacks = _.appContextService.getExternalCallbacks(externalCallbackType);
          if (externalCallbacks && externalCallbacks.size > 0) {
            let updatedNewData = newData;
            for (const callback of externalCallbacks) {
              let result;
              if (externalCallbackType === 'packagePolicyPostCreate') {
                result = await callback(updatedNewData, soClient, esClient, context, request);
                updatedNewData = _types.PackagePolicySchema.validate(result);
              } else {
                result = await callback(updatedNewData, soClient, esClient, context, request);
              }
              if (externalCallbackType === 'packagePolicyCreate') {
                updatedNewData = _types.NewPackagePolicySchema.validate(result);
              } else if (externalCallbackType === 'packagePolicyUpdate') {
                const omitted = {
                  ...(0, _lodash.omit)(result, ['id', 'version', 'revision', 'updated_at', 'updated_by', 'created_at', 'created_by', 'elasticsearch']),
                  inputs: result.inputs.map(input => (0, _lodash.omit)(input, ['compiled_input']))
                };
                updatedNewData = _types.UpdatePackagePolicySchema.validate(omitted);
              }
            }
            newData = updatedNewData;
          }
          return newData;
        }
      }
    } catch (error) {
      logger.error(`Error running external callbacks for ${externalCallbackType}:`);
      logger.error(error);
      throw error;
    }
  }
  async runPostDeleteExternalCallbacks(deletedPackagePolicies, soClient, esClient, context, request) {
    const externalCallbacks = _.appContextService.getExternalCallbacks('packagePolicyPostDelete');
    const errorsThrown = [];
    if (externalCallbacks && externalCallbacks.size > 0) {
      for (const callback of externalCallbacks) {
        // Failures from an external callback should not prevent other external callbacks from being
        // executed. Errors (if any) will be collected and `throw`n after processing the entire set
        try {
          await callback(deletedPackagePolicies, soClient, esClient, context, request);
        } catch (error) {
          errorsThrown.push(error);
        }
      }
      if (errorsThrown.length > 0) {
        throw new _errors.FleetError(`${errorsThrown.length} encountered while executing package post delete external callbacks`, errorsThrown);
      }
    }
  }
  async runDeleteExternalCallbacks(deletedPackagePolices, soClient, esClient) {
    const externalCallbacks = _.appContextService.getExternalCallbacks('packagePolicyDelete');
    const errorsThrown = [];
    if (externalCallbacks && externalCallbacks.size > 0) {
      for (const callback of externalCallbacks) {
        // Failures from an external callback should not prevent other external callbacks from being
        // executed. Errors (if any) will be collected and `throw`n after processing the entire set
        try {
          await callback(deletedPackagePolices, soClient, esClient);
        } catch (error) {
          errorsThrown.push(error);
        }
      }
      if (errorsThrown.length > 0) {
        throw new _errors.FleetError(`${errorsThrown.length} encountered while executing package delete external callbacks`, errorsThrown);
      }
    }
  }
  fetchAllItemIds(soClient, {
    perPage = 1000,
    kuery
  } = {}) {
    // TODO:PT Question for fleet team: do I need to `auditLoggingService.writeCustomSoAuditLog()` here? Its only IDs

    return (0, _create_so_find_iterable.createSoFindIterable)({
      soClient,
      findRequest: {
        type: SAVED_OBJECT_TYPE,
        perPage,
        sortField: 'created_at',
        sortOrder: 'asc',
        fields: [],
        filter: kuery ? (0, _saved_object.normalizeKuery)(SAVED_OBJECT_TYPE, kuery) : undefined
      },
      resultsMapper: data => {
        return data.saved_objects.map(packagePolicySO => packagePolicySO.id);
      }
    });
  }
  fetchAllItems(soClient, {
    perPage = 1000,
    kuery,
    sortOrder = 'asc',
    sortField = 'created_at'
  } = {}) {
    return (0, _create_so_find_iterable.createSoFindIterable)({
      soClient,
      findRequest: {
        type: SAVED_OBJECT_TYPE,
        sortField,
        sortOrder,
        perPage,
        filter: kuery ? (0, _saved_object.normalizeKuery)(SAVED_OBJECT_TYPE, kuery) : undefined
      },
      resultsMapper(data) {
        return data.saved_objects.map(packagePolicySO => {
          _audit_logging.auditLoggingService.writeCustomSoAuditLog({
            action: 'find',
            id: packagePolicySO.id,
            savedObjectType: _constants3.PACKAGE_POLICY_SAVED_OBJECT_TYPE
          });
          return (0, _package_policies.mapPackagePolicySavedObjectToPackagePolicy)(packagePolicySO);
        });
      }
    });
  }
}
class PackagePolicyServiceImpl extends PackagePolicyClientImpl {
  asScoped(request) {
    const preflightCheck = async ({
      fleetAuthz: fleetRequiredAuthz
    }) => {
      const authz = await (0, _security.getAuthzFromRequest)(request);
      if ((0, _security.doesNotHaveRequiredFleetAuthz)(authz, fleetRequiredAuthz)) {
        throw new _errors.FleetUnauthorizedError('Not authorized to this action on integration policies');
      }
    };
    return new PackagePolicyClientWithAuthz(preflightCheck);
  }
  get asInternalUser() {
    return new PackagePolicyClientWithAuthz();
  }
}
exports.PackagePolicyServiceImpl = PackagePolicyServiceImpl;
var _runPreflight = /*#__PURE__*/new WeakMap();
class PackagePolicyClientWithAuthz extends PackagePolicyClientImpl {
  constructor(preflightCheck) {
    super();
    _classPrivateFieldInitSpec(this, _runPreflight, {
      writable: true,
      value: async fleetAuthzConfig => {
        if (this.preflightCheck) {
          return await this.preflightCheck(fleetAuthzConfig);
        }
      }
    });
    this.preflightCheck = preflightCheck;
  }
  async create(soClient, esClient, packagePolicy, options, context, request) {
    await (0, _classPrivateFieldGet2.default)(this, _runPreflight).call(this, {
      fleetAuthz: {
        integrations: {
          writeIntegrationPolicies: true
        }
      }
    });
    return super.create(soClient, esClient, packagePolicy, options, context, request);
  }
}
function validatePackagePolicyOrThrow(packagePolicy, pkgInfo) {
  const validationResults = (0, _services.validatePackagePolicy)(packagePolicy, pkgInfo, _jsYaml.safeLoad);
  if ((0, _services.validationHasErrors)(validationResults)) {
    const responseFormattedValidationErrors = Object.entries((0, _std.getFlattenedObject)(validationResults)).map(([key, value]) => ({
      key,
      message: value
    })).filter(({
      message
    }) => !!message);
    if (responseFormattedValidationErrors.length) {
      throw new _errors.PackagePolicyValidationError(_i18n.i18n.translate('xpack.fleet.packagePolicyInvalidError', {
        defaultMessage: 'Package policy is invalid: {errors}',
        values: {
          errors: responseFormattedValidationErrors.map(({
            key,
            message
          }) => `${key}: ${message}`).join('\n')
        }
      }));
    }
  }
}

// the option `allEnabled` is only used to generate inputs integration templates where everything is enabled by default
// it shouldn't be used in the normal install flow
function getInputsWithStreamIds(packagePolicy, packagePolicyId, allEnabled) {
  return packagePolicy.inputs.map(input => {
    return {
      ...input,
      enabled: !!allEnabled ? true : input.enabled,
      streams: input.streams.map(stream => ({
        ...stream,
        enabled: !!allEnabled ? true : stream.enabled,
        id: packagePolicyId ? `${input.type}-${stream.data_stream.dataset}-${packagePolicyId}` : `${input.type}-${stream.data_stream.dataset}`
      }))
    };
  });
}
async function _compilePackagePolicyInputs(pkgInfo, vars, inputs) {
  const inputsPromises = inputs.map(async input => {
    const compiledInput = await _compilePackagePolicyInput(pkgInfo, vars, input);
    const compiledStreams = await _compilePackageStreams(pkgInfo, vars, input);
    return {
      ...input,
      compiled_input: compiledInput,
      streams: compiledStreams
    };
  });
  return Promise.all(inputsPromises);
}
async function _compilePackagePolicyInput(pkgInfo, vars, input) {
  var _pkgInfo$policy_templ, _pkgInfo$policy_templ2;
  const packagePolicyTemplate = input.policy_template ? (_pkgInfo$policy_templ = pkgInfo.policy_templates) === null || _pkgInfo$policy_templ === void 0 ? void 0 : _pkgInfo$policy_templ.find(policyTemplate => policyTemplate.name === input.policy_template) : (_pkgInfo$policy_templ2 = pkgInfo.policy_templates) === null || _pkgInfo$policy_templ2 === void 0 ? void 0 : _pkgInfo$policy_templ2[0];
  if (!input.enabled || !packagePolicyTemplate) {
    return undefined;
  }
  const packageInputs = (0, _services.getNormalizedInputs)(packagePolicyTemplate);
  if (!packageInputs.length) {
    return undefined;
  }
  const packageInput = packageInputs.find(pkgInput => pkgInput.type === input.type);
  if (!packageInput) {
    throw new Error(`Input template not found, unable to find input type ${input.type}`);
  }
  if (!packageInput.template_path) {
    return undefined;
  }
  const [pkgInputTemplate] = await (0, _assets.getAssetsData)(pkgInfo, path => path.endsWith(`/agent/input/${packageInput.template_path}`));
  if (!pkgInputTemplate || !pkgInputTemplate.buffer) {
    throw new Error(`Unable to load input template at /agent/input/${packageInput.template_path}`);
  }
  return (0, _agent.compileTemplate)(
  // Populate template variables from package- and input-level vars
  Object.assign({}, vars, input.vars), pkgInputTemplate.buffer.toString());
}
async function _compilePackageStreams(pkgInfo, vars, input) {
  const streamsPromises = input.streams.map(stream => _compilePackageStream(pkgInfo, vars, input, stream));
  return await Promise.all(streamsPromises);
}

// temporary export to enable testing pending refactor https://github.com/elastic/kibana/issues/112386
// TODO: Move this logic into `package_policies_to_agent_permissions.ts` since this is not a package policy concern
// and is based entirely on the package contents
function _applyIndexPrivileges(packageDataStream, stream) {
  var _packageDataStream$el, _packageDataStream$el2, _packageDataStream$el3, _packageDataStream$el4;
  const streamOut = (0, _lodash.cloneDeep)(stream);
  if (packageDataStream !== null && packageDataStream !== void 0 && (_packageDataStream$el = packageDataStream.elasticsearch) !== null && _packageDataStream$el !== void 0 && _packageDataStream$el.dynamic_dataset) {
    var _streamOut$data_strea;
    streamOut.data_stream.elasticsearch = (_streamOut$data_strea = streamOut.data_stream.elasticsearch) !== null && _streamOut$data_strea !== void 0 ? _streamOut$data_strea : {};
    streamOut.data_stream.elasticsearch.dynamic_dataset = packageDataStream.elasticsearch.dynamic_dataset;
  }
  if (packageDataStream !== null && packageDataStream !== void 0 && (_packageDataStream$el2 = packageDataStream.elasticsearch) !== null && _packageDataStream$el2 !== void 0 && _packageDataStream$el2.dynamic_namespace) {
    var _streamOut$data_strea2;
    streamOut.data_stream.elasticsearch = (_streamOut$data_strea2 = streamOut.data_stream.elasticsearch) !== null && _streamOut$data_strea2 !== void 0 ? _streamOut$data_strea2 : {};
    streamOut.data_stream.elasticsearch.dynamic_namespace = packageDataStream.elasticsearch.dynamic_namespace;
  }
  const indexPrivileges = packageDataStream === null || packageDataStream === void 0 ? void 0 : (_packageDataStream$el3 = packageDataStream.elasticsearch) === null || _packageDataStream$el3 === void 0 ? void 0 : (_packageDataStream$el4 = _packageDataStream$el3.privileges) === null || _packageDataStream$el4 === void 0 ? void 0 : _packageDataStream$el4.indices;
  if (!(indexPrivileges !== null && indexPrivileges !== void 0 && indexPrivileges.length)) {
    return streamOut;
  }
  const [valid, invalid] = (0, _lodash.partition)(indexPrivileges, permission => DATA_STREAM_ALLOWED_INDEX_PRIVILEGES.has(permission));
  if (invalid.length) {
    _.appContextService.getLogger().warn(`Ignoring invalid or forbidden index privilege(s) in "${stream.id}" data stream: ${invalid}`);
  }
  if (valid.length) {
    var _streamOut$data_strea3;
    streamOut.data_stream.elasticsearch = (_streamOut$data_strea3 = streamOut.data_stream.elasticsearch) !== null && _streamOut$data_strea3 !== void 0 ? _streamOut$data_strea3 : {};
    streamOut.data_stream.elasticsearch.privileges = {
      indices: valid
    };
  }
  return streamOut;
}
async function _compilePackageStream(pkgInfo, vars, input, streamIn) {
  let stream = streamIn;
  if (!stream.enabled) {
    return {
      ...stream,
      compiled_stream: undefined
    };
  }
  const packageDataStreams = (0, _services.getNormalizedDataStreams)(pkgInfo);
  if (!packageDataStreams) {
    throw new Error('Stream template not found, no data streams');
  }
  const packageDataStream = packageDataStreams.find(pkgDataStream => pkgDataStream.dataset === stream.data_stream.dataset);
  if (!packageDataStream) {
    throw new Error(`Stream template not found, unable to find dataset ${stream.data_stream.dataset}`);
  }
  stream = _applyIndexPrivileges(packageDataStream, streamIn);
  const streamFromPkg = (packageDataStream.streams || []).find(pkgStream => pkgStream.input === input.type);
  if (!streamFromPkg) {
    throw new Error(`Stream template not found, unable to find stream for input ${input.type}`);
  }
  if (!streamFromPkg.template_path) {
    throw new Error(`Stream template path not found for dataset ${stream.data_stream.dataset}`);
  }
  const datasetPath = packageDataStream.path;
  const [pkgStreamTemplate] = await (0, _assets.getAssetsData)(pkgInfo, path => path.endsWith(streamFromPkg.template_path), datasetPath);
  if (!pkgStreamTemplate || !pkgStreamTemplate.buffer) {
    throw new Error(`Unable to load stream template ${streamFromPkg.template_path} for dataset ${stream.data_stream.dataset}`);
  }
  const yaml = (0, _agent.compileTemplate)(
  // Populate template variables from package-, input-, and stream-level vars
  Object.assign({}, vars, input.vars, stream.vars), pkgStreamTemplate.buffer.toString());
  stream.compiled_stream = yaml;
  return {
    ...stream
  };
}
function enforceFrozenInputs(oldInputs, newInputs, force = false) {
  const resultInputs = [...newInputs];
  for (const input of resultInputs) {
    const oldInput = oldInputs.find(i => i.type === input.type);
    if (oldInput !== null && oldInput !== void 0 && oldInput.keep_enabled) input.enabled = oldInput.enabled;
    if (input.vars && oldInput !== null && oldInput !== void 0 && oldInput.vars) {
      input.vars = _enforceFrozenVars(oldInput.vars, input.vars, force);
    }
    if (input.streams && oldInput !== null && oldInput !== void 0 && oldInput.streams) {
      for (const stream of input.streams) {
        const oldStream = oldInput.streams.find(s => s.id === stream.id);
        if (oldStream !== null && oldStream !== void 0 && oldStream.keep_enabled) stream.enabled = oldStream.enabled;
        if (stream.vars && oldStream !== null && oldStream !== void 0 && oldStream.vars) {
          stream.vars = _enforceFrozenVars(oldStream.vars, stream.vars, force);
        }
      }
    }
  }
  return resultInputs;
}
function _enforceFrozenVars(oldVars, newVars, force = false) {
  const resultVars = {};
  for (const [key, val] of Object.entries(newVars)) {
    var _oldVars$key;
    if ((_oldVars$key = oldVars[key]) !== null && _oldVars$key !== void 0 && _oldVars$key.frozen) {
      if (force) {
        resultVars[key] = val;
      } else if (!(0, _lodash.isEqual)(oldVars[key].value, val.value) || oldVars[key].type !== val.type) {
        throw new _errors.PackagePolicyValidationError(`${key} is a frozen variable and cannot be modified`);
      } else {
        resultVars[key] = oldVars[key];
      }
    } else {
      resultVars[key] = val;
    }
  }
  for (const [key, val] of Object.entries(oldVars)) {
    if (!newVars[key] && val.frozen) {
      resultVars[key] = val;
    }
  }
  return resultVars;
}
const packagePolicyService = new PackagePolicyClientImpl();
exports.packagePolicyService = packagePolicyService;
async function getPackageInfoForPackagePolicies(packagePolicies, soClient) {
  const pkgInfoMap = new Map();
  packagePolicies.forEach(({
    package: pkg
  }) => {
    if (pkg) {
      pkgInfoMap.set(`${pkg.name}-${pkg.version}`, pkg);
    }
  });
  const resultMap = new Map();
  await (0, _pMap.default)(pkgInfoMap.keys(), async pkgKey => {
    const pkgInfo = pkgInfoMap.get(pkgKey);
    if (pkgInfo) {
      const pkgInfoData = await (0, _packages.getPackageInfo)({
        savedObjectsClient: soClient,
        pkgName: pkgInfo.name,
        pkgVersion: pkgInfo.version,
        prerelease: true
      });
      resultMap.set(pkgKey, pkgInfoData);
    }
  });
  return resultMap;
}
function updatePackageInputs(basePackagePolicy, packageInfo, inputsUpdated, dryRun) {
  var _packageInfo$policy_t;
  if (!inputsUpdated) return basePackagePolicy;
  const availablePolicyTemplates = (_packageInfo$policy_t = packageInfo.policy_templates) !== null && _packageInfo$policy_t !== void 0 ? _packageInfo$policy_t : [];
  const inputs = [...basePackagePolicy.inputs.filter(input => {
    var _policyTemplate$input, _policyTemplate$input2;
    if (!input.policy_template) {
      return true;
    }
    const policyTemplate = availablePolicyTemplates.find(({
      name
    }) => name === input.policy_template);

    // Ignore any policy templates removed in the new package version
    if (!policyTemplate) {
      return false;
    }

    // Ignore any inputs removed from this policy template in the new package version
    const policyTemplateStillIncludesInput = (0, _services.isInputOnlyPolicyTemplate)(policyTemplate) ? policyTemplate.input === input.type : (_policyTemplate$input = (_policyTemplate$input2 = policyTemplate.inputs) === null || _policyTemplate$input2 === void 0 ? void 0 : _policyTemplate$input2.some(policyTemplateInput => policyTemplateInput.type === input.type)) !== null && _policyTemplate$input !== void 0 ? _policyTemplate$input : false;
    return policyTemplateStillIncludesInput;
  })];
  for (const update of inputsUpdated) {
    let originalInput;
    if (update.policy_template) {
      // If the updated value defines a policy template, try to find an original input
      // with the same policy template value
      const matchingInput = inputs.find(i => i.type === update.type && i.policy_template === update.policy_template);

      // If we didn't find an input with the same policy template, try to look for one
      // with the same type, but with an undefined policy template. This ensures we catch
      // cases where we're upgrading an older policy from before policy template was
      // reliably define on package policy inputs.
      originalInput = matchingInput || inputs.find(i => i.type === update.type && !i.policy_template);
    } else {
      // For inputs that don't specify a policy template, just grab the first input
      // that matches its `type`
      originalInput = inputs.find(i => i.type === update.type);
    }

    // If there's no corresponding input on the original package policy, just
    // take the override value from the new package as-is. This case typically
    // occurs when inputs or package policy templates are added/removed between versions.
    if (originalInput === undefined) {
      inputs.push(update);
      continue;
    }

    // For flags like this, we only want to override the original value if it was set
    // as `undefined` in the original object. An explicit true/false value should be
    // persisted from the original object to the result after the override process is complete.
    if (originalInput.enabled === undefined && update.enabled !== undefined) {
      originalInput.enabled = update.enabled;
    }
    if (originalInput.keep_enabled === undefined && update.keep_enabled !== undefined) {
      originalInput.keep_enabled = update.keep_enabled;
    }

    // `policy_template` should always be defined, so if we have an older policy here we need
    // to ensure we set it
    if (originalInput.policy_template === undefined && update.policy_template !== undefined) {
      originalInput.policy_template = update.policy_template;
    }
    if (update.vars) {
      const indexOfInput = inputs.indexOf(originalInput);
      inputs[indexOfInput] = deepMergeVars(originalInput, update, true);
      originalInput = inputs[indexOfInput];
    }
    if (update.streams) {
      var _originalInput;
      const isInputPkgUpdate = packageInfo.type === 'input' && update.streams.length === 1 && ((_originalInput = originalInput) === null || _originalInput === void 0 ? void 0 : _originalInput.streams.length) === 1;
      for (const stream of update.streams) {
        var _originalInput2, _originalStream;
        let originalStream = (_originalInput2 = originalInput) === null || _originalInput2 === void 0 ? void 0 : _originalInput2.streams.find(s => s.data_stream.dataset === stream.data_stream.dataset);

        // this handles the input only pkg case where the new stream cannot have a dataset name
        // so will never match. Input only packages only ever have one stream.
        if (!originalStream && isInputPkgUpdate) {
          var _originalInput3;
          originalStream = {
            ...update.streams[0],
            vars: (_originalInput3 = originalInput) === null || _originalInput3 === void 0 ? void 0 : _originalInput3.streams[0].vars
          };
        }
        if (originalStream === undefined) {
          originalInput.streams.push(stream);
          continue;
        }
        if (((_originalStream = originalStream) === null || _originalStream === void 0 ? void 0 : _originalStream.enabled) === undefined) {
          originalStream.enabled = stream.enabled;
        }
        if (stream.vars) {
          // streams wont match for input pkgs
          const indexOfStream = isInputPkgUpdate ? 0 : originalInput.streams.indexOf(originalStream);
          originalInput.streams[indexOfStream] = deepMergeVars(originalStream, stream, true);
          originalStream = originalInput.streams[indexOfStream];
        }
      }
    }

    // Filter all stream that have been removed from the input
    originalInput.streams = originalInput.streams.filter(originalStream => {
      var _update$streams$some, _update$streams;
      return (_update$streams$some = (_update$streams = update.streams) === null || _update$streams === void 0 ? void 0 : _update$streams.some(s => s.data_stream.dataset === originalStream.data_stream.dataset)) !== null && _update$streams$some !== void 0 ? _update$streams$some : false;
    });
  }
  const resultingPackagePolicy = {
    ...basePackagePolicy,
    inputs
  };
  const validationResults = (0, _services.validatePackagePolicy)(resultingPackagePolicy, packageInfo, _jsYaml.safeLoad);
  if ((0, _services.validationHasErrors)(validationResults)) {
    const responseFormattedValidationErrors = Object.entries((0, _std.getFlattenedObject)(validationResults)).map(([key, value]) => ({
      key,
      message: value
    })).filter(({
      message
    }) => !!message);
    if (responseFormattedValidationErrors.length) {
      if (dryRun) {
        return {
          ...resultingPackagePolicy,
          errors: responseFormattedValidationErrors
        };
      }
      throw new _errors.PackagePolicyValidationError(_i18n.i18n.translate('xpack.fleet.packagePolicyInvalidError', {
        defaultMessage: 'Package policy is invalid: {errors}',
        values: {
          errors: responseFormattedValidationErrors.map(({
            key,
            message
          }) => `${key}: ${message}`).join('\n')
        }
      }));
    }
  }
  return resultingPackagePolicy;
}
function preconfigurePackageInputs(basePackagePolicy, packageInfo, preconfiguredInputs) {
  if (!preconfiguredInputs) return basePackagePolicy;
  const inputs = [...basePackagePolicy.inputs];
  for (const preconfiguredInput of preconfiguredInputs) {
    // Preconfiguration does not currently support multiple policy templates, so overrides will have an undefined
    // policy template, so we only match on `type` in that case.
    let originalInput = preconfiguredInput.policy_template ? inputs.find(i => i.type === preconfiguredInput.type && i.policy_template === preconfiguredInput.policy_template) : inputs.find(i => i.type === preconfiguredInput.type);

    // If the input do not exist skip
    if (originalInput === undefined) {
      continue;
    }
    if (preconfiguredInput.enabled !== undefined) {
      originalInput.enabled = preconfiguredInput.enabled;
    }
    if (preconfiguredInput.keep_enabled !== undefined) {
      originalInput.keep_enabled = preconfiguredInput.keep_enabled;
    }
    if (preconfiguredInput.vars) {
      const indexOfInput = inputs.indexOf(originalInput);
      inputs[indexOfInput] = deepMergeVars(originalInput, preconfiguredInput);
      originalInput = inputs[indexOfInput];
    }
    if (preconfiguredInput.streams) {
      for (const stream of preconfiguredInput.streams) {
        var _originalInput4;
        let originalStream = (_originalInput4 = originalInput) === null || _originalInput4 === void 0 ? void 0 : _originalInput4.streams.find(s => s.data_stream.dataset === stream.data_stream.dataset);
        if (originalStream === undefined) {
          continue;
        }
        if (stream.enabled !== undefined) {
          originalStream.enabled = stream.enabled;
        }
        if (stream.vars) {
          const indexOfStream = originalInput.streams.indexOf(originalStream);
          originalInput.streams[indexOfStream] = deepMergeVars(originalStream, stream);
          originalStream = originalInput.streams[indexOfStream];
        }
      }
    }
  }
  const resultingPackagePolicy = {
    ...basePackagePolicy,
    inputs
  };
  validatePackagePolicyOrThrow(resultingPackagePolicy, packageInfo);
  return resultingPackagePolicy;
}

// input only packages cannot have their namespace or dataset modified
function _validateRestrictedFieldsNotModifiedOrThrow(opts) {
  const {
    pkgInfo,
    oldPackagePolicy,
    packagePolicyUpdate
  } = opts;
  if (pkgInfo.type !== 'input') return;
  const {
    namespace,
    inputs
  } = packagePolicyUpdate;
  if (namespace && namespace !== oldPackagePolicy.namespace) {
    throw new _errors.PackagePolicyValidationError(_i18n.i18n.translate('xpack.fleet.updatePackagePolicy.namespaceCannotBeModified', {
      defaultMessage: 'Package policy namespace cannot be modified for input only packages, please create a new package policy.'
    }));
  }
  if (inputs) {
    for (const input of inputs) {
      const oldInput = oldPackagePolicy.inputs.find(i => i.id === input.id);
      if (oldInput) {
        for (const stream of input.streams || []) {
          var _oldStream$vars, _oldStream$vars$DATAS, _stream$vars, _stream$vars$DATASET_;
          const oldStream = oldInput.streams.find(s => s.data_stream.dataset === stream.data_stream.dataset);
          if (oldStream && oldStream !== null && oldStream !== void 0 && (_oldStream$vars = oldStream.vars) !== null && _oldStream$vars !== void 0 && _oldStream$vars[_constants2.DATASET_VAR_NAME] && (oldStream === null || oldStream === void 0 ? void 0 : (_oldStream$vars$DATAS = oldStream.vars[_constants2.DATASET_VAR_NAME]) === null || _oldStream$vars$DATAS === void 0 ? void 0 : _oldStream$vars$DATAS.value) !== (stream === null || stream === void 0 ? void 0 : (_stream$vars = stream.vars) === null || _stream$vars === void 0 ? void 0 : (_stream$vars$DATASET_ = _stream$vars[_constants2.DATASET_VAR_NAME]) === null || _stream$vars$DATASET_ === void 0 ? void 0 : _stream$vars$DATASET_.value)) {
            var _oldStream$vars$DATAS2, _stream$vars2, _stream$vars2$DATASET;
            // seeing this error in dev? Package policy must be called with prepareInputPackagePolicyDataset function first in UI code
            _.appContextService.getLogger().debug(`Rejecting package policy update due to dataset change, old val '${oldStream === null || oldStream === void 0 ? void 0 : (_oldStream$vars$DATAS2 = oldStream.vars[_constants2.DATASET_VAR_NAME]) === null || _oldStream$vars$DATAS2 === void 0 ? void 0 : _oldStream$vars$DATAS2.value}, new val '${JSON.stringify(stream === null || stream === void 0 ? void 0 : (_stream$vars2 = stream.vars) === null || _stream$vars2 === void 0 ? void 0 : (_stream$vars2$DATASET = _stream$vars2[_constants2.DATASET_VAR_NAME]) === null || _stream$vars2$DATASET === void 0 ? void 0 : _stream$vars2$DATASET.value)}'`);
            throw new _errors.PackagePolicyValidationError(_i18n.i18n.translate('xpack.fleet.updatePackagePolicy.datasetCannotBeModified', {
              defaultMessage: 'Package policy dataset cannot be modified for input only packages, please create a new package policy.'
            }));
          }
        }
      }
    }
  }
}
async function validateIsNotHostedPolicy(soClient, id, force = false, errorMessage) {
  const agentPolicy = await _agent_policy.agentPolicyService.get(soClient, id, false);
  if (!agentPolicy) {
    throw new Error('Agent policy not found');
  }
  if (agentPolicy.is_managed && !force) {
    throw new _errors.HostedAgentPolicyRestrictionRelatedError(errorMessage !== null && errorMessage !== void 0 ? errorMessage : `Cannot update integrations of hosted agent policy ${id}`);
  }
}
function sendUpdatePackagePolicyTelemetryEvent(soClient, updatedPkgPolicies, oldPackagePolicies) {
  updatedPkgPolicies.forEach(updatedPkgPolicy => {
    if (updatedPkgPolicy.package) {
      var _oldPkgPolicy$package;
      const {
        name,
        version
      } = updatedPkgPolicy.package;
      const oldPkgPolicy = oldPackagePolicies.find(packagePolicy => packagePolicy.id === updatedPkgPolicy.id);
      const oldVersion = oldPkgPolicy === null || oldPkgPolicy === void 0 ? void 0 : (_oldPkgPolicy$package = oldPkgPolicy.package) === null || _oldPkgPolicy$package === void 0 ? void 0 : _oldPkgPolicy$package.version;
      if (oldVersion && oldVersion !== version) {
        const upgradeTelemetry = {
          packageName: name,
          currentVersion: oldVersion,
          newVersion: version,
          status: 'success',
          eventType: 'package-policy-upgrade'
        };
        (0, _upgrade_sender.sendTelemetryEvents)(_.appContextService.getLogger(), _.appContextService.getTelemetryEventsSender(), upgradeTelemetry);
        _.appContextService.getLogger().info(`Package policy upgraded successfully`);
        _.appContextService.getLogger().debug(JSON.stringify(upgradeTelemetry));
      }
    }
  });
}
function deepMergeVars(original, override, keepOriginalValue = false) {
  if (!original.vars) {
    original.vars = {
      ...override.vars
    };
  }
  const result = {
    ...original
  };
  const overrideVars = Array.isArray(override.vars) ? override.vars : Object.entries(override.vars).map(([key, rest]) => ({
    name: key,
    ...rest
  }));
  for (const {
    name,
    ...overrideVal
  } of overrideVars) {
    const originalVar = original.vars[name];
    result.vars[name] = {
      ...originalVar,
      ...overrideVal
    };

    // Ensure that any value from the original object is persisted on the newly merged resulting object,
    // even if we merge other data about the given variable
    if (keepOriginalValue && (originalVar === null || originalVar === void 0 ? void 0 : originalVar.value) !== undefined) {
      result.vars[name].value = originalVar.value;
    }
  }
  return result;
}
async function requireUniqueName(soClient, packagePolicy, id) {
  const existingPoliciesWithName = await packagePolicyService.list(soClient, {
    perPage: _constants2.SO_SEARCH_LIMIT,
    kuery: `${_constants3.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.name:"${packagePolicy.name}"`
  });
  const policiesToCheck = id ?
  // Check that the name does not exist already but exclude the current package policy
  ((existingPoliciesWithName === null || existingPoliciesWithName === void 0 ? void 0 : existingPoliciesWithName.items) || []).filter(p => p.id !== id) : existingPoliciesWithName === null || existingPoliciesWithName === void 0 ? void 0 : existingPoliciesWithName.items;
  if (policiesToCheck.length > 0) {
    throw new _errors.PackagePolicyNameExistsError(`An integration policy with the name ${packagePolicy.name} already exists. Please rename it or choose a different name.`);
  }
}