"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.UpdateAgentTagsActionRunner = void 0;
exports.updateTagsBatch = updateTagsBatch;
var _uuid = require("uuid");
var _lodash = require("lodash");
var _constants = require("../../constants");
var _app_context = require("../app_context");
var _action_runner = require("./action_runner");
var _bulk_action_types = require("./bulk_action_types");
var _filter_hosted_agents = require("./filter_hosted_agents");
var _actions = require("./actions");
var _retry_helper = require("./retry_helper");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class UpdateAgentTagsActionRunner extends _action_runner.ActionRunner {
  async processAgents(agents) {
    var _this$actionParams, _this$actionParams2;
    return await updateTagsBatch(this.soClient, this.esClient, agents, {}, {
      tagsToAdd: (_this$actionParams = this.actionParams) === null || _this$actionParams === void 0 ? void 0 : _this$actionParams.tagsToAdd,
      tagsToRemove: (_this$actionParams2 = this.actionParams) === null || _this$actionParams2 === void 0 ? void 0 : _this$actionParams2.tagsToRemove,
      actionId: this.actionParams.actionId,
      total: this.actionParams.total
    });
  }
  getTaskType() {
    return _bulk_action_types.BulkActionTaskType.UPDATE_AGENT_TAGS_RETRY;
  }
  getActionType() {
    return 'UPDATE_TAGS';
  }
}
exports.UpdateAgentTagsActionRunner = UpdateAgentTagsActionRunner;
async function updateTagsBatch(soClient, esClient, givenAgents, outgoingErrors, options) {
  var _options$actionId, _res$updated, _res$failures, _res$version_conflict, _options$total;
  const errors = {
    ...outgoingErrors
  };
  const hostedAgentError = `Cannot modify tags on a hosted agent`;
  const filteredAgents = await (0, _filter_hosted_agents.filterHostedPolicies)(soClient, givenAgents, errors, hostedAgentError);
  const agentIds = filteredAgents.map(agent => agent.id);
  const actionId = (_options$actionId = options.actionId) !== null && _options$actionId !== void 0 ? _options$actionId : (0, _uuid.v4)();
  if (agentIds.length === 0) {
    _app_context.appContextService.getLogger().debug('No agents to update tags, returning');
    return {
      actionId,
      updated: 0,
      took: 0
    };
  }
  _app_context.appContextService.getLogger().debug(`Agents to update tags in batch: ${agentIds.length}, tagsToAdd: ${options.tagsToAdd}, tagsToRemove: ${options.tagsToRemove}`);
  let res;
  try {
    res = await esClient.updateByQuery({
      query: {
        terms: {
          _id: agentIds
        }
      },
      index: _constants.AGENTS_INDEX,
      refresh: true,
      wait_for_completion: true,
      script: {
        source: `
      if (ctx._source.tags == null) {
        ctx._source.tags = [];
      }
      if (params.tagsToAdd.length == 1 && params.tagsToRemove.length == 1) {
        ctx._source.tags.replaceAll(tag -> params.tagsToRemove[0] == tag ? params.tagsToAdd[0] : tag);
      } else {
        ctx._source.tags.removeAll(params.tagsToRemove);
      }
      ctx._source.tags.addAll(params.tagsToAdd);

      LinkedHashSet uniqueSet = new LinkedHashSet();
      uniqueSet.addAll(ctx._source.tags);

      ctx._source.tags = uniqueSet.toArray();

      ctx._source.updated_at = params.updatedAt;
      `,
        lang: 'painless',
        params: {
          tagsToAdd: (0, _lodash.uniq)(options.tagsToAdd),
          tagsToRemove: (0, _lodash.uniq)(options.tagsToRemove),
          updatedAt: new Date().toISOString()
        }
      },
      conflicts: 'proceed' // relying on the task to retry in case of conflicts - retry only conflicted agents
    });
  } catch (error) {
    throw new Error('Caught error: ' + JSON.stringify(error).slice(0, 1000));
  }
  _app_context.appContextService.getLogger().debug(JSON.stringify(res).slice(0, 1000));

  // creating unique ids to use as agentId, as we don't have all agent ids in case of action by kuery
  const getUuidArray = count => Array.from({
    length: count
  }, () => (0, _uuid.v4)());
  const updatedCount = (_res$updated = res.updated) !== null && _res$updated !== void 0 ? _res$updated : 0;
  const updatedIds = getUuidArray(updatedCount);
  const failures = (_res$failures = res.failures) !== null && _res$failures !== void 0 ? _res$failures : [];
  const failureCount = failures.length;
  const isLastRetry = options.retryCount === _retry_helper.MAX_RETRY_COUNT;
  const versionConflictCount = (_res$version_conflict = res.version_conflicts) !== null && _res$version_conflict !== void 0 ? _res$version_conflict : 0;
  const versionConflictIds = isLastRetry ? getUuidArray(versionConflictCount) : [];

  // creating an action doc so that update tags  shows up in activity
  // the logic only saves agent count in the action that updated, failed or in case of last retry, conflicted
  // this ensures that the action status count will be accurate
  await (0, _actions.createAgentAction)(esClient, {
    id: actionId,
    agents: updatedIds.concat(failures.map(failure => failure.id)).concat(isLastRetry ? versionConflictIds : []),
    created_at: new Date().toISOString(),
    type: 'UPDATE_TAGS',
    total: (_options$total = options.total) !== null && _options$total !== void 0 ? _options$total : res.total
  });
  _app_context.appContextService.getLogger().debug(`action doc wrote on ${updatedCount + failureCount + (isLastRetry ? versionConflictCount : 0)} agentIds, updated: ${updatedCount}, failed: ${failureCount}, version_conflicts: ${versionConflictCount}`);

  // writing successful action results
  if (updatedCount > 0) {
    await (0, _actions.bulkCreateAgentActionResults)(esClient, updatedIds.map(id => ({
      agentId: id,
      actionId
    })));
    _app_context.appContextService.getLogger().debug(`action updated result wrote on ${updatedCount} agents`);
  }

  // writing failures from es update
  if (failures.length > 0) {
    await (0, _actions.bulkCreateAgentActionResults)(esClient, failures.map(failure => ({
      agentId: failure.id,
      actionId,
      error: failure.cause.reason
    })));
    _app_context.appContextService.getLogger().debug(`action failed result wrote on ${failureCount} agents`);
  }
  if (versionConflictCount > 0) {
    // write out error results on last retry, so action is not stuck in progress
    if (options.retryCount === _retry_helper.MAX_RETRY_COUNT) {
      await (0, _actions.bulkCreateAgentActionResults)(esClient, versionConflictIds.map(id => ({
        agentId: id,
        actionId,
        error: 'version conflict on last retry'
      })));
      _app_context.appContextService.getLogger().debug(`action conflict result wrote on ${versionConflictCount} agents`);
    }
    throw new Error(`version conflict of ${versionConflictCount} agents`);
  }
  return {
    actionId,
    updated: res.updated,
    took: res.took
  };
}