"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IndicesLogic = void 0;
var _kea = require("kea");
var _api = require("../../../../../common/types/api");
var _cancel_syncs_api_logic = require("../../api/connector/cancel_syncs_api_logic");
var _delete_index_api_logic = require("../../api/index/delete_index_api_logic");
var _fetch_index_api_logic = require("../../api/index/fetch_index_api_logic");
var _fetch_indices_api_logic = require("../../api/index/fetch_indices_api_logic");
var _types = require("../../types");
var _indices = require("../../utils/indices");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const IndicesLogic = (0, _kea.kea)({
  actions: {
    closeDeleteModal: true,
    fetchIndices: ({
      from,
      onlyShowSearchOptimizedIndices,
      returnHiddenIndices,
      searchQuery,
      size
    }) => ({
      from,
      onlyShowSearchOptimizedIndices,
      returnHiddenIndices,
      searchQuery,
      size
    }),
    onPaginate: newPageIndex => ({
      newPageIndex
    }),
    openDeleteModal: indexName => ({
      indexName
    }),
    setIsFirstRequest: true
  },
  connect: {
    actions: [_cancel_syncs_api_logic.CancelSyncsApiLogic, ['apiSuccess as cancelSuccess'], _fetch_index_api_logic.FetchIndexApiLogic, ['makeRequest as fetchIndexDetails'], _fetch_indices_api_logic.FetchIndicesAPILogic, ['makeRequest', 'apiSuccess', 'apiError'], _delete_index_api_logic.DeleteIndexApiLogic, ['apiError as deleteError', 'apiSuccess as deleteSuccess', 'makeRequest as deleteIndex']],
    values: [_fetch_index_api_logic.FetchIndexApiLogic, ['data as indexDetails', 'status as indexDetailsStatus'], _fetch_indices_api_logic.FetchIndicesAPILogic, ['data', 'status'], _delete_index_api_logic.DeleteIndexApiLogic, ['status as deleteStatus']]
  },
  listeners: ({
    actions,
    values
  }) => ({
    cancelSuccess: async () => {
      actions.fetchIndexDetails({
        indexName: values.deleteModalIndexName
      });
    },
    deleteSuccess: () => {
      actions.closeDeleteModal();
      actions.fetchIndices(values.searchParams);
    },
    fetchIndices: async (input, breakpoint) => {
      await breakpoint(150);
      actions.makeRequest(input);
    },
    openDeleteModal: ({
      indexName
    }) => {
      actions.fetchIndexDetails({
        indexName
      });
    }
  }),
  path: ['enterprise_search', 'content', 'indices_logic'],
  reducers: () => ({
    deleteModalIndexName: ['', {
      closeDeleteModal: () => '',
      openDeleteModal: (_, {
        indexName
      }) => indexName
    }],
    isDeleteModalVisible: [false, {
      closeDeleteModal: () => false,
      openDeleteModal: () => true
    }],
    isFirstRequest: [true, {
      apiError: () => false,
      apiSuccess: () => false,
      setIsFirstRequest: () => true
    }],
    searchParams: [{
      from: 0,
      onlyShowSearchOptimizedIndices: false,
      returnHiddenIndices: false,
      size: 20
    }, {
      apiSuccess: (_, {
        meta,
        onlyShowSearchOptimizedIndices,
        returnHiddenIndices,
        searchQuery
      }) => ({
        from: meta.page.from,
        onlyShowSearchOptimizedIndices,
        returnHiddenIndices,
        searchQuery,
        size: meta.page.size
      }),
      onPaginate: (state, {
        newPageIndex
      }) => ({
        ...state,
        from: (newPageIndex - 1) * state.size
      })
    }]
  }),
  selectors: ({
    selectors
  }) => ({
    deleteModalIndex: [() => [selectors.deleteModalIndexName, selectors.indexDetails], (indexName, indexDetails) => indexName === (indexDetails === null || indexDetails === void 0 ? void 0 : indexDetails.name) ? indexDetails : null],
    deleteModalIndexHasInProgressSyncs: [() => [selectors.deleteModalIndex], index => index ? index.has_in_progress_syncs : false],
    deleteModalIngestionMethod: [() => [selectors.indexDetails], index => index ? (0, _indices.getIngestionMethod)(index) : _types.IngestionMethod.API],
    hasNoIndices: [
    // We need this to show the landing page on the overview page if there are no indices
    // We can't rely just on there being no indices, because user might have entered a search query
    () => [selectors.data], data => {
      var _ref;
      return (_ref = (data === null || data === void 0 ? void 0 : data.isInitialRequest) && (data === null || data === void 0 ? void 0 : data.indices) && data.indices.length === 0) !== null && _ref !== void 0 ? _ref : false;
    }],
    indices: [() => [selectors.data], data => data !== null && data !== void 0 && data.indices ? data.indices.map(_indices.indexToViewIndex) : []],
    isDeleteLoading: [() => [selectors.deleteStatus], status => [_api.Status.LOADING].includes(status)],
    isFetchIndexDetailsLoading: [() => [selectors.indexDetailsStatus], status => [_api.Status.IDLE, _api.Status.LOADING].includes(status)],
    isLoading: [() => [selectors.status, selectors.isFirstRequest], (status, isFirstRequest) => [_api.Status.LOADING, _api.Status.IDLE].includes(status) && isFirstRequest],
    meta: [() => [selectors.data], data => {
      var _data$meta;
      return (_data$meta = data === null || data === void 0 ? void 0 : data.meta) !== null && _data$meta !== void 0 ? _data$meta : {
        page: {
          from: 0,
          size: 20,
          total: 0
        }
      };
    }]
  })
});
exports.IndicesLogic = IndicesLogic;