"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.columnsFromLocatorFactory = columnsFromLocatorFactory;
exports.getColumns = void 0;
var _server = require("@kbn/saved-search-plugin/server");
var _discoverUtils = require("@kbn/discover-utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function isStringArray(arr) {
  return Array.isArray(arr) && arr.every(p => typeof p === 'string');
}

/**
 * @internal
 */
const getColumns = async (services, index, savedSearch) => {
  var _savedSearch$columns;
  const [hideTimeColumn, useFieldsFromSource] = await Promise.all([services.uiSettings.get(_discoverUtils.DOC_HIDE_TIME_COLUMN_SETTING), services.uiSettings.get(_discoverUtils.SEARCH_FIELDS_FROM_SOURCE)]);

  // Add/adjust columns from the saved search attributes and UI Settings
  let columns;
  let columnsNext;
  let timeFieldName;
  // ignore '_source' column: it may be the only column when the user wishes to export all fields
  const columnsTemp = (_savedSearch$columns = savedSearch.columns) === null || _savedSearch$columns === void 0 ? void 0 : _savedSearch$columns.filter(col => col !== '_source');
  if (typeof columnsTemp !== 'undefined' && columnsTemp.length > 0 && isStringArray(columnsTemp)) {
    columns = columnsTemp;

    // conditionally add the time field column:
    if (index !== null && index !== void 0 && index.timeFieldName && !hideTimeColumn) {
      timeFieldName = index.timeFieldName;
    }
    if (timeFieldName && !columnsTemp.includes(timeFieldName)) {
      columns = [timeFieldName, ...columns];
    }

    /*
     * For querying performance, the searchSource object must have fields set.
     * Otherwise, the requests will ask for all fields, even if only a few are really needed.
     * Discover does not set fields, since having all fields is needed for the UI.
     */
    if (!useFieldsFromSource && columns.length) {
      columnsNext = columns;
    }
  }
  return {
    timeFieldName,
    columns: columnsNext
  };
};

/**
 * @internal
 */
exports.getColumns = getColumns;
function columnsFromLocatorFactory(services) {
  /**
   * Allows consumers to retrieve a set of selected fields from a search in DiscoverAppLocatorParams
   *
   * @public
   */
  const columnsFromLocator = async params => {
    if (!params.savedSearchId) {
      throw new Error(`Saved Search ID is required in DiscoverAppLocatorParams`);
    }
    const savedSearch = await (0, _server.getSavedSearch)(params.savedSearchId, services);
    const index = savedSearch.searchSource.getField('index');
    if (!index) {
      throw new Error(`Search Source is missing the "index" field`);
    }
    const {
      columns
    } = await getColumns(services, index, savedSearch);
    return columns;
  };
  return columnsFromLocator;
}