"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.loadSavedSearch = void 0;
var _lodash = require("lodash");
var _get_data_view_by_text_based_query_lang = require("../utils/get_data_view_by_text_based_query_lang");
var _is_text_based_query = require("../utils/is_text_based_query");
var _resolve_data_view = require("../utils/resolve_data_view");
var _cleanup_url_state = require("../utils/cleanup_url_state");
var _get_valid_filters = require("../../../utils/get_valid_filters");
var _add_log = require("../../../utils/add_log");
var _discover_app_state_container = require("./discover_app_state_container");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Loading persisted saved searches or existing ones and updating services accordingly
 * @param params
 * @param deps
 */
const loadSavedSearch = async (params, deps) => {
  (0, _add_log.addLog)('[discoverState] loadSavedSearch');
  const {
    savedSearchId
  } = params !== null && params !== void 0 ? params : {};
  const {
    appStateContainer,
    internalStateContainer,
    savedSearchContainer,
    services
  } = deps;
  const appStateExists = !appStateContainer.isEmptyURL();
  const appState = appStateExists ? appStateContainer.getState() : undefined;

  // Loading the saved search or creating a new one
  let nextSavedSearch = savedSearchId ? await savedSearchContainer.load(savedSearchId) : await savedSearchContainer.new(await getStateDataView(params, {
    services,
    appState,
    internalStateContainer
  }));

  // Cleaning up the previous state
  services.filterManager.setAppFilters([]);
  services.data.query.queryString.clearQuery();

  // reset appState in case a saved search with id is loaded and
  // the url is empty so the saved search is loaded in a clean
  // state else it might be updated by the previous app state
  if (!appStateExists) {
    appStateContainer.set({});
  }

  // Update saved search by a given app state (in URL)
  if (appState) {
    if (savedSearchId && appState.index) {
      var _nextSavedSearch$sear;
      // This is for the case appState is overwriting the loaded saved search data view
      const savedSearchDataViewId = (_nextSavedSearch$sear = nextSavedSearch.searchSource.getField('index')) === null || _nextSavedSearch$sear === void 0 ? void 0 : _nextSavedSearch$sear.id;
      const stateDataView = await getStateDataView(params, {
        services,
        appState,
        internalStateContainer,
        savedSearch: nextSavedSearch
      });
      const dataViewDifferentToAppState = stateDataView.id !== savedSearchDataViewId;
      if (stateDataView && (dataViewDifferentToAppState || !savedSearchDataViewId)) {
        nextSavedSearch.searchSource.setField('index', stateDataView);
      }
    }
    nextSavedSearch = savedSearchContainer.update({
      nextDataView: nextSavedSearch.searchSource.getField('index'),
      nextState: appState
    });
  }

  // Update app state container with the next state derived from the next saved search
  const nextAppState = (0, _discover_app_state_container.getInitialState)(undefined, nextSavedSearch, services);
  const mergedAppState = appState ? {
    ...nextAppState,
    ...(0, _cleanup_url_state.cleanupUrlState)({
      ...appState
    }, services.uiSettings)
  } : nextAppState;
  appStateContainer.resetToState(mergedAppState);

  // Update all other services and state containers by the next saved search
  updateBySavedSearch(nextSavedSearch, deps);
  return nextSavedSearch;
};

/**
 * Update services and state containers based on the given saved search
 * @param savedSearch
 * @param deps
 */
exports.loadSavedSearch = loadSavedSearch;
function updateBySavedSearch(savedSearch, deps) {
  const {
    dataStateContainer,
    internalStateContainer,
    services,
    setDataView
  } = deps;
  const savedSearchDataView = savedSearch.searchSource.getField('index');
  setDataView(savedSearchDataView);
  if (!savedSearchDataView.isPersisted()) {
    internalStateContainer.transitions.appendAdHocDataViews(savedSearchDataView);
  }

  // Finally notify dataStateContainer, data.query and filterManager about new derived state
  dataStateContainer.reset(savedSearch);
  // set data service filters
  const filters = savedSearch.searchSource.getField('filter');
  if (Array.isArray(filters) && filters.length) {
    services.data.query.filterManager.setAppFilters((0, _lodash.cloneDeep)(filters));
  }
  // some filters may not be valid for this context, so update
  // the filter manager with a modified list of valid filters
  const currentFilters = services.filterManager.getFilters();
  const validFilters = (0, _get_valid_filters.getValidFilters)(savedSearchDataView, currentFilters);
  if (!(0, _lodash.isEqual)(currentFilters, validFilters)) {
    services.filterManager.setFilters(validFilters);
  }
  // set data service query
  const query = savedSearch.searchSource.getField('query');
  if (query) {
    services.data.query.queryString.setQuery(query);
  }
}

// Get the data view to actually use. There are several conditions to consider which data view is used
// 1. If a data view is passed in, use that
// 2. If an appState with index set is passed in, use the data view from that
// 3. If a saved search is passed in, use the data view from that
// And provide a fallback data view if 2. or 3. don't exist, were deleted or invalid
const getStateDataView = async (params, {
  savedSearch,
  appState,
  services,
  internalStateContainer
}) => {
  const {
    dataView,
    dataViewSpec
  } = params !== null && params !== void 0 ? params : {};
  if (dataView) {
    return dataView;
  }
  const query = appState === null || appState === void 0 ? void 0 : appState.query;
  if ((0, _is_text_based_query.isTextBasedQuery)(query)) {
    return await (0, _get_data_view_by_text_based_query_lang.getDataViewByTextBasedQueryLang)(query, dataView, services);
  }
  const result = await (0, _resolve_data_view.loadAndResolveDataView)({
    id: appState === null || appState === void 0 ? void 0 : appState.index,
    dataViewSpec,
    savedSearch,
    isTextBasedQuery: (0, _is_text_based_query.isTextBasedQuery)(appState === null || appState === void 0 ? void 0 : appState.query)
  }, {
    services,
    internalStateContainer
  });
  return result.dataView;
};