"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildStateSubscribe = void 0;
var _lodash = require("lodash");
var _discover_app_state_container = require("../../services/discover_app_state_container");
var _add_log = require("../../../../utils/add_log");
var _is_text_based_query = require("../../utils/is_text_based_query");
var _types = require("../../../types");
var _resolve_data_view = require("../../utils/resolve_data_view");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Builds a subscribe function for the AppStateContainer, that is executed when the AppState changes in URL
 * or programmatically. It's main purpose is to detect which changes should trigger a refetch of the data.
 * @param stateContainer
 */
const buildStateSubscribe = ({
  appState,
  dataState,
  internalState,
  savedSearchState,
  services,
  setDataView
}) => async nextState => {
  const prevState = appState.getPrevious();
  const nextQuery = nextState.query;
  const savedSearch = savedSearchState.getState();
  const prevQuery = savedSearch.searchSource.getField('query');
  const queryChanged = !(0, _lodash.isEqual)(nextQuery, prevQuery) || !(0, _lodash.isEqual)(nextQuery, prevState.query);
  if ((0, _discover_app_state_container.isEqualState)(prevState, nextState) && !queryChanged) {
    (0, _add_log.addLog)('[appstate] subscribe update ignored due to no changes', {
      prevState,
      nextState
    });
    return;
  }
  (0, _add_log.addLog)('[appstate] subscribe triggered', nextState);
  const {
    hideChart,
    interval,
    breakdownField,
    sampleSize,
    sort,
    index
  } = prevState;
  const isTextBasedQueryLang = (0, _is_text_based_query.isTextBasedQuery)(nextQuery);
  if (isTextBasedQueryLang) {
    const isTextBasedQueryLangPrev = (0, _is_text_based_query.isTextBasedQuery)(prevQuery);
    if (!isTextBasedQueryLangPrev) {
      savedSearchState.update({
        nextState
      });
      dataState.reset(savedSearch);
    }
  }
  // Cast to boolean to avoid false positives when comparing
  // undefined and false, which would trigger a refetch
  const chartDisplayChanged = Boolean(nextState.hideChart) !== Boolean(hideChart);
  const chartIntervalChanged = nextState.interval !== interval && !isTextBasedQueryLang;
  const breakdownFieldChanged = nextState.breakdownField !== breakdownField;
  const sampleSizeChanged = nextState.sampleSize !== sampleSize;
  const docTableSortChanged = !(0, _lodash.isEqual)(nextState.sort, sort) && !isTextBasedQueryLang;
  const dataViewChanged = !(0, _lodash.isEqual)(nextState.index, index) && !isTextBasedQueryLang;
  let savedSearchDataView;
  // NOTE: this is also called when navigating from discover app to context app
  if (nextState.index && dataViewChanged) {
    const {
      dataView: nextDataView,
      fallback
    } = await (0, _resolve_data_view.loadAndResolveDataView)({
      id: nextState.index,
      savedSearch,
      isTextBasedQuery: (0, _is_text_based_query.isTextBasedQuery)(nextState === null || nextState === void 0 ? void 0 : nextState.query)
    }, {
      internalStateContainer: internalState,
      services
    });

    // If the requested data view is not found, don't try to load it,
    // and instead reset the app state to the fallback data view
    if (fallback) {
      appState.update({
        index: nextDataView.id
      }, true);
      return;
    }
    savedSearch.searchSource.setField('index', nextDataView);
    dataState.reset(savedSearch);
    setDataView(nextDataView);
    savedSearchDataView = nextDataView;
  }
  savedSearchState.update({
    nextDataView: savedSearchDataView,
    nextState
  });
  if (dataViewChanged && dataState.getInitialFetchStatus() === _types.FetchStatus.UNINITIALIZED) {
    // stop execution if given data view has changed, and it's not configured to initially start a search in Discover
    return;
  }
  if (chartDisplayChanged || chartIntervalChanged || breakdownFieldChanged || sampleSizeChanged || docTableSortChanged || dataViewChanged || queryChanged) {
    const logData = {
      chartDisplayChanged: logEntry(chartDisplayChanged, hideChart, nextState.hideChart),
      chartIntervalChanged: logEntry(chartIntervalChanged, interval, nextState.interval),
      breakdownFieldChanged: logEntry(breakdownFieldChanged, breakdownField, nextState.breakdownField),
      docTableSortChanged: logEntry(docTableSortChanged, sort, nextState.sort),
      dataViewChanged: logEntry(dataViewChanged, index, nextState.index),
      queryChanged: logEntry(queryChanged, prevQuery, nextQuery)
    };
    (0, _add_log.addLog)('[buildStateSubscribe] state changes triggers data fetching', JSON.stringify(logData, null, 2));
    dataState.fetch();
  }
};
exports.buildStateSubscribe = buildStateSubscribe;
const logEntry = (changed, prevState, nextState) => ({
  changed,
  prevState,
  nextState
});