"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.loadDashboardState = void 0;
exports.migrateLegacyQuery = migrateLegacyQuery;
var _uuid = require("uuid");
var _lodash = require("lodash");
var _public = require("@kbn/embeddable-plugin/public");
var _public2 = require("@kbn/kibana-utils-plugin/public");
var _public3 = require("@kbn/presentation-util-plugin/public");
var _common = require("@kbn/controls-plugin/common");
var _public4 = require("@kbn/data-plugin/public");
var _common2 = require("../../../../common");
var _migrate_dashboard_input = require("./migrate_dashboard_input");
var _dashboard_versioning = require("./dashboard_versioning");
var _dashboard_content_management_service = require("../dashboard_content_management_service");
var _dashboard_constants = require("../../../dashboard_constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function migrateLegacyQuery(query) {
  // Lucene was the only option before, so language-less queries are all lucene
  if (!(0, _lodash.has)(query, 'language')) {
    return {
      query,
      language: 'lucene'
    };
  }
  return query;
}
const loadDashboardState = async ({
  id,
  data,
  embeddable,
  contentManagement,
  savedObjectsTagging
}) => {
  var _ref, _savedObjectsTagging$, _savedObjectsTagging$2;
  const {
    search: dataSearchService,
    query: {
      queryString
    }
  } = data;
  const savedObjectId = id;
  const embeddableId = (0, _uuid.v4)();
  const newDashboardState = {
    ..._dashboard_constants.DEFAULT_DASHBOARD_INPUT,
    id: embeddableId
  };

  /**
   * This is a newly created dashboard, so there is no saved object state to load.
   */
  if (!savedObjectId) {
    return {
      dashboardInput: newDashboardState,
      dashboardFound: true,
      newDashboardCreated: true
    };
  }

  /**
   * Load the saved object from Content Management
   */
  let rawDashboardContent;
  let resolveMeta;
  const cachedDashboard = _dashboard_content_management_service.dashboardContentManagementCache.fetchDashboard(id);
  if (cachedDashboard) {
    /** If the dashboard exists in the cache, use the cached version to load the dashboard */
    ({
      item: rawDashboardContent,
      meta: resolveMeta
    } = cachedDashboard);
  } else {
    /** Otherwise, fetch and load the dashboard from the content management client, and add it to the cache */
    const result = await contentManagement.client.get({
      contentTypeId: _dashboard_constants.DASHBOARD_CONTENT_ID,
      id
    }).catch(e => {
      throw new _public2.SavedObjectNotFound(_dashboard_constants.DASHBOARD_CONTENT_ID, id);
    });
    ({
      item: rawDashboardContent,
      meta: resolveMeta
    } = result);
    const {
      outcome: loadOutcome
    } = resolveMeta;
    if (loadOutcome !== 'aliasMatch') {
      /**
       * Only add the dashboard to the cache if it does not require a redirect - otherwise, the meta
       * alias info gets cached and prevents the dashboard contents from being updated
       */
      _dashboard_content_management_service.dashboardContentManagementCache.addDashboard(result);
    }
  }
  if (!rawDashboardContent || !rawDashboardContent.version) {
    return {
      dashboardInput: newDashboardState,
      dashboardFound: false,
      dashboardId: savedObjectId
    };
  }

  /**
   * Inject saved object references back into the saved object attributes
   */
  const {
    references,
    attributes: rawAttributes,
    managed
  } = rawDashboardContent;
  const attributes = (() => {
    if (!references || references.length === 0) return rawAttributes;
    return (0, _common2.injectReferences)({
      references,
      attributes: rawAttributes
    }, {
      embeddablePersistableStateService: embeddable
    });
  })();

  /**
   * Create search source and pull filters and query from it.
   */
  const searchSourceJSON = attributes.kibanaSavedObjectMeta.searchSourceJSON;
  const searchSource = await (async () => {
    if (!searchSourceJSON) {
      return await dataSearchService.searchSource.create();
    }
    try {
      let searchSourceValues = (0, _public4.parseSearchSourceJSON)(searchSourceJSON);
      searchSourceValues = (0, _public4.injectSearchSourceReferences)(searchSourceValues, references);
      return await dataSearchService.searchSource.create(searchSourceValues);
    } catch (error) {
      return await dataSearchService.searchSource.create();
    }
  })();
  const filters = (0, _public3.cleanFiltersForSerialize)((_ref = searchSource === null || searchSource === void 0 ? void 0 : searchSource.getOwnField('filter')) !== null && _ref !== void 0 ? _ref : []);
  const query = migrateLegacyQuery((searchSource === null || searchSource === void 0 ? void 0 : searchSource.getOwnField('query')) || queryString.getDefaultQuery() // TODO SAVED DASHBOARDS determine if migrateLegacyQuery is still needed
  );

  const {
    refreshInterval,
    description,
    timeRestore,
    optionsJSON,
    panelsJSON,
    timeFrom,
    version,
    timeTo,
    title
  } = attributes;
  const timeRange = timeRestore && timeFrom && timeTo ? {
    from: timeFrom,
    to: timeTo
  } : undefined;

  /**
   * Parse panels and options from JSON
   */
  const options = optionsJSON ? JSON.parse(optionsJSON) : undefined;
  const panels = (0, _common2.convertSavedPanelsToPanelMap)(panelsJSON ? JSON.parse(panelsJSON) : []);
  const {
    dashboardInput,
    anyMigrationRun
  } = (0, _migrate_dashboard_input.migrateDashboardInput)({
    ..._dashboard_constants.DEFAULT_DASHBOARD_INPUT,
    ...options,
    id: embeddableId,
    refreshInterval,
    timeRestore,
    description,
    timeRange,
    filters,
    panels,
    query,
    title,
    viewMode: _public.ViewMode.VIEW,
    // dashboards loaded from saved object default to view mode. If it was edited recently, the view mode from session storage will override this.
    tags: (_savedObjectsTagging$ = (_savedObjectsTagging$2 = savedObjectsTagging.getTagIdsFromReferences) === null || _savedObjectsTagging$2 === void 0 ? void 0 : _savedObjectsTagging$2.call(savedObjectsTagging, references)) !== null && _savedObjectsTagging$ !== void 0 ? _savedObjectsTagging$ : [],
    controlGroupInput: attributes.controlGroupInput && (0, _common.rawControlGroupAttributesToControlGroupInput)(attributes.controlGroupInput),
    version: (0, _dashboard_versioning.convertNumberToDashboardVersion)(version)
  }, embeddable);
  return {
    managed,
    resolveMeta,
    dashboardInput,
    anyMigrationRun,
    dashboardFound: true,
    dashboardId: savedObjectId
  };
};
exports.loadDashboardState = loadDashboardState;