"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getShouldRefresh = getShouldRefresh;
exports.getUnsavedChanges = getUnsavedChanges;
exports.reducersToIgnore = exports.keysNotConsideredUnsavedChanges = void 0;
exports.startDiffingDashboardState = startDiffingDashboardState;
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _dashboard_diffing_functions = require("./dashboard_diffing_functions");
var _plugin_services = require("../../../services/plugin_services");
var _dashboard_constants = require("../../../dashboard_constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * An array of reducers which cannot cause unsaved changes. Unsaved changes only compares the explicit input
 * and the last saved input, so we can safely ignore any output reducers, and most componentState reducers.
 * This is only for performance reasons, because the diffing function itself can be quite heavy.
 */
const reducersToIgnore = ['setTimeslice', 'setFullScreenMode', 'setExpandedPanelId', 'setHasUnsavedChanges'];

/**
 * Some keys will often have deviated from their last saved state, but should not persist over reloads
 */
exports.reducersToIgnore = reducersToIgnore;
const keysToOmitFromSessionStorage = ['lastReloadRequestTime', 'executionContext', 'timeslice', 'id', 'timeRange',
// Current behaviour expects time range not to be backed up. Revisit this?
'refreshInterval'];

/**
 * Some keys will often have deviated from their last saved state, but should be
 * ignored when calculating whether or not this dashboard has unsaved changes.
 */
const keysNotConsideredUnsavedChanges = ['lastReloadRequestTime', 'executionContext', 'timeslice', 'viewMode', 'id'];

/**
 * input keys that will cause a new session to be created.
 */
exports.keysNotConsideredUnsavedChanges = keysNotConsideredUnsavedChanges;
const sessionChangeKeys = ['query', 'filters', 'timeRange', 'timeslice', 'timeRestore', 'lastReloadRequestTime',
// also refetch when chart settings change
'syncColors', 'syncCursor', 'syncTooltips'];

/**
 * Does an initial diff between @param initialInput and @param initialLastSavedInput, and created a middleware
 * which listens to the redux store and checks for & publishes the unsaved changes on dispatches.
 */
function startDiffingDashboardState(creationOptions) {
  const checkForUnsavedChangesSubject$ = new _rxjs.Subject();
  this.diffingSubscription.add(checkForUnsavedChangesSubject$.pipe((0, _rxjs.startWith)(null), (0, _rxjs.debounceTime)(_dashboard_constants.CHANGE_CHECK_DEBOUNCE), (0, _rxjs.switchMap)(() => {
    return new _rxjs.Observable(observer => {
      const {
        explicitInput: currentInput,
        componentState: {
          lastSavedInput
        }
      } = this.getState();
      getUnsavedChanges.bind(this)(lastSavedInput, currentInput).then(unsavedChanges => {
        if (observer.closed) return;
        updateUnsavedChangesState.bind(this)(unsavedChanges);
        if (creationOptions !== null && creationOptions !== void 0 && creationOptions.useSessionStorageIntegration) {
          backupUnsavedChanges.bind(this)(unsavedChanges);
        }
      });
    });
  })).subscribe());
  const diffingMiddleware = store => next => action => {
    var _action$type$split;
    const dispatchedActionName = (_action$type$split = action.type.split('/')) === null || _action$type$split === void 0 ? void 0 : _action$type$split[1];
    if (dispatchedActionName && dispatchedActionName !== 'updateEmbeddableReduxOutput' &&
    // ignore any generic output updates.
    !reducersToIgnore.includes(dispatchedActionName)) {
      checkForUnsavedChangesSubject$.next(null);
    }
    next(action);
  };
  return diffingMiddleware;
}

/**
 * Does a shallow diff between @param lastInput and @param input and
 * @returns an object out of the keys which are different.
 */
async function getUnsavedChanges(lastInput, input) {
  const allKeys = [...new Set([...Object.keys(lastInput), ...Object.keys(input)])];
  const keyComparePromises = allKeys.map(key => new Promise(resolve => {
    if (input[key] === undefined && lastInput[key] === undefined) {
      resolve({
        key,
        isEqual: true
      });
    }
    (0, _dashboard_diffing_functions.isKeyEqualAsync)(key, {
      container: this,
      currentValue: input[key],
      currentInput: input,
      lastValue: lastInput[key],
      lastInput
    }, _dashboard_diffing_functions.unsavedChangesDiffingFunctions).then(isEqual => resolve({
      key,
      isEqual
    }));
  }));
  const inputChanges = (await Promise.allSettled(keyComparePromises)).reduce((changes, current) => {
    if (current.status === 'fulfilled') {
      const {
        key,
        isEqual
      } = current.value;
      if (!isEqual) changes[key] = input[key];
    }
    return changes;
  }, {});
  return inputChanges;
}
function getShouldRefresh(lastInput, input) {
  for (const key of sessionChangeKeys) {
    if (input[key] === undefined && lastInput[key] === undefined) {
      continue;
    }
    if (!(0, _dashboard_diffing_functions.isKeyEqual)(key, {
      container: this,
      currentValue: input[key],
      currentInput: input,
      lastValue: lastInput[key],
      lastInput
    }, _dashboard_diffing_functions.shouldRefreshDiffingFunctions)) {
      return true;
    }
  }
  return false;
}
function updateUnsavedChangesState(unsavedChanges) {
  // dispatch has unsaved changes state
  const hasChanges = Object.keys((0, _lodash.omit)(unsavedChanges, keysNotConsideredUnsavedChanges)).length > 0;
  if (this.getState().componentState.hasUnsavedChanges !== hasChanges) {
    this.dispatch.setHasUnsavedChanges(hasChanges);
  }
}
function backupUnsavedChanges(unsavedChanges) {
  const {
    dashboardBackup
  } = _plugin_services.pluginServices.getServices();
  dashboardBackup.setState(this.getDashboardSavedObjectId(), (0, _lodash.omit)(unsavedChanges, keysToOmitFromSessionStorage));
}