"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getVirtualVersionMap = exports.getModelVersionMapForTypes = exports.getLatestModelVersion = exports.getLatestMigrationVersion = exports.getCurrentVirtualVersion = void 0;
var _semver = _interopRequireDefault(require("semver"));
var _conversion = require("./conversion");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Represents the virtual version of a given SO type.
 * The virtual version is a compatibility format between the old
 * migration system's versioning, based on the stack version, and the new model versioning.
 *
 * A virtual version is a plain semver version. Depending on its major version value, the
 * underlying version can be the following:
 * - Major < 10: Old migrations system (stack versions), using the equivalent value (e.g `8.7.0` => migration version `8.7.0`)
 * - Major == 10: Model versions, using the `10.{modelVersion}.0` format (e.g `10.3.0` => model version 3)
 */

/**
 * A map of SO type name to Model Version.
 */

/**
 * A map of SO type name to {@link VirtualVersion}.
 */

/**
 * Returns the latest registered model version number for the given type.
 */
const getLatestModelVersion = type => {
  var _type$modelVersions;
  const versionMap = typeof type.modelVersions === 'function' ? type.modelVersions() : (_type$modelVersions = type.modelVersions) !== null && _type$modelVersions !== void 0 ? _type$modelVersions : {};
  return Object.keys(versionMap).reduce((memo, current) => {
    return Math.max(memo, (0, _conversion.assertValidModelVersion)(current));
  }, 0);
};
exports.getLatestModelVersion = getLatestModelVersion;
const getLatestMigrationVersion = type => {
  var _type$migrations;
  const migrationMap = typeof type.migrations === 'function' ? type.migrations() : (_type$migrations = type.migrations) !== null && _type$migrations !== void 0 ? _type$migrations : {};
  return Object.keys(migrationMap).reduce((memo, current) => {
    return _semver.default.gt(memo, current) ? memo : current;
  }, '0.0.0');
};

/**
 * Build a version map for the given types.
 */
exports.getLatestMigrationVersion = getLatestMigrationVersion;
const getModelVersionMapForTypes = types => {
  return types.reduce((versionMap, type) => {
    versionMap[type.name] = getLatestModelVersion(type);
    return versionMap;
  }, {});
};

/**
 * Returns the current virtual version for the given type.
 * It will either be the latest model version if the type
 * already switched to using them (switchToModelVersionAt is set),
 * or the latest migration version for the type otherwise.
 */
exports.getModelVersionMapForTypes = getModelVersionMapForTypes;
const getCurrentVirtualVersion = type => {
  if (type.switchToModelVersionAt) {
    const modelVersion = getLatestModelVersion(type);
    return (0, _conversion.modelVersionToVirtualVersion)(modelVersion);
  } else {
    return getLatestMigrationVersion(type);
  }
};

/**
 * Returns a map of virtual model version for the given types.
 * See {@link getCurrentVirtualVersion}
 */
exports.getCurrentVirtualVersion = getCurrentVirtualVersion;
const getVirtualVersionMap = types => {
  return types.reduce((versionMap, type) => {
    versionMap[type.name] = getCurrentVirtualVersion(type);
    return versionMap;
  }, {});
};
exports.getVirtualVersionMap = getVirtualVersionMap;