"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getVulnMgmtAccountsStatsQuery = exports.getPostureAccountsStatsQuery = exports.getIndexAccountStats = exports.getCloudAccountsStats = exports.getAllCloudAccountsStats = void 0;
var _get_package_policy_id_mapping = require("../../../../common/runtime_mappings/get_package_policy_id_mapping");
var _get_identifier_runtime_mapping = require("../../../../common/runtime_mappings/get_identifier_runtime_mapping");
var _helpers = require("../../../../common/utils/helpers");
var _constants = require("../../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getPostureAccountsStatsQuery = index => ({
  index,
  runtime_mappings: {
    ...(0, _get_identifier_runtime_mapping.getIdentifierRuntimeMapping)(),
    ...(0, _get_package_policy_id_mapping.getPackagePolicyIdRuntimeMapping)()
  },
  query: {
    match_all: {}
  },
  aggs: {
    accounts: {
      terms: {
        field: 'asset_identifier',
        order: {
          _count: 'desc'
        },
        size: 100
      },
      aggs: {
        cloud_provider: {
          top_metrics: {
            metrics: {
              field: 'cloud.provider'
            },
            size: 1,
            sort: {
              '@timestamp': 'desc'
            }
          }
        },
        latest_doc_updated_timestamp: {
          top_metrics: {
            metrics: {
              field: '@timestamp'
            },
            size: 1,
            sort: {
              '@timestamp': 'desc'
            }
          }
        },
        benchmark_id: {
          top_metrics: {
            metrics: {
              field: 'rule.benchmark.id'
            },
            size: 1,
            sort: {
              '@timestamp': 'desc'
            }
          }
        },
        benchmark_version: {
          top_metrics: {
            metrics: {
              field: 'rule.benchmark.version'
            },
            size: 1,
            sort: {
              '@timestamp': 'desc'
            }
          }
        },
        benchmark_name: {
          top_metrics: {
            metrics: {
              field: 'rule.benchmark.name'
            },
            size: 1,
            sort: {
              '@timestamp': 'desc'
            }
          }
        },
        passed_findings_count: {
          filter: {
            bool: {
              filter: [{
                bool: {
                  should: [{
                    term: {
                      'result.evaluation': 'passed'
                    }
                  }],
                  minimum_should_match: 1
                }
              }]
            }
          }
        },
        failed_findings_count: {
          filter: {
            bool: {
              filter: [{
                bool: {
                  should: [{
                    term: {
                      'result.evaluation': 'failed'
                    }
                  }],
                  minimum_should_match: 1
                }
              }]
            }
          }
        },
        // KSPM QUERY FIELDS
        kubernetes_version: {
          top_metrics: {
            metrics: {
              field: 'cloudbeat.kubernetes.version'
            },
            size: 1,
            sort: {
              '@timestamp': 'desc'
            }
          }
        },
        resources: {
          filter: {
            bool: {
              filter: [{
                bool: {
                  should: [{
                    term: {
                      'resource.sub_type': 'Pod'
                    }
                  }],
                  minimum_should_match: 1
                }
              }]
            }
          },
          aggs: {
            pods_count: {
              cardinality: {
                field: 'resource.id'
              }
            }
          }
        },
        nodes_count: {
          cardinality: {
            field: 'host.name'
          }
        },
        agents_count: {
          cardinality: {
            field: 'agent.id'
          }
        },
        package_policy_id: {
          terms: {
            field: 'package_policy_identifier',
            order: {
              _count: 'desc'
            },
            size: 100
          }
        }
      }
    }
  },
  size: 0,
  _source: false
});
exports.getPostureAccountsStatsQuery = getPostureAccountsStatsQuery;
const getVulnMgmtAccountsStatsQuery = index => ({
  index,
  runtime_mappings: (0, _get_package_policy_id_mapping.getPackagePolicyIdRuntimeMapping)(),
  query: {
    match_all: {}
  },
  aggs: {
    accounts: {
      terms: {
        field: 'cloud.account.id',
        order: {
          _count: 'desc'
        },
        size: 100
      },
      aggs: {
        latest_doc_updated_timestamp: {
          top_metrics: {
            metrics: {
              field: '@timestamp'
            },
            size: 1,
            sort: {
              '@timestamp': 'desc'
            }
          }
        },
        cloud_provider: {
          top_metrics: {
            metrics: {
              field: 'cloud.provider'
            },
            size: 1,
            sort: {
              '@timestamp': 'desc'
            }
          }
        },
        package_policy_id: {
          terms: {
            field: 'package_policy_identifier',
            order: {
              _count: 'desc'
            },
            size: 100
          }
        }
      }
    }
  },
  size: 0,
  _source: false
});
exports.getVulnMgmtAccountsStatsQuery = getVulnMgmtAccountsStatsQuery;
const cloudBaseStats = account => {
  var _account$package_poli, _account$package_poli2, _account$package_poli3;
  return {
    account_id: account.key,
    latest_doc_count: account.doc_count,
    latest_doc_updated_timestamp: account.latest_doc_updated_timestamp.top[0].metrics['@timestamp'],
    cloud_provider: account.cloud_provider.top[0].metrics['cloud.provider'],
    package_policy_id: (_account$package_poli = (_account$package_poli2 = account.package_policy_id) === null || _account$package_poli2 === void 0 ? void 0 : (_account$package_poli3 = _account$package_poli2.buckets[0]) === null || _account$package_poli3 === void 0 ? void 0 : _account$package_poli3.key) !== null && _account$package_poli !== void 0 ? _account$package_poli : null
  };
};
const getPostureManagementStats = account => ({
  posture_management_stats: {
    posture_score: (0, _helpers.calculatePostureScore)(account.passed_findings_count.doc_count, account.failed_findings_count.doc_count),
    passed_findings_count: account.passed_findings_count.doc_count,
    failed_findings_count: account.failed_findings_count.doc_count,
    benchmark_name: account.benchmark_name.top[0].metrics['rule.benchmark.name'],
    benchmark_version: account.benchmark_version.top[0].metrics['rule.benchmark.version']
  }
});
const getKspmStats = account => ({
  kspm_stats: {
    kubernetes_version: account.kubernetes_version.top[0].metrics['cloudbeat.kubernetes.version'],
    agents_count: account.agents_count.value,
    nodes_count: account.nodes_count.value,
    pods_count: account.resources.pods_count.value
  }
});
const kspmCloudProviders = {
  cis_eks: 'aws',
  cis_gke: 'gcp',
  cis_k8s: null,
  cis_ake: 'azure'
};
const cspmBenchmarkIds = ['cis_aws', 'cis_azure', 'cis_gcp'];
const kspmBenchmarkIds = ['cis_eks', 'cis_ake', 'cis_gke', 'cis_k8s'];
const getCloudProvider = ruleBenchmarkId => {
  return kspmCloudProviders[ruleBenchmarkId];
};
const getPostureType = ruleBenchmarkId => {
  if (cspmBenchmarkIds.includes(ruleBenchmarkId)) {
    return _constants.CSPM_POLICY_TEMPLATE;
  } else if (kspmBenchmarkIds.includes(ruleBenchmarkId)) {
    return _constants.KSPM_POLICY_TEMPLATE;
  }
  return undefined;
};
const getCloudAccountsStats = (aggregatedResourcesStats, logger) => {
  const accounts = aggregatedResourcesStats.accounts.buckets;
  const cloudAccountsStats = accounts.map(account => {
    var _account$benchmark_id, _account$benchmark_id2, _account$benchmark_id3;
    const cloudAccount = cloudBaseStats(account);
    const postureType = getPostureType((_account$benchmark_id = account.benchmark_id) === null || _account$benchmark_id === void 0 ? void 0 : (_account$benchmark_id2 = _account$benchmark_id.top) === null || _account$benchmark_id2 === void 0 ? void 0 : (_account$benchmark_id3 = _account$benchmark_id2[0]) === null || _account$benchmark_id3 === void 0 ? void 0 : _account$benchmark_id3.metrics['rule.benchmark.id']);
    if (!postureType) {
      return {
        ...cloudAccount,
        product: _constants.VULN_MGMT_POLICY_TEMPLATE
      };
    }
    if (postureType === _constants.CSPM_POLICY_TEMPLATE) {
      return {
        ...cloudAccount,
        product: _constants.CSPM_POLICY_TEMPLATE,
        ...getPostureManagementStats(account)
      };
    }
    return {
      ...cloudAccount,
      product: _constants.KSPM_POLICY_TEMPLATE,
      ...getPostureManagementStats(account),
      ...getKspmStats(account),
      cloud_provider: getCloudProvider(account.benchmark_id.top[0].metrics['rule.benchmark.id'])
    };
  });
  logger.info('Cloud Account Stats telemetry: accounts stats was sent');
  return cloudAccountsStats;
};
exports.getCloudAccountsStats = getCloudAccountsStats;
const getIndexAccountStats = async (esClient, logger, index, getAccountQuery) => {
  const accountsStatsResponse = await esClient.search(getAccountQuery(index));
  return accountsStatsResponse.aggregations ? getCloudAccountsStats(accountsStatsResponse.aggregations, logger) : [];
};
exports.getIndexAccountStats = getIndexAccountStats;
const getAllCloudAccountsStats = async (esClient, logger) => {
  try {
    const indices = [_constants.LATEST_FINDINGS_INDEX_DEFAULT_NS, _constants.LATEST_VULNERABILITIES_INDEX_DEFAULT_NS];
    const [findingIndex, vulnerabilitiesIndex] = await Promise.all(indices.map(async index => ({
      exists: await esClient.indices.exists({
        index
      }),
      name: index
    })));
    let postureIndexAccountStats = [];
    let vulnerabilityIndexAccountStats = [];
    if (!findingIndex.exists && !vulnerabilitiesIndex.exists) return [];
    if (findingIndex.exists) {
      postureIndexAccountStats = await getIndexAccountStats(esClient, logger, findingIndex.name, getPostureAccountsStatsQuery);
    }
    if (vulnerabilitiesIndex.exists) {
      vulnerabilityIndexAccountStats = await getIndexAccountStats(esClient, logger, vulnerabilitiesIndex.name, getVulnMgmtAccountsStatsQuery);
    }
    return [...postureIndexAccountStats, ...vulnerabilityIndexAccountStats];
  } catch (e) {
    logger.error(`Failed to get cloud account stats v2 ${e}`);
    logger.error(`Failed to get cloud account stats v2 ${e.stack}`);
    return [];
  }
};
exports.getAllCloudAccountsStats = getAllCloudAccountsStats;