"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createDetectionRuleFromVulnerabilityFinding = void 0;
var _i18n = require("@kbn/i18n");
var _constants = require("../../../../common/constants");
var _create_detection_rule = require("../../../common/api/create_detection_rule");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_RULE_RISK_SCORE = 0;
const DEFAULT_RULE_SEVERITY = 'low';
const DEFAULT_RULE_ENABLED = true;
const DEFAULT_RULE_AUTHOR = 'Elastic';
const DEFAULT_RULE_LICENSE = 'Elastic License v2';
const DEFAULT_MAX_ALERTS_PER_RULE = 100;
const ALERT_SUPPRESSION_FIELD = 'resource.id';
const ALERT_TIMESTAMP_FIELD = 'event.ingested';
const ALERT_SEVERITY_MAP_FIELD = 'vulnerability.severity';
const DEFAULT_INVESTIGATION_FIELDS = {
  field_names: ['resource.name', 'resource.id']
};
var RuleSeverityMapping;
(function (RuleSeverityMapping) {
  RuleSeverityMapping["Low"] = "low";
  RuleSeverityMapping["Medium"] = "medium";
  RuleSeverityMapping["High"] = "high";
  RuleSeverityMapping["Critical"] = "critical";
})(RuleSeverityMapping || (RuleSeverityMapping = {}));
var AlertSuppressionMissingFieldsStrategy;
(function (AlertSuppressionMissingFieldsStrategy) {
  AlertSuppressionMissingFieldsStrategy["DoNotSuppress"] = "doNotSuppress";
  AlertSuppressionMissingFieldsStrategy["Suppress"] = "suppress";
})(AlertSuppressionMissingFieldsStrategy || (AlertSuppressionMissingFieldsStrategy = {}));
const CSP_RULE_TAG = 'Cloud Security';
const CNVM_RULE_TAG = 'CNVM';
const CNVM_RULE_TAG_DATA_SOURCE = 'Data Source: Cloud Native Vulnerability Management';
const CNVM_RULE_TAG_USE_CASE = 'Use Case: Vulnerability';
const CNVM_RULE_TAG_OS = 'OS: Linux';
const STATIC_RULE_TAGS = [CSP_RULE_TAG, CNVM_RULE_TAG, CNVM_RULE_TAG_DATA_SOURCE, CNVM_RULE_TAG_USE_CASE, CNVM_RULE_TAG_OS];
const generateVulnerabilitiesTags = vulnerability => {
  return [...STATIC_RULE_TAGS, vulnerability.id];
};
const getVulnerabilityRuleName = vulnerability => {
  return _i18n.i18n.translate('xpack.csp.vulnerabilities.detectionRuleNamePrefix', {
    defaultMessage: 'Vulnerability: {vulnerabilityId}',
    values: {
      vulnerabilityId: vulnerability.id
    }
  });
};
const generateVulnerabilitiesRuleQuery = vulnerability => {
  const currentTimestamp = new Date().toISOString();
  return `vulnerability.id: "${vulnerability.id}" AND event.ingested >= "${currentTimestamp}"`;
};

/*
 * Creates a detection rule from a Vulnerability
 */
const createDetectionRuleFromVulnerabilityFinding = async (http, vulnerability) => {
  return await (0, _create_detection_rule.createDetectionRule)({
    http,
    rule: {
      type: 'query',
      language: 'kuery',
      license: DEFAULT_RULE_LICENSE,
      author: [DEFAULT_RULE_AUTHOR],
      filters: [],
      false_positives: [],
      risk_score: DEFAULT_RULE_RISK_SCORE,
      risk_score_mapping: [],
      severity: DEFAULT_RULE_SEVERITY,
      severity_mapping: [{
        field: ALERT_SEVERITY_MAP_FIELD,
        value: _constants.VULNERABILITIES_SEVERITY.LOW,
        operator: 'equals',
        severity: RuleSeverityMapping.Low
      }, {
        field: ALERT_SEVERITY_MAP_FIELD,
        value: _constants.VULNERABILITIES_SEVERITY.MEDIUM,
        operator: 'equals',
        severity: RuleSeverityMapping.Medium
      }, {
        field: ALERT_SEVERITY_MAP_FIELD,
        value: _constants.VULNERABILITIES_SEVERITY.HIGH,
        operator: 'equals',
        severity: RuleSeverityMapping.High
      }, {
        field: ALERT_SEVERITY_MAP_FIELD,
        value: _constants.VULNERABILITIES_SEVERITY.CRITICAL,
        operator: 'equals',
        severity: RuleSeverityMapping.Critical
      }, {
        field: ALERT_SEVERITY_MAP_FIELD,
        value: _constants.VULNERABILITIES_SEVERITY.UNKNOWN,
        operator: 'equals',
        severity: RuleSeverityMapping.Low
      }],
      threat: [],
      interval: '1h',
      from: `now-${_constants.LATEST_VULNERABILITIES_RETENTION_POLICY}`,
      to: 'now',
      max_signals: DEFAULT_MAX_ALERTS_PER_RULE,
      timestamp_override: ALERT_TIMESTAMP_FIELD,
      timestamp_override_fallback_disabled: false,
      actions: [],
      enabled: DEFAULT_RULE_ENABLED,
      alert_suppression: {
        group_by: [ALERT_SUPPRESSION_FIELD],
        missing_fields_strategy: AlertSuppressionMissingFieldsStrategy.Suppress
      },
      index: [_constants.VULNERABILITIES_INDEX_PATTERN],
      query: generateVulnerabilitiesRuleQuery(vulnerability),
      references: vulnerability.reference ? [vulnerability.reference] : [],
      name: getVulnerabilityRuleName(vulnerability),
      description: vulnerability.description,
      tags: generateVulnerabilitiesTags(vulnerability),
      investigation_fields: DEFAULT_INVESTIGATION_FIELDS
    }
  });
};
exports.createDetectionRuleFromVulnerabilityFinding = createDetectionRuleFromVulnerabilityFinding;