"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getFindingsDetectionRuleSearchTags = exports.createDetectionRuleFromFinding = void 0;
var _constants = require("../../../../common/constants");
var _create_detection_rule = require("../../../common/api/create_detection_rule");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEFAULT_RULE_RISK_SCORE = 0;
const DEFAULT_RULE_SEVERITY = 'low';
const DEFAULT_RULE_ENABLED = true;
const DEFAULT_RULE_AUTHOR = 'Elastic';
const DEFAULT_RULE_LICENSE = 'Elastic License v2';
const DEFAULT_MAX_ALERTS_PER_RULE = 100;
const ALERT_SUPPRESSION_FIELD = 'resource.id';
const ALERT_TIMESTAMP_FIELD = 'event.ingested';
const DEFAULT_INVESTIGATION_FIELDS = {
  field_names: ['resource.name', 'resource.id', 'resource.type', 'resource.sub_type']
};
var AlertSuppressionMissingFieldsStrategy;
(function (AlertSuppressionMissingFieldsStrategy) {
  AlertSuppressionMissingFieldsStrategy["DoNotSuppress"] = "doNotSuppress";
  AlertSuppressionMissingFieldsStrategy["Suppress"] = "suppress";
})(AlertSuppressionMissingFieldsStrategy || (AlertSuppressionMissingFieldsStrategy = {}));
const convertReferencesLinksToArray = input => {
  if (!input) {
    return [];
  }
  // Match all URLs in the input string using a regular expression
  const matches = input.match(/(https?:\/\/\S+)/g);
  if (!matches) {
    return [];
  }

  // Remove the numbers and new lines
  return matches.map(link => link.replace(/^\d+\. /, '').replace(/\n/g, ''));
};
const CSP_RULE_TAG = 'Cloud Security';
const CSP_RULE_TAG_USE_CASE = 'Use Case: Configuration Audit';
const CSP_RULE_TAG_DATA_SOURCE_PREFIX = 'Data Source: ';
const STATIC_RULE_TAGS = [CSP_RULE_TAG, CSP_RULE_TAG_USE_CASE];

/*
 * Returns an array of CspFinding tags that can be used to search and filter a detection rule
 */
const getFindingsDetectionRuleSearchTags = ({
  rule
}) => {
  // ex: cis_gcp to ['CIS', 'GCP']
  const benchmarkIdTags = rule.benchmark.id.split('_').map(tag => tag.toUpperCase());
  // ex: 'CIS GCP 1.1'
  const benchmarkRuleNumberTag = `${rule.benchmark.id.replace('_', ' ').toUpperCase()} ${rule.benchmark.rule_number}`;
  return benchmarkIdTags.concat([benchmarkRuleNumberTag]);
};
exports.getFindingsDetectionRuleSearchTags = getFindingsDetectionRuleSearchTags;
const generateFindingsTags = finding => {
  return [STATIC_RULE_TAGS].concat(getFindingsDetectionRuleSearchTags(finding)).concat(finding.rule.benchmark.posture_type ? [finding.rule.benchmark.posture_type.toUpperCase(), `${CSP_RULE_TAG_DATA_SOURCE_PREFIX}${finding.rule.benchmark.posture_type.toUpperCase()}`] : []).concat(finding.rule.benchmark.posture_type === 'cspm' ? ['Domain: Cloud'] : ['Domain: Container']).flat();
};
const generateFindingsRuleQuery = finding => {
  const currentTimestamp = new Date().toISOString();
  return `rule.benchmark.rule_number: "${finding.rule.benchmark.rule_number}"
    AND rule.benchmark.id: "${finding.rule.benchmark.id}"
    AND result.evaluation: "failed"
    AND event.ingested >= "${currentTimestamp}"`;
};

/*
 * Creates a detection rule from a CspFinding
 */
const createDetectionRuleFromFinding = async (http, finding) => {
  return await (0, _create_detection_rule.createDetectionRule)({
    http,
    rule: {
      type: 'query',
      language: 'kuery',
      license: DEFAULT_RULE_LICENSE,
      author: [DEFAULT_RULE_AUTHOR],
      filters: [],
      false_positives: [],
      risk_score: DEFAULT_RULE_RISK_SCORE,
      risk_score_mapping: [],
      severity: DEFAULT_RULE_SEVERITY,
      severity_mapping: [],
      threat: [],
      interval: '1h',
      from: `now-${_constants.LATEST_FINDINGS_RETENTION_POLICY}`,
      to: 'now',
      max_signals: DEFAULT_MAX_ALERTS_PER_RULE,
      timestamp_override: ALERT_TIMESTAMP_FIELD,
      timestamp_override_fallback_disabled: false,
      actions: [],
      enabled: DEFAULT_RULE_ENABLED,
      alert_suppression: {
        group_by: [ALERT_SUPPRESSION_FIELD],
        missing_fields_strategy: AlertSuppressionMissingFieldsStrategy.Suppress
      },
      index: [_constants.FINDINGS_INDEX_PATTERN],
      query: generateFindingsRuleQuery(finding),
      references: convertReferencesLinksToArray(finding.rule.references),
      name: finding.rule.name,
      description: finding.rule.rationale,
      tags: generateFindingsTags(finding),
      investigation_fields: DEFAULT_INVESTIGATION_FIELDS,
      note: finding.rule.remediation
    }
  });
};
exports.createDetectionRuleFromFinding = createDetectionRuleFromFinding;