"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.maybeCreateTemplate = maybeCreateTemplate;
exports.upsertTemplate = upsertTemplate;
var _constants = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function templateExists(template, existing) {
  if (existing === null) {
    return false;
  }
  if (existing.index_templates.length === 0) {
    return false;
  }
  const checkPatterns = Array.isArray(template.index_patterns) ? template.index_patterns : [template.index_patterns];
  return existing.index_templates.some(t => {
    const {
      priority: existingPriority = 0
    } = t.index_template;
    const {
      priority: incomingPriority = 0
    } = template;
    if (existingPriority !== incomingPriority) {
      return false;
    }
    const existingPatterns = Array.isArray(t.index_template.index_patterns) ? t.index_template.index_patterns : [t.index_template.index_patterns];
    if (checkPatterns.every(p => p && existingPatterns.includes(p))) {
      return true;
    }
    return false;
  });
}

// interface IndexPatternJson {
//   index_patterns: string[];
//   name: string;
//   template: {
//     mappings: Record<string, any>;
//     settings: Record<string, any>;
//   };
// }
async function maybeCreateTemplate({
  esClient,
  template,
  logger
}) {
  const pattern = _constants.ASSETS_INDEX_PREFIX + '*';
  template.index_patterns = [pattern];
  let existing = null;
  try {
    existing = await esClient.indices.getIndexTemplate({
      name: template.name
    });
  } catch (error) {
    if ((error === null || error === void 0 ? void 0 : error.name) !== 'ResponseError' || (error === null || error === void 0 ? void 0 : error.statusCode) !== 404) {
      logger.warn(`Asset manager index template lookup failed: ${error.message}`);
    }
  }
  try {
    if (!templateExists(template, existing)) {
      await esClient.indices.putIndexTemplate(template);
    }
  } catch (error) {
    logger.error(`Asset manager index template creation failed: ${error.message}`);
    return;
  }
  logger.info(`Asset manager index template is up to date (use debug logging to see what was installed)`);
  logger.debug(`Asset manager index template: ${JSON.stringify(template)}`);
}
async function upsertTemplate({
  esClient,
  template,
  logger
}) {
  const pattern = _constants.ASSETS_INDEX_PREFIX + '*';
  template.index_patterns = [pattern];
  try {
    await esClient.indices.putIndexTemplate(template);
  } catch (error) {
    logger.error(`Error updating asset manager index template: ${error.message}`);
    return;
  }
  logger.info(`Asset manager index template is up to date (use debug logging to see what was installed)`);
  logger.debug(`Asset manager index template: ${JSON.stringify(template)}`);
}