"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RulesSettingsFlappingClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _server = require("@kbn/core/server");
var _common = require("../../../common");
var _retry_if_conflicts = require("../../lib/retry_if_conflicts");
var _schemas = require("../schemas");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const verifyFlappingSettings = flappingSettings => {
  const {
    lookBackWindow,
    statusChangeThreshold
  } = flappingSettings;
  if (lookBackWindow < _common.MIN_LOOK_BACK_WINDOW || lookBackWindow > _common.MAX_LOOK_BACK_WINDOW) {
    throw _boom.default.badRequest(`Invalid lookBackWindow value, must be between ${_common.MIN_LOOK_BACK_WINDOW} and ${_common.MAX_LOOK_BACK_WINDOW}, but got: ${lookBackWindow}.`);
  }
  if (statusChangeThreshold < _common.MIN_STATUS_CHANGE_THRESHOLD || statusChangeThreshold > _common.MAX_STATUS_CHANGE_THRESHOLD) {
    throw _boom.default.badRequest(`Invalid statusChangeThreshold value, must be between ${_common.MIN_STATUS_CHANGE_THRESHOLD} and ${_common.MAX_STATUS_CHANGE_THRESHOLD}, but got: ${statusChangeThreshold}.`);
  }
  if (lookBackWindow < statusChangeThreshold) {
    throw _boom.default.badRequest(`Invalid values,lookBackWindow (${lookBackWindow}) must be equal to or greater than statusChangeThreshold (${statusChangeThreshold}).`);
  }
};
class RulesSettingsFlappingClient {
  constructor(options) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "savedObjectsClient", void 0);
    (0, _defineProperty2.default)(this, "getModificationMetadata", void 0);
    this.logger = options.logger;
    this.savedObjectsClient = options.savedObjectsClient;
    this.getModificationMetadata = options.getModificationMetadata;
  }
  async get() {
    const rulesSettings = await this.getOrCreate();
    if (!rulesSettings.attributes.flapping) {
      this.logger.error('Failed to get flapping rules setting for current space.');
      throw new Error('Failed to get flapping rules setting for current space. Flapping settings are undefined');
    }
    return rulesSettings.attributes.flapping;
  }
  async update(newFlappingProperties) {
    return await (0, _retry_if_conflicts.retryIfConflicts)(this.logger, 'ruleSettingsClient.flapping.update()', async () => await this.updateWithOCC(newFlappingProperties));
  }
  async updateWithOCC(newFlappingProperties) {
    try {
      _schemas.flappingSchema.validate(newFlappingProperties);
      verifyFlappingSettings(newFlappingProperties);
    } catch (e) {
      this.logger.error(`Failed to verify new flapping settings properties when updating. Error: ${e}`);
      throw e;
    }
    const {
      attributes,
      version
    } = await this.getOrCreate();
    if (!attributes.flapping) {
      throw new Error('Flapping settings are undefined');
    }
    const modificationMetadata = await this.getModificationMetadata();
    try {
      const result = await this.savedObjectsClient.update(_common.RULES_SETTINGS_SAVED_OBJECT_TYPE, _common.RULES_SETTINGS_FLAPPING_SAVED_OBJECT_ID, {
        flapping: {
          ...attributes.flapping,
          ...newFlappingProperties,
          updatedAt: modificationMetadata.updatedAt,
          updatedBy: modificationMetadata.updatedBy
        }
      }, {
        version
      });
      return result.attributes.flapping;
    } catch (e) {
      const errorMessage = 'savedObjectsClient errored trying to update flapping settings';
      this.logger.error(`${errorMessage}: ${e}`);
      throw _boom.default.boomify(e, {
        message: errorMessage
      });
    }
  }
  async getSettings() {
    try {
      return await this.savedObjectsClient.get(_common.RULES_SETTINGS_SAVED_OBJECT_TYPE, _common.RULES_SETTINGS_FLAPPING_SAVED_OBJECT_ID);
    } catch (e) {
      this.logger.error(`Failed to get flapping rules setting for current space. Error: ${e}`);
      throw e;
    }
  }
  async createSettings() {
    const modificationMetadata = await this.getModificationMetadata();
    try {
      return await this.savedObjectsClient.create(_common.RULES_SETTINGS_SAVED_OBJECT_TYPE, {
        flapping: {
          ..._common.DEFAULT_FLAPPING_SETTINGS,
          ...modificationMetadata
        }
      }, {
        id: _common.RULES_SETTINGS_FLAPPING_SAVED_OBJECT_ID,
        overwrite: true
      });
    } catch (e) {
      this.logger.error(`Failed to create flapping rules setting for current space. Error: ${e}`);
      throw e;
    }
  }

  /**
   * Helper function to ensure that a rules-settings saved object always exists.
   * Ensures the creation of the saved object is done lazily during retrieval.
   */
  async getOrCreate() {
    try {
      return await this.getSettings();
    } catch (e) {
      if (_server.SavedObjectsErrorHelpers.isNotFoundError(e)) {
        this.logger.info('Creating new default flapping rules settings for current space.');
        return await this.createSettings();
      }
      this.logger.error(`Failed to get flapping rules setting for current space. Error: ${e}`);
      throw e;
    }
  }
}
exports.RulesSettingsFlappingClient = RulesSettingsFlappingClient;