"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateScheduleLimit = exports.getScheduleFrequency = void 0;
var _parse_duration = require("../../../../../common/parse_duration");
var _schema = require("./schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const convertIntervalToFrequency = (context, schedule) => {
  let scheduleFrequency = 0;
  try {
    // Normalize the interval (period) in terms of minutes
    scheduleFrequency = (0, _parse_duration.convertDurationToFrequency)(schedule);
  } catch (e) {
    context.logger.warn(`Failed to parse rule schedule interval for schedule frequency calculation: ${e.message}`);
  }
  return scheduleFrequency;
};
const getScheduleFrequency = async context => {
  var _response$aggregation, _response$aggregation2;
  const response = await context.internalSavedObjectsRepository.find({
    type: 'alert',
    filter: 'alert.attributes.enabled: true',
    namespaces: ['*'],
    aggs: {
      schedule_intervals: {
        terms: {
          field: 'alert.attributes.schedule.interval'
        }
      }
    }
  });
  const buckets = (_response$aggregation = (_response$aggregation2 = response.aggregations) === null || _response$aggregation2 === void 0 ? void 0 : _response$aggregation2.schedule_intervals.buckets) !== null && _response$aggregation !== void 0 ? _response$aggregation : [];
  const totalScheduledPerMinute = buckets.reduce((result, {
    key,
    doc_count: occurrence
  }) => {
    const scheduleFrequency = convertIntervalToFrequency(context, key);

    // Sum up all of the frequencies, since this is an aggregation.
    return result + scheduleFrequency * occurrence;
  }, 0);
  const result = {
    totalScheduledPerMinute,
    remainingSchedulesPerMinute: Math.max(context.maxScheduledPerMinute - totalScheduledPerMinute, 0)
  };
  try {
    _schema.getSchemaFrequencyResultSchema.validate(result);
  } catch (e) {
    context.logger.warn(`Error validating rule schedules per minute: ${e}`);
  }
  return result;
};
exports.getScheduleFrequency = getScheduleFrequency;
const validateScheduleLimit = async params => {
  const {
    context,
    prevInterval = [],
    updatedInterval = []
  } = params;
  const prevIntervalArray = Array.isArray(prevInterval) ? prevInterval : [prevInterval];
  const updatedIntervalArray = Array.isArray(updatedInterval) ? updatedInterval : [updatedInterval];
  const prevSchedulePerMinute = prevIntervalArray.reduce((result, interval) => {
    const scheduleFrequency = convertIntervalToFrequency(context, interval);
    return result + scheduleFrequency;
  }, 0);
  const updatedSchedulesPerMinute = updatedIntervalArray.reduce((result, interval) => {
    const scheduleFrequency = convertIntervalToFrequency(context, interval);
    return result + scheduleFrequency;
  }, 0);
  const {
    remainingSchedulesPerMinute
  } = await getScheduleFrequency(context);

  // Compute the new remaining schedules per minute if we are editing rules.
  // So we add back the edited schedules, since we assume those are being edited.
  const computedRemainingSchedulesPerMinute = remainingSchedulesPerMinute + prevSchedulePerMinute;
  if (computedRemainingSchedulesPerMinute < updatedSchedulesPerMinute) {
    return {
      interval: updatedSchedulesPerMinute,
      intervalAvailable: remainingSchedulesPerMinute
    };
  }
  return null;
};
exports.validateScheduleLimit = validateScheduleLimit;