"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.significantItemsHandlerFactory = void 0;
var _async = require("async");
var _mlAggUtils = require("@kbn/ml-agg-utils");
var _i18n = require("@kbn/i18n");
var _actions = require("../../../../common/api/log_rate_analysis/actions");
var _is_request_aborted_error = require("../../../lib/is_request_aborted_error");
var _fetch_significant_categories = require("../queries/fetch_significant_categories");
var _fetch_significant_term_p_values = require("../queries/fetch_significant_term_p_values");
var _constants = require("../response_stream_utils/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const significantItemsHandlerFactory = ({
  abortSignal,
  client,
  logDebugMessage,
  logger,
  requestBody,
  responseStream,
  stateHandler,
  version
}) => async ({
  fieldCandidates,
  textFieldCandidates
}) => {
  var _requestBody$override;
  let fieldCandidatesCount = fieldCandidates.length;

  // This will store the combined count of detected significant log patterns and keywords
  let fieldValuePairsCount = 0;
  const significantCategories = [];
  if (version === '1') {
    var _overrides$significan, _overrides, _overrides$significan2;
    significantCategories.push(...((_overrides$significan = (_overrides = requestBody.overrides) === null || _overrides === void 0 ? void 0 : (_overrides$significan2 = _overrides.significantTerms) === null || _overrides$significan2 === void 0 ? void 0 : _overrides$significan2.filter(d => d.type === _mlAggUtils.SIGNIFICANT_ITEM_TYPE.LOG_PATTERN)) !== null && _overrides$significan !== void 0 ? _overrides$significan : []));
  }
  if (version === '2') {
    var _overrides$significan3, _overrides2, _overrides2$significa;
    significantCategories.push(...((_overrides$significan3 = (_overrides2 = requestBody.overrides) === null || _overrides2 === void 0 ? void 0 : (_overrides2$significa = _overrides2.significantItems) === null || _overrides2$significa === void 0 ? void 0 : _overrides2$significa.filter(d => d.type === _mlAggUtils.SIGNIFICANT_ITEM_TYPE.LOG_PATTERN)) !== null && _overrides$significan3 !== void 0 ? _overrides$significan3 : []));
  }

  // Get significant categories of text fields
  if (textFieldCandidates.length > 0) {
    significantCategories.push(...(await (0, _fetch_significant_categories.fetchSignificantCategories)(client, requestBody, textFieldCandidates, logger, stateHandler.sampleProbability(), responseStream.pushError, abortSignal)));
    if (significantCategories.length > 0) {
      responseStream.push((0, _actions.addSignificantItemsAction)(significantCategories, version));
    }
  }
  const significantTerms = [];
  if (version === '1') {
    var _overrides$significan4, _overrides3, _overrides3$significa;
    significantTerms.push(...((_overrides$significan4 = (_overrides3 = requestBody.overrides) === null || _overrides3 === void 0 ? void 0 : (_overrides3$significa = _overrides3.significantTerms) === null || _overrides3$significa === void 0 ? void 0 : _overrides3$significa.filter(d => d.type === _mlAggUtils.SIGNIFICANT_ITEM_TYPE.KEYWORD)) !== null && _overrides$significan4 !== void 0 ? _overrides$significan4 : []));
  }
  if (version === '2') {
    var _overrides$significan5, _overrides4, _overrides4$significa;
    significantTerms.push(...((_overrides$significan5 = (_overrides4 = requestBody.overrides) === null || _overrides4 === void 0 ? void 0 : (_overrides4$significa = _overrides4.significantItems) === null || _overrides4$significa === void 0 ? void 0 : _overrides4$significa.filter(d => d.type === _mlAggUtils.SIGNIFICANT_ITEM_TYPE.KEYWORD)) !== null && _overrides$significan5 !== void 0 ? _overrides$significan5 : []));
  }
  const fieldsToSample = new Set();
  let remainingFieldCandidates;
  let loadingStepSizePValues = _constants.PROGRESS_STEP_P_VALUES;
  if ((_requestBody$override = requestBody.overrides) !== null && _requestBody$override !== void 0 && _requestBody$override.remainingFieldCandidates) {
    var _requestBody$override2, _requestBody$override3, _requestBody$override4, _requestBody$override5;
    fieldCandidates.push(...((_requestBody$override2 = requestBody.overrides) === null || _requestBody$override2 === void 0 ? void 0 : _requestBody$override2.remainingFieldCandidates));
    remainingFieldCandidates = (_requestBody$override3 = requestBody.overrides) === null || _requestBody$override3 === void 0 ? void 0 : _requestBody$override3.remainingFieldCandidates;
    fieldCandidatesCount = fieldCandidates.length;
    loadingStepSizePValues = _constants.LOADED_FIELD_CANDIDATES + _constants.PROGRESS_STEP_P_VALUES - ((_requestBody$override4 = (_requestBody$override5 = requestBody.overrides) === null || _requestBody$override5 === void 0 ? void 0 : _requestBody$override5.loaded) !== null && _requestBody$override4 !== void 0 ? _requestBody$override4 : _constants.PROGRESS_STEP_P_VALUES);
  } else {
    remainingFieldCandidates = fieldCandidates;
  }
  logDebugMessage('Fetch p-values.');
  const pValuesQueue = (0, _async.queue)(async function (fieldCandidate) {
    stateHandler.loaded(1 / fieldCandidatesCount * loadingStepSizePValues, false);
    let pValues;
    try {
      pValues = await (0, _fetch_significant_term_p_values.fetchSignificantTermPValues)(client, requestBody, [fieldCandidate], logger, stateHandler.sampleProbability(), responseStream.pushError, abortSignal);
    } catch (e) {
      if (!(0, _is_request_aborted_error.isRequestAbortedError)(e)) {
        logger.error(`Failed to fetch p-values for '${fieldCandidate}', got: \n${e.toString()}`);
        responseStream.pushError(`Failed to fetch p-values for '${fieldCandidate}'.`);
      }
      return;
    }
    remainingFieldCandidates = remainingFieldCandidates.filter(d => d !== fieldCandidate);
    if (pValues.length > 0) {
      pValues.forEach(d => {
        fieldsToSample.add(d.fieldName);
      });
      significantTerms.push(...pValues);
      responseStream.push((0, _actions.addSignificantItemsAction)(pValues, version));
    }
    responseStream.push((0, _actions.updateLoadingStateAction)({
      ccsWarning: false,
      loaded: stateHandler.loaded(),
      loadingState: _i18n.i18n.translate('xpack.aiops.logRateAnalysis.loadingState.identifiedFieldValuePairs', {
        defaultMessage: 'Identified {fieldValuePairsCount, plural, one {# significant field/value pair} other {# significant field/value pairs}}.',
        values: {
          fieldValuePairsCount
        }
      }),
      remainingFieldCandidates
    }));
  }, _constants.MAX_CONCURRENT_QUERIES);
  pValuesQueue.push(fieldCandidates, err => {
    if (err) {
      logger.error(`Failed to fetch p-values.', got: \n${err.toString()}`);
      responseStream.pushError(`Failed to fetch p-values.`);
      pValuesQueue.kill();
      responseStream.end();
    } else if (stateHandler.shouldStop()) {
      logDebugMessage('shouldStop fetching p-values.');
      pValuesQueue.kill();
      responseStream.end();
    }
  });
  await pValuesQueue.drain();
  fieldValuePairsCount = significantCategories.length + significantTerms.length;
  if (fieldValuePairsCount === 0) {
    logDebugMessage('Stopping analysis, did not find significant terms.');
    responseStream.endWithUpdatedLoadingState();
    return;
  }
  return {
    fieldValuePairsCount,
    significantCategories,
    significantTerms
  };
};
exports.significantItemsHandlerFactory = significantItemsHandlerFactory;