"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.indexInfoHandlerFactory = void 0;
var _i18n = require("@kbn/i18n");
var _actions = require("../../../../common/api/log_rate_analysis/actions");
var _is_request_aborted_error = require("../../../lib/is_request_aborted_error");
var _fetch_index_info = require("../queries/fetch_index_info");
var _constants = require("../response_stream_utils/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const indexInfoHandlerFactory = options => async () => {
  var _requestBody$override;
  const {
    abortSignal,
    client,
    logDebugMessage,
    logger,
    requestBody,
    responseStream,
    stateHandler
  } = options;
  const fieldCandidates = [];
  let fieldCandidatesCount = fieldCandidates.length;
  const textFieldCandidates = [];
  let totalDocCount = 0;
  let zeroDocsFallback = false;
  if (!((_requestBody$override = requestBody.overrides) !== null && _requestBody$override !== void 0 && _requestBody$override.remainingFieldCandidates)) {
    logDebugMessage('Fetch index information.');
    responseStream.push((0, _actions.updateLoadingStateAction)({
      ccsWarning: false,
      loaded: stateHandler.loaded(),
      loadingState: _i18n.i18n.translate('xpack.aiops.logRateAnalysis.loadingState.loadingIndexInformation', {
        defaultMessage: 'Loading index information.'
      })
    }));
    try {
      const indexInfo = await (0, _fetch_index_info.fetchIndexInfo)(client, requestBody, ['message', 'error.message'], abortSignal);
      fieldCandidates.push(...indexInfo.fieldCandidates);
      fieldCandidatesCount = fieldCandidates.length;
      textFieldCandidates.push(...indexInfo.textFieldCandidates);
      totalDocCount = indexInfo.deviationTotalDocCount;
      zeroDocsFallback = indexInfo.zeroDocsFallback;
    } catch (e) {
      if (!(0, _is_request_aborted_error.isRequestAbortedError)(e)) {
        logger.error(`Failed to fetch index information, got: \n${e.toString()}`);
        responseStream.pushError(`Failed to fetch index information.`);
      }
      responseStream.end();
      return;
    }
    logDebugMessage(`Total document count: ${totalDocCount}`);
    stateHandler.loaded(_constants.LOADED_FIELD_CANDIDATES, false);
    responseStream.pushPingWithTimeout();
    responseStream.push((0, _actions.updateLoadingStateAction)({
      ccsWarning: false,
      loaded: stateHandler.loaded(),
      loadingState: _i18n.i18n.translate('xpack.aiops.logRateAnalysis.loadingState.identifiedFieldCandidates', {
        defaultMessage: 'Identified {fieldCandidatesCount, plural, one {# field candidate} other {# field candidates}}.',
        values: {
          fieldCandidatesCount
        }
      })
    }));
    responseStream.push((0, _actions.setZeroDocsFallback)(zeroDocsFallback));
    if (fieldCandidatesCount === 0) {
      responseStream.endWithUpdatedLoadingState();
    } else if (stateHandler.shouldStop()) {
      logDebugMessage('shouldStop after fetching field candidates.');
      responseStream.end();
      return;
    }
  }
  return {
    fieldCandidates,
    textFieldCandidates,
    zeroDocsFallback
  };
};
exports.indexInfoHandlerFactory = indexInfoHandlerFactory;