"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.convertToLens = void 0;
var _common = require("@kbn/data-plugin/common");
var _convert_to_lens = require("@kbn/visualizations-plugin/common/convert_to_lens");
var _uuid = require("uuid");
var _enums = require("../../../common/enums");
var _services = require("../../services");
var _datasource = require("../lib/datasource");
var _series = require("../lib/series");
var _xy = require("../lib/configurations/xy");
var _convert = require("../lib/convert");
var _metrics = require("../lib/metrics");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const excludeMetaFromLayers = layers => {
  const newLayers = {};
  Object.entries(layers).forEach(([layerId, layer]) => {
    const columns = layer.columns.map(_convert.excludeMetaFromColumn);
    newLayers[layerId] = {
      ...layer,
      columns
    };
  });
  return newLayers;
};
const invalidModelError = () => new Error('Invalid model');
const convertToLens = async ({
  params: model
}) => {
  const dataViews = (0, _services.getDataViewsStart)();
  try {
    const extendedLayers = {};
    const seriesNum = model.series.filter(series => !series.hidden).length;

    // handle multiple layers/series
    for (const [layerIdx, series] of model.series.entries()) {
      if (series.hidden) {
        continue;
      }

      // not valid time shift
      if (series.offset_time && (0, _common.parseTimeShift)(series.offset_time) === 'invalid') {
        throw invalidModelError();
      }
      if (!(0, _metrics.isValidMetrics)(series.metrics, _enums.PANEL_TYPES.TIMESERIES)) {
        throw invalidModelError();
      }
      const datasourceInfo = await (0, _datasource.extractOrGenerateDatasourceInfo)(model.index_pattern, model.time_field, Boolean(series.override_index_pattern), series.series_index_pattern, series.series_time_field, dataViews);
      if (!datasourceInfo) {
        throw invalidModelError();
      }
      const {
        indexPatternId,
        indexPattern,
        timeField
      } = datasourceInfo;
      if (!timeField) {
        throw invalidModelError();
      }
      const dateHistogramColumn = (0, _convert.convertToDateHistogramColumn)(model, series, indexPattern, {
        fieldName: timeField,
        isSplit: false
      });
      if (dateHistogramColumn === null) {
        throw invalidModelError();
      }
      // handle multiple metrics
      const metricsColumns = (0, _series.getMetricsColumns)(series, indexPattern, seriesNum, {
        isStaticValueColumnSupported: true
      });
      if (metricsColumns === null) {
        throw invalidModelError();
      }
      const bucketsColumns = (0, _series.getBucketsColumns)(model, series, metricsColumns, indexPattern, true);
      if (bucketsColumns === null) {
        throw invalidModelError();
      }
      const isReferenceLine = metricsColumns.length === 1 && metricsColumns[0].operationType === 'static_value';

      // only static value without split is supported
      if (isReferenceLine && bucketsColumns.length) {
        throw invalidModelError();
      }
      const layerId = (0, _uuid.v4)();
      extendedLayers[layerIdx] = {
        ignoreGlobalFilters: Boolean(model.ignore_global_filter || series.ignore_global_filter),
        indexPatternId,
        layerId,
        columns: isReferenceLine ? [...metricsColumns] : [...metricsColumns, dateHistogramColumn, ...bucketsColumns],
        columnOrder: []
      };
    }
    const configLayers = await (0, _xy.getLayers)(extendedLayers, model, dataViews);
    if (configLayers === null) {
      throw invalidModelError();
    }
    const configuration = (0, _xy.getConfigurationForTimeseries)(model, configLayers);
    const layers = Object.values(excludeMetaFromLayers(extendedLayers));
    const annotationIndexPatterns = configuration.layers.reduce((acc, layer) => {
      if ((0, _convert_to_lens.isAnnotationsLayer)(layer)) {
        return [...acc, layer.indexPatternId];
      }
      return acc;
    }, []);
    return {
      type: 'lnsXY',
      layers,
      configuration,
      indexPatternIds: [...(0, _convert_to_lens.getIndexPatternIds)(layers), ...annotationIndexPatterns]
    };
  } catch (e) {
    return null;
  }
};
exports.convertToLens = convertToLens;