"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CUSTOM_METRIC = void 0;
exports.isCompatibleAggregation = isCompatibleAggregation;
exports.safeMakeLabel = safeMakeLabel;
exports.useAvailableOptions = useAvailableOptions;
exports.useFallbackMetric = useFallbackMetric;
exports.useValidation = useValidation;
var _react = require("react");
var _i18n = require("@kbn/i18n");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const DEFAULT_METRIC = 'custom';
const CUSTOM_METRIC = {
  text: _i18n.i18n.translate('visDefaultEditor.controls.customMetricLabel', {
    defaultMessage: 'Custom metric'
  }),
  value: DEFAULT_METRIC
};
exports.CUSTOM_METRIC = CUSTOM_METRIC;
function useCompatibleAggCallback(aggFilter) {
  /* eslint-disable-next-line react-hooks/exhaustive-deps */
  return (0, _react.useCallback)(isCompatibleAggregation(aggFilter), [aggFilter]);
}

/**
 * the effect is used to set up a default metric aggregation in case,
 * when previously selected metric has been removed
 */
function useFallbackMetric(setValue, aggFilter, metricAggs, value, fallbackValue) {
  const isCompatibleAgg = useCompatibleAggCallback(aggFilter);
  (0, _react.useEffect)(() => {
    if (metricAggs && value && value !== DEFAULT_METRIC) {
      // ensure that metric is set to a valid agg
      const respAgg = metricAggs.filter(isCompatibleAgg).find(aggregation => aggregation.id === value);
      if (!respAgg && value !== fallbackValue) {
        setValue(fallbackValue);
      }
    }
  }, [setValue, isCompatibleAgg, metricAggs, value, fallbackValue]);
}

/**
 * this makes an array of available options in appropriate format for EuiSelect,
 * calculates if an option is disabled
 */
function useAvailableOptions(aggFilter, metricAggs = [], defaultOptions = []) {
  const isCompatibleAgg = useCompatibleAggCallback(aggFilter);
  const options = (0, _react.useMemo)(() => [...metricAggs.map(respAgg => ({
    text: _i18n.i18n.translate('visDefaultEditor.controls.definiteMetricLabel', {
      defaultMessage: 'Metric: {metric}',
      values: {
        metric: safeMakeLabel(respAgg)
      }
    }),
    value: respAgg.id,
    disabled: !isCompatibleAgg(respAgg)
  })), CUSTOM_METRIC, ...defaultOptions], [metricAggs, defaultOptions, isCompatibleAgg]);
  return options;
}

/**
 * the effect is used to set up the editor form validity
 * and reset it if a param has been removed
 */
function useValidation(setValidity, isValid) {
  (0, _react.useEffect)(() => {
    setValidity(isValid);
    return () => setValidity(true);
  }, [isValid, setValidity]);
}
function safeMakeLabel(agg) {
  try {
    return agg.makeLabel();
  } catch (e) {
    return _i18n.i18n.translate('visDefaultEditor.controls.aggNotValidLabel', {
      defaultMessage: '- agg not valid -'
    });
  }
}
function isCompatibleAggregation(aggFilter) {
  return agg => {
    return !aggFilter.includes(`!${agg.type.name}`);
  };
}