"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildAggDescription = buildAggDescription;
exports.getAggParamsToRender = getAggParamsToRender;
exports.getAggTypeOptions = getAggTypeOptions;
exports.isInvalidParamsTouched = isInvalidParamsTouched;
var _lodash = require("lodash");
var _agg_filters = require("../agg_filters");
var _utils = require("../utils");
var _agg_params_map = require("./agg_params_map");
var _schemas = require("../schemas");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function getAggParamsToRender({
  agg,
  editorConfig,
  metricAggs,
  state,
  schemas,
  hideCustomLabel
}) {
  const params = {
    basic: [],
    advanced: []
  };
  const paramsToRender = agg.type && agg.type.params
  // Filter out, i.e. don't render, any parameter that is hidden via the editor config.
  .filter(param => !(0, _lodash.get)(editorConfig, [param.name, 'hidden'], false)) || [];
  const schema = (0, _schemas.getSchemaByName)(schemas, agg.schema);
  // build collection of agg params components
  paramsToRender.forEach((param, index) => {
    let indexedFields = [];
    let fields;
    if (hideCustomLabel && param.name === 'customLabel') {
      return;
    }
    if (param.name === 'timeShift') {
      return;
    }
    // if field param exists, compute allowed fields
    if (param.type === 'field') {
      let availableFields = param.getAvailableFields(agg);
      // should be refactored in the future to provide a more general way
      // for visualization to override some agg config settings
      if (agg.type.name === 'top_hits' && param.name === 'field') {
        const allowStrings = (0, _lodash.get)(schema, `aggSettings[${agg.type.name}].allowStrings`, false);
        if (!allowStrings) {
          availableFields = availableFields.filter(field => field.type === 'number');
        }
      }
      fields = (0, _agg_filters.filterAggTypeFields)(availableFields, agg);
      indexedFields = (0, _utils.groupAndSortBy)(fields, 'type', 'displayName', 'name');
      if (fields && !indexedFields.length && index > 0) {
        // don't draw the rest of the options if there are no indexed fields and it's an extra param (index > 0).
        return;
      }
    }
    const type = param.advanced ? 'advanced' : 'basic';
    let paramEditor;
    if (agg.type.subtype && _agg_params_map.aggParamsMap[agg.type.subtype]) {
      paramEditor = (0, _lodash.get)(_agg_params_map.aggParamsMap, [agg.type.subtype, param.name]);
    } else {
      const aggType = agg.type.type;
      const aggName = agg.type.name;
      const aggParams = (0, _lodash.get)(_agg_params_map.aggParamsMap, [aggType, aggName], {});
      paramEditor = (0, _lodash.get)(aggParams, param.name);
    }
    if (!paramEditor) {
      paramEditor = (0, _lodash.get)(_agg_params_map.aggParamsMap, ['common', param.type]);
    }

    // show params with an editor component
    if (paramEditor) {
      params[type].push({
        agg,
        aggParam: param,
        editorConfig,
        indexedFields,
        paramEditor,
        metricAggs,
        state,
        value: agg.params[param.name],
        schemas,
        hideCustomLabel
      });
    }
  });
  return params;
}
function getAggTypeOptions(aggTypes, agg, indexPattern, groupName, allowedAggs) {
  const aggTypeOptions = (0, _agg_filters.filterAggTypes)(aggTypes[groupName], indexPattern, agg, allowedAggs);
  return (0, _utils.groupAndSortBy)(aggTypeOptions, 'subtype', 'title');
}

/**
 * Calculates a ngModel touched state.
 * If an aggregation is not selected, it returns a value of touched agg selector state.
 * Else if there are no invalid agg params, it returns false.
 * Otherwise it returns true if each invalid param is touched.
 * @param aggType Selected aggregation.
 * @param aggTypeState State of aggregation selector.
 * @param aggParams State of aggregation parameters.
 */
function isInvalidParamsTouched(aggType, aggTypeState, aggParams) {
  if (!aggType) {
    return aggTypeState.touched;
  }
  const invalidParams = Object.values(aggParams).filter(param => !param.valid);
  if ((0, _lodash.isEmpty)(invalidParams)) {
    return false;
  }
  return invalidParams.every(param => param.touched);
}
function buildAggDescription(agg) {
  let description = '';
  if (agg.type && agg.type.makeLabel) {
    try {
      description = agg.type.makeLabel(agg);
    } catch (e) {
      // Date Histogram's `makeLabel` implementation invokes 'write' method for each param, including interval's 'write',
      // which throws an error when interval is undefined.
    }
  }
  return description;
}