"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerStatsRoute = registerStatsRoute;
var _configSchema = require("@kbn/config-schema");
var _i18n = require("@kbn/i18n");
var _rxjs = require("rxjs");
var _server = require("@kbn/core/server");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const SNAPSHOT_REGEX = /-snapshot/i;
function registerStatsRoute({
  router,
  config,
  collectorSet,
  metrics,
  overallStatus$
}) {
  const getClusterUuid = async asCurrentUser => {
    const body = await asCurrentUser.info({
      filter_path: 'cluster_uuid'
    });
    const {
      cluster_uuid: uuid
    } = body;
    return uuid;
  };
  router.get({
    path: '/api/stats',
    options: {
      authRequired: !config.allowAnonymous,
      // The `api` tag ensures that unauthenticated calls receive a 401 rather than a 302 redirect to login page.
      // The `security:acceptJWT` tag allows route to be accessed with JWT credentials. It points to
      // ROUTE_TAG_ACCEPT_JWT from '@kbn/security-plugin/server' that cannot be imported here directly.
      tags: ['api', 'security:acceptJWT'],
      access: 'public' // needs to be public to allow access from "system" users like metricbeat.
    },

    validate: {
      query: _configSchema.schema.object({
        extended: _configSchema.schema.oneOf([_configSchema.schema.literal(''), _configSchema.schema.boolean()], {
          defaultValue: false
        }),
        legacy: _configSchema.schema.oneOf([_configSchema.schema.literal(''), _configSchema.schema.boolean()], {
          defaultValue: false
        }),
        exclude_usage: _configSchema.schema.oneOf([_configSchema.schema.literal(''), _configSchema.schema.boolean()], {
          defaultValue: true
        })
      })
    }
  }, async (context, req, res) => {
    const requestQuery = req.query;
    const isExtended = requestQuery.extended === '' || requestQuery.extended;
    const isLegacy = requestQuery.legacy === '' || requestQuery.legacy;
    let extended = {};
    if (isExtended) {
      const core = await context.core;
      const {
        asInternalUser
      } = core.elasticsearch.client;
      // as of https://github.com/elastic/kibana/pull/151082, usage will always be an empty object.

      const clusterUuid = await getClusterUuid(asInternalUser);
      const extendedClusterUuid = isLegacy ? {
        clusterUuid
      } : {
        cluster_uuid: clusterUuid
      };
      extended = {
        usage: {},
        ...extendedClusterUuid
      };
    }

    // Guaranteed to resolve immediately due to replay effect on getOpsMetrics$
    const {
      collected_at: collectedAt,
      ...lastMetrics
    } = await (0, _rxjs.firstValueFrom)(metrics.getOpsMetrics$());
    const overallStatus = await (0, _rxjs.firstValueFrom)(overallStatus$);
    const kibanaStats = collectorSet.toApiFieldNames({
      ...lastMetrics,
      kibana: {
        uuid: config.uuid,
        name: config.server.name,
        index: config.kibanaIndex,
        host: config.server.hostname,
        locale: _i18n.i18n.getLocale(),
        transport_address: `${config.server.hostname}:${config.server.port}`,
        version: config.kibanaVersion.replace(SNAPSHOT_REGEX, ''),
        snapshot: SNAPSHOT_REGEX.test(config.kibanaVersion),
        status: ServiceStatusToLegacyState[overallStatus.level.toString()]
      },
      last_updated: collectedAt.toISOString(),
      collection_interval_in_millis: metrics.collectionInterval
    });
    const body = {
      ...kibanaStats,
      ...extended
    };
    return res.ok({
      body
    });
  });
}
const ServiceStatusToLegacyState = {
  [_server.ServiceStatusLevels.critical.toString()]: 'red',
  [_server.ServiceStatusLevels.unavailable.toString()]: 'red',
  [_server.ServiceStatusLevels.degraded.toString()]: 'yellow',
  [_server.ServiceStatusLevels.available.toString()]: 'green'
};