"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.UiActionsService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _actions2 = require("../actions");
var _trigger_internal = require("../triggers/trigger_internal");
var _ui_actions_execution_service = require("./ui_actions_execution_service");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

class UiActionsService {
  constructor({
    triggers: _triggers = new Map(),
    actions: _actions = new Map(),
    triggerToActions: _triggerToActions = new Map()
  } = {}) {
    (0, _defineProperty2.default)(this, "executionService", new _ui_actions_execution_service.UiActionsExecutionService());
    (0, _defineProperty2.default)(this, "triggers", void 0);
    (0, _defineProperty2.default)(this, "actions", void 0);
    (0, _defineProperty2.default)(this, "triggerToActions", void 0);
    (0, _defineProperty2.default)(this, "registerTrigger", trigger => {
      if (this.triggers.has(trigger.id)) {
        throw new Error(`Trigger [trigger.id = ${trigger.id}] already registered.`);
      }
      const triggerInternal = new _trigger_internal.TriggerInternal(this, trigger);
      this.triggers.set(trigger.id, triggerInternal);
      this.triggerToActions.set(trigger.id, []);
    });
    (0, _defineProperty2.default)(this, "getTrigger", triggerId => {
      const trigger = this.triggers.get(triggerId);
      if (!trigger) {
        throw new Error(`Trigger [triggerId = ${triggerId}] does not exist.`);
      }
      return trigger.contract;
    });
    (0, _defineProperty2.default)(this, "registerAction", definition => {
      if (this.actions.has(definition.id)) {
        throw new Error(`Action [action.id = ${definition.id}] already registered.`);
      }
      const action = new _actions2.ActionInternal(definition);
      this.actions.set(action.id, action);
      return action;
    });
    (0, _defineProperty2.default)(this, "unregisterAction", actionId => {
      if (!this.actions.has(actionId)) {
        throw new Error(`Action [action.id = ${actionId}] is not registered.`);
      }
      this.actions.delete(actionId);
    });
    (0, _defineProperty2.default)(this, "hasAction", actionId => {
      return this.actions.has(actionId);
    });
    (0, _defineProperty2.default)(this, "attachAction", (triggerId, actionId) => {
      const trigger = this.triggers.get(triggerId);
      if (!trigger) {
        throw new Error(`No trigger [triggerId = ${triggerId}] exists, for attaching action [actionId = ${actionId}].`);
      }
      const actionIds = this.triggerToActions.get(triggerId);
      if (!actionIds.find(id => id === actionId)) {
        this.triggerToActions.set(triggerId, [...actionIds, actionId]);
      }
    });
    (0, _defineProperty2.default)(this, "detachAction", (triggerId, actionId) => {
      const trigger = this.triggers.get(triggerId);
      if (!trigger) {
        throw new Error(`No trigger [triggerId = ${triggerId}] exists, for detaching action [actionId = ${actionId}].`);
      }
      const actionIds = this.triggerToActions.get(triggerId);
      this.triggerToActions.set(triggerId, actionIds.filter(id => id !== actionId));
    });
    /**
     * `addTriggerAction` is similar to `attachAction` as it attaches action to a
     * trigger, but it also registers the action, if it has not been registered, yet.
     */
    (0, _defineProperty2.default)(this, "addTriggerAction", (triggerId, action) => {
      if (!this.actions.has(action.id)) this.registerAction(action);
      this.attachAction(triggerId, action.id);
    });
    (0, _defineProperty2.default)(this, "getAction", id => {
      if (!this.actions.has(id)) {
        throw new Error(`Action [action.id = ${id}] not registered.`);
      }
      return this.actions.get(id);
    });
    (0, _defineProperty2.default)(this, "getTriggerActions", triggerId => {
      // This line checks if trigger exists, otherwise throws.
      this.getTrigger(triggerId);
      const actionIds = this.triggerToActions.get(triggerId);
      const actions = actionIds.map(actionId => this.actions.get(actionId)).filter(Boolean);
      return actions;
    });
    (0, _defineProperty2.default)(this, "getTriggerCompatibleActions", async (triggerId, context) => {
      const actions = this.getTriggerActions(triggerId);
      const isCompatibles = await Promise.all(actions.map(action => action.isCompatible({
        ...context,
        trigger: this.getTrigger(triggerId)
      })));
      return actions.reduce((acc, action, i) => {
        if (isCompatibles[i]) {
          acc.push(action);
        }
        return acc;
      }, []);
    });
    /**
     * @deprecated
     *
     * Use `plugins.uiActions.getTrigger(triggerId).exec(params)` instead.
     */
    (0, _defineProperty2.default)(this, "executeTriggerActions", async (triggerId, context) => {
      const trigger = this.getTrigger(triggerId);
      await trigger.exec(context);
    });
    /**
     * Removes all registered triggers and actions.
     */
    (0, _defineProperty2.default)(this, "clear", () => {
      this.actions.clear();
      this.triggers.clear();
      this.triggerToActions.clear();
    });
    /**
     * "Fork" a separate instance of `UiActionsService` that inherits all existing
     * triggers and actions, but going forward all new triggers and actions added
     * to this instance of `UiActionsService` are only available within this instance.
     */
    (0, _defineProperty2.default)(this, "fork", () => {
      const triggers = new Map();
      const actions = new Map();
      const triggerToActions = new Map();
      for (const [key, value] of this.triggers.entries()) triggers.set(key, value);
      for (const [key, value] of this.actions.entries()) actions.set(key, value);
      for (const [key, value] of this.triggerToActions.entries()) triggerToActions.set(key, [...value]);
      return new UiActionsService({
        triggers,
        actions,
        triggerToActions
      });
    });
    this.triggers = _triggers;
    this.actions = _actions;
    this.triggerToActions = _triggerToActions;
  }
}
exports.UiActionsService = UiActionsService;