"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.categoryHasFields = exports.TABLE_HEIGHT = exports.RESET_FIELDS_CLASS_NAME = exports.FIELD_BROWSER_WIDTH = exports.CLOSE_BUTTON_CLASS_NAME = exports.CATEGORY_TABLE_CLASS_NAME = void 0;
exports.filterBrowserFieldsByFieldName = filterBrowserFieldsByFieldName;
exports.isEscape = exports.getIconFromType = exports.getFieldCount = exports.getExampleText = exports.getEmptyValue = exports.getDescription = exports.getCategory = exports.filterSelectedBrowserFields = void 0;
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _fp = require("lodash/fp");
var _translations = require("../translations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const FIELD_BROWSER_WIDTH = 925;
exports.FIELD_BROWSER_WIDTH = FIELD_BROWSER_WIDTH;
const TABLE_HEIGHT = 260;

/** Returns true if the specified category has at least one field */
exports.TABLE_HEIGHT = TABLE_HEIGHT;
const categoryHasFields = category => category.fields != null && Object.keys(category.fields).length > 0;

/** Returns the count of fields in the specified category */
exports.categoryHasFields = categoryHasFields;
const getFieldCount = category => category != null && category.fields != null ? Object.keys(category.fields).length : 0;
exports.getFieldCount = getFieldCount;
const matchesSystemField = (field, searchTerm) => {
  const termsToMatch = [_translations.CASES.toLocaleLowerCase(), _translations.MAINTENANCE_WINDOWS.toLocaleLowerCase(), 'cases', 'maintenance windows'];
  const fieldsToMatch = [_ruleDataUtils.ALERT_CASE_IDS, _ruleDataUtils.ALERT_MAINTENANCE_WINDOW_IDS];
  const term = searchTerm.toLocaleLowerCase();
  const matchesField = fieldsToMatch.includes(field);
  const matchesTerm = termsToMatch.some(termToMatch => termToMatch.includes(term));
  return matchesField && matchesTerm;
};

/**
 * Filters the specified `BrowserFields` to return a new collection where every
 * category contains at least one field name that matches the specified substring.
 */
function filterBrowserFieldsByFieldName({
  browserFields,
  substring
}) {
  const trimmedSubstring = substring.trim();
  // an empty search param will match everything, so return the original browserFields
  if (trimmedSubstring === '') {
    return browserFields;
  }
  const result = {};
  for (const [categoryName, categoryDescriptor] of Object.entries(browserFields)) {
    if (!categoryDescriptor.fields) {
      // ignore any category that is missing fields. This is not expected to happen.
      continue;
    }

    // keep track of whether this category had a matching field, if so, we should emit it into the result
    let hadAMatch = false;

    // The fields that matched, for this `categoryName`
    const filteredFields = {};
    for (const [fieldName, fieldDescriptor] of Object.entries(categoryDescriptor.fields)) {
      // For historical reasons, we consider the name as it appears on the field descriptor, not the `fieldName` (attribute name) itself.
      // It is unclear if there is any point in continuing to do this.
      const fieldNameFromDescriptor = fieldDescriptor.name;
      if (!fieldNameFromDescriptor) {
        // Ignore any field that is missing a name in its descriptor. This is not expected to happen.
        continue;
      }

      // Check if this field matches (via substring comparison) the passed substring
      if (fieldNameFromDescriptor !== null && (fieldNameFromDescriptor.includes(trimmedSubstring) || matchesSystemField(fieldNameFromDescriptor, trimmedSubstring))) {
        // this field is a match, so we should emit this category into the result object.
        hadAMatch = true;

        // emit this field
        filteredFields[fieldName] = fieldDescriptor;
      }
    }
    if (hadAMatch) {
      // if at least one field matches, emit the category, but replace the `fields` attribute with the filtered fields
      result[categoryName] = {
        ...browserFields[categoryName],
        fields: filteredFields
      };
    }
  }
  return result;
}

/**
 * Filters the selected `BrowserFields` to return a new collection where every
 * category contains at least one field that is present in the `columnIds`.
 */
const filterSelectedBrowserFields = ({
  browserFields,
  columnIds
}) => {
  const selectedFieldIds = new Set(columnIds);
  const result = {};
  for (const [categoryName, categoryDescriptor] of Object.entries(browserFields)) {
    if (!categoryDescriptor.fields) {
      // ignore any category that is missing fields. This is not expected to happen.
      continue;
    }

    // keep track of whether this category had a selected field, if so, we should emit it into the result
    let hadSelected = false;

    // The selected fields for this `categoryName`
    const selectedFields = {};
    for (const [fieldName, fieldDescriptor] of Object.entries(categoryDescriptor.fields)) {
      // For historical reasons, we consider the name as it appears on the field descriptor, not the `fieldName` (attribute name) itself.
      // It is unclear if there is any point in continuing to do this.
      const fieldNameFromDescriptor = fieldDescriptor.name;
      if (!fieldNameFromDescriptor) {
        // Ignore any field that is missing a name in its descriptor. This is not expected to happen.
        continue;
      }
      if (selectedFieldIds.has(fieldNameFromDescriptor)) {
        hadSelected = true;
        selectedFields[fieldName] = fieldDescriptor;
      }
    }
    if (hadSelected) {
      result[categoryName] = {
        ...browserFields[categoryName],
        fields: selectedFields
      };
    }
  }
  return result;
};
exports.filterSelectedBrowserFields = filterSelectedBrowserFields;
const getIconFromType = type => {
  switch (type) {
    case 'string': // fall through
    case 'keyword':
      return 'string';
    case 'number': // fall through
    case 'long':
      return 'number';
    case 'date':
      return 'clock';
    case 'ip':
    case 'geo_point':
      return 'globe';
    case 'object':
      return 'questionInCircle';
    case 'float':
      return 'number';
    default:
      return 'questionInCircle';
  }
};
exports.getIconFromType = getIconFromType;
const getEmptyValue = () => '—';
exports.getEmptyValue = getEmptyValue;
const getCategory = fieldName => {
  var _fieldNameArray$;
  const fieldNameArray = fieldName === null || fieldName === void 0 ? void 0 : fieldName.split('.');
  if ((fieldNameArray === null || fieldNameArray === void 0 ? void 0 : fieldNameArray.length) === 1) {
    return 'base';
  }
  return (_fieldNameArray$ = fieldNameArray === null || fieldNameArray === void 0 ? void 0 : fieldNameArray[0]) !== null && _fieldNameArray$ !== void 0 ? _fieldNameArray$ : '(unknown)';
};
exports.getCategory = getCategory;
const getDescription = (fieldName, ecsFlat) => {
  var _ecsFlat$fieldName$de, _ecsFlat$fieldName;
  return (_ecsFlat$fieldName$de = (_ecsFlat$fieldName = ecsFlat[fieldName]) === null || _ecsFlat$fieldName === void 0 ? void 0 : _ecsFlat$fieldName.description) !== null && _ecsFlat$fieldName$de !== void 0 ? _ecsFlat$fieldName$de : '';
};

/** Returns example text, or an empty string if the field does not have an example */
exports.getDescription = getDescription;
const getExampleText = example => !(0, _fp.isEmpty)(example) ? `Example: ${example}` : '';

/** Returns `true` if the escape key was pressed */
exports.getExampleText = getExampleText;
const isEscape = event => event.key === 'Escape';
exports.isEscape = isEscape;
const CATEGORY_TABLE_CLASS_NAME = 'category-table';
exports.CATEGORY_TABLE_CLASS_NAME = CATEGORY_TABLE_CLASS_NAME;
const CLOSE_BUTTON_CLASS_NAME = 'close-button';
exports.CLOSE_BUTTON_CLASS_NAME = CLOSE_BUTTON_CLASS_NAME;
const RESET_FIELDS_CLASS_NAME = 'reset-fields';
exports.RESET_FIELDS_CLASS_NAME = RESET_FIELDS_CLASS_NAME;