"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useLoadTagsQuery = useLoadTagsQuery;
var _react = require("react");
var _i18n = require("@kbn/i18n");
var _reactQuery = require("@tanstack/react-query");
var _aggregate = require("../lib/rule_api/aggregate");
var _kibana = require("../../common/lib/kibana");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const EMPTY_TAGS = [];

// React query will refetch all prev pages when the cache keys change:
// https://github.com/TanStack/query/discussions/3576
function useLoadTagsQuery(props) {
  const {
    enabled,
    refresh,
    search,
    perPage,
    page = 1
  } = props;
  const {
    http,
    notifications: {
      toasts
    }
  } = (0, _kibana.useKibana)().services;
  const queryFn = ({
    pageParam
  }) => {
    if (pageParam) {
      return (0, _aggregate.loadRuleTags)({
        http,
        perPage: pageParam.perPage,
        page: pageParam.page,
        search
      });
    }
    return (0, _aggregate.loadRuleTags)({
      http,
      perPage,
      page,
      search
    });
  };
  const onErrorFn = () => {
    toasts.addDanger(_i18n.i18n.translate('xpack.triggersActionsUI.sections.rulesList.unableToLoadRuleTags', {
      defaultMessage: 'Unable to load rule tags'
    }));
  };
  const getNextPageParam = lastPage => {
    const totalPages = Math.max(1, Math.ceil(lastPage.total / lastPage.perPage));
    if (totalPages === lastPage.page) {
      return;
    }
    return {
      ...lastPage,
      page: lastPage.page + 1
    };
  };
  const {
    refetch,
    data,
    fetchNextPage,
    isLoading,
    isFetching,
    hasNextPage,
    isFetchingNextPage
  } = (0, _reactQuery.useInfiniteQuery)({
    queryKey: ['loadRuleTags', search, perPage, page, {
      refresh: refresh === null || refresh === void 0 ? void 0 : refresh.toISOString()
    }],
    queryFn,
    onError: onErrorFn,
    enabled,
    getNextPageParam,
    refetchOnWindowFocus: false
  });
  const tags = (0, _react.useMemo)(() => {
    return (data === null || data === void 0 ? void 0 : data.pages.reduce((result, current) => {
      return result.concat(current.data);
    }, [])) || EMPTY_TAGS;
  }, [data]);
  return {
    tags,
    hasNextPage,
    refetch,
    isLoading: isLoading || isFetching || isFetchingNextPage,
    fetchNextPage
  };
}