"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerTelemetryOptInStatsRoutes = registerTelemetryOptInStatsRoutes;
exports.sendTelemetryOptInStatus = sendTelemetryOptInStatus;
var _nodeFetch = _interopRequireDefault(require("node-fetch"));
var _configSchema = require("@kbn/config-schema");
var _routes = require("../../common/routes");
var _telemetry_config = require("../../common/telemetry_config");
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

async function sendTelemetryOptInStatus(telemetryCollectionManager, config, statsGetterConfig) {
  const {
    appendServerlessChannelsSuffix,
    sendUsageTo,
    newOptInStatus,
    currentKibanaVersion
  } = config;
  const optInStatusUrl = (0, _telemetry_config.getTelemetryChannelEndpoint)({
    appendServerlessChannelsSuffix,
    env: sendUsageTo,
    channelName: 'optInStatus'
  });
  const optInStatusPayload = await telemetryCollectionManager.getOptInStats(newOptInStatus, statsGetterConfig);
  await Promise.all(optInStatusPayload.map(async ({
    clusterUuid,
    stats
  }) => {
    return await (0, _nodeFetch.default)(optInStatusUrl, {
      method: 'post',
      body: typeof stats === 'string' ? stats : JSON.stringify(stats),
      headers: {
        'Content-Type': 'application/json',
        'X-Elastic-Stack-Version': currentKibanaVersion,
        'X-Elastic-Cluster-ID': clusterUuid,
        'X-Elastic-Content-Encoding': _constants.PAYLOAD_CONTENT_ENCODING
      }
    });
  }));
}
function registerTelemetryOptInStatsRoutes(router, telemetryCollectionManager) {
  router.versioned.post({
    access: 'public',
    // It's not used across Kibana, and I didn't want to remove it in this PR just in case.
    path: _routes.GetOptInStatsRoutePathBasedV2
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: {
        body: _configSchema.schema.object({
          enabled: _configSchema.schema.boolean(),
          unencrypted: _configSchema.schema.boolean({
            defaultValue: true
          })
        })
      },
      response: {
        200: {
          body: _configSchema.schema.arrayOf(_configSchema.schema.object({
            clusterUuid: _configSchema.schema.string(),
            stats: _configSchema.schema.object({
              cluster_uuid: _configSchema.schema.string(),
              opt_in_status: _configSchema.schema.boolean()
            })
          }))
        },
        503: {
          body: _configSchema.schema.string()
        }
      }
    }
  }, async (context, req, res) => {
    try {
      const newOptInStatus = req.body.enabled;
      const unencrypted = req.body.unencrypted;
      if (!(await telemetryCollectionManager.shouldGetTelemetry())) {
        // We probably won't reach here because there is a license check in the auth phase of the HTTP requests.
        // But let's keep it here should that changes at any point.
        return res.customError({
          statusCode: 503,
          body: `Can't fetch telemetry at the moment because some services are down. Check the /status page for more details.`
        });
      }
      const statsGetterConfig = {
        unencrypted
      };
      const optInStatus = await telemetryCollectionManager.getOptInStats(newOptInStatus, statsGetterConfig);
      const body = optInStatus;
      return res.ok({
        body
      });
    } catch (err) {
      return res.ok({
        body: []
      });
    }
  });
}