"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.scheduleCleanUpTask = exports.registerCleanUpTask = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _synthetics_monitor_client = require("../synthetics_monitor/synthetics_monitor_client");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SYNTHETICS_SERVICE_CLEAN_UP_TASK_TYPE = 'Synthetics:Clean-Up-Package-Policies';
const SYNTHETICS_SERVICE_CLEAN_UP_TASK_ID = 'SyntheticsService:clean-up-package-policies-task-id';
const SYNTHETICS_SERVICE_CLEAN_UP_INTERVAL_DEFAULT = '60m';
const DELETE_BROWSER_MINUTES = 15;
const DELETE_LIGHTWEIGHT_MINUTES = 2;
const registerCleanUpTask = (taskManager, serverSetup) => {
  const {
    logger
  } = serverSetup;
  const interval = SYNTHETICS_SERVICE_CLEAN_UP_INTERVAL_DEFAULT;
  taskManager.registerTaskDefinitions({
    [SYNTHETICS_SERVICE_CLEAN_UP_TASK_TYPE]: {
      title: 'Synthetics Plugin Clean Up Task',
      description: 'This task which runs periodically to clean up run once monitors.',
      timeout: '1m',
      maxAttempts: 3,
      createTaskRunner: ({
        taskInstance
      }) => {
        return {
          // Perform the work of the task. The return value should fit the TaskResult interface.
          async run() {
            logger.info(`Executing synthetics clean up task: ${SYNTHETICS_SERVICE_CLEAN_UP_TASK_ID}`);
            const {
              state
            } = taskInstance;
            try {
              var _serverSetup$coreStar, _serverSetup$coreStar2;
              const esClient = (_serverSetup$coreStar = serverSetup.coreStart) === null || _serverSetup$coreStar === void 0 ? void 0 : (_serverSetup$coreStar2 = _serverSetup$coreStar.elasticsearch) === null || _serverSetup$coreStar2 === void 0 ? void 0 : _serverSetup$coreStar2.client.asInternalUser;
              if (esClient) {
                const {
                  fleet
                } = serverSetup.pluginsStart;
                const {
                  savedObjects
                } = serverSetup.coreStart;
                const soClient = savedObjects.createInternalRepository();
                const {
                  items
                } = await fleet.packagePolicyService.list(soClient, {
                  kuery: getFilterForTestNowRun()
                });
                const allItems = items.map(item => {
                  const minutesAgo = (0, _moment.default)().diff((0, _moment.default)(item.created_at), 'minutes');
                  const isBrowser = item.name === _synthetics_monitor_client.BROWSER_TEST_NOW_RUN;
                  if (isBrowser) {
                    return {
                      isBrowser: true,
                      id: item.id,
                      shouldDelete: minutesAgo > DELETE_BROWSER_MINUTES
                    };
                  } else {
                    return {
                      isBrowser: false,
                      id: item.id,
                      shouldDelete: minutesAgo > DELETE_LIGHTWEIGHT_MINUTES
                    };
                  }
                });
                const toDelete = allItems.filter(item => item.shouldDelete);
                if (toDelete.length > 0) {
                  await fleet.packagePolicyService.delete(soClient, esClient, toDelete.map(item => item.id), {
                    force: true
                  });
                }
                const remaining = allItems.filter(item => !item.shouldDelete);
                if (remaining.length === 0) {
                  return {
                    state,
                    schedule: {
                      interval: '24h'
                    }
                  };
                } else {
                  return {
                    state,
                    schedule: {
                      interval: '20m'
                    }
                  };
                }
              }
            } catch (e) {
              logger.error(e);
            }
            return {
              state,
              schedule: {
                interval
              }
            };
          }
        };
      }
    }
  });
};
exports.registerCleanUpTask = registerCleanUpTask;
const scheduleCleanUpTask = async ({
  logger,
  pluginsStart
}) => {
  const interval = SYNTHETICS_SERVICE_CLEAN_UP_INTERVAL_DEFAULT;
  try {
    var _taskInstance$schedul;
    const taskInstance = await pluginsStart.taskManager.ensureScheduled({
      id: SYNTHETICS_SERVICE_CLEAN_UP_TASK_ID,
      taskType: SYNTHETICS_SERVICE_CLEAN_UP_TASK_TYPE,
      schedule: {
        interval
      },
      params: {},
      state: {},
      scope: ['uptime']
    });
    logger === null || logger === void 0 ? void 0 : logger.info(`Task ${SYNTHETICS_SERVICE_CLEAN_UP_TASK_ID} scheduled with interval ${(_taskInstance$schedul = taskInstance.schedule) === null || _taskInstance$schedul === void 0 ? void 0 : _taskInstance$schedul.interval}.`);
    await pluginsStart.taskManager.runSoon(SYNTHETICS_SERVICE_CLEAN_UP_TASK_ID);
  } catch (e) {
    logger === null || logger === void 0 ? void 0 : logger.error(e);
    logger === null || logger === void 0 ? void 0 : logger.error(`Error running synthetics clean up task: ${SYNTHETICS_SERVICE_CLEAN_UP_TASK_ID}, ${e === null || e === void 0 ? void 0 : e.message}`);
  }
};
exports.scheduleCleanUpTask = scheduleCleanUpTask;
const getFilterForTestNowRun = () => {
  const pkg = 'ingest-package-policies';
  let filter = `${pkg}.package.name:synthetics and ${pkg}.is_managed:true`;
  const lightweight = `${pkg}.name: ${_synthetics_monitor_client.LIGHTWEIGHT_TEST_NOW_RUN}`;
  const browser = `${pkg}.name: ${_synthetics_monitor_client.BROWSER_TEST_NOW_RUN}`;
  filter = `${filter} and (${lightweight} or ${browser})`;
  return filter;
};