"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateLocation = validateLocation;
exports.validateMonitor = validateMonitor;
exports.validateProjectMonitor = validateProjectMonitor;
var t = _interopRequireWildcard(require("io-ts"));
var _i18n = require("@kbn/i18n");
var _Either = require("fp-ts/lib/Either");
var _securitysolutionIoTsUtils = require("@kbn/securitysolution-io-ts-utils");
var _runtime_types = require("../../../common/runtime_types");
var _monitor_defaults = require("../../../common/constants/monitor_defaults");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const monitorTypeToCodecMap = {
  [_runtime_types.MonitorTypeEnum.ICMP]: _runtime_types.ICMPFieldsCodec,
  [_runtime_types.MonitorTypeEnum.TCP]: _runtime_types.TCPFieldsCodec,
  [_runtime_types.MonitorTypeEnum.HTTP]: _runtime_types.HTTPFieldsCodec,
  [_runtime_types.MonitorTypeEnum.BROWSER]: _runtime_types.BrowserFieldsCodec
};
/**
 * Validates monitor fields with respect to the relevant Codec identified by object's 'type' property.
 * @param monitorFields {MonitorFields} The mixed type representing the possible monitor payload from UI.
 */
function validateMonitor(monitorFields) {
  const {
    [_runtime_types.ConfigKey.MONITOR_TYPE]: monitorType
  } = monitorFields;
  const decodedType = _runtime_types.MonitorTypeCodec.decode(monitorType);
  if ((0, _Either.isLeft)(decodedType)) {
    return {
      valid: false,
      reason: INVALID_TYPE_ERROR,
      details: (0, _securitysolutionIoTsUtils.formatErrors)(decodedType.left).join(' | '),
      payload: monitorFields
    };
  }

  // Cast it to ICMPCodec to satisfy typing. During runtime, correct codec will be used to decode.
  const SyntheticsMonitorCodec = monitorTypeToCodecMap[monitorType];
  if (!SyntheticsMonitorCodec) {
    return {
      valid: false,
      reason: INVALID_PAYLOAD_ERROR,
      details: '',
      payload: monitorFields
    };
  }
  if (!_monitor_defaults.ALLOWED_SCHEDULES_IN_MINUTES.includes(monitorFields[_runtime_types.ConfigKey.SCHEDULE].number)) {
    return {
      valid: false,
      reason: INVALID_SCHEDULE_ERROR,
      details: INVALID_SCHEDULE_DETAILS(monitorFields[_runtime_types.ConfigKey.SCHEDULE].number),
      payload: monitorFields
    };
  }
  const ExactSyntheticsMonitorCodec = t.exact(SyntheticsMonitorCodec);
  const decodedMonitor = ExactSyntheticsMonitorCodec.decode(monitorFields);
  if ((0, _Either.isLeft)(decodedMonitor)) {
    return {
      valid: false,
      reason: INVALID_SCHEMA_ERROR(monitorType),
      details: (0, _securitysolutionIoTsUtils.formatErrors)(decodedMonitor.left).join(' | '),
      payload: monitorFields
    };
  }
  return {
    valid: true,
    reason: '',
    details: '',
    payload: monitorFields,
    decodedMonitor: decodedMonitor.right
  };
}
function validateProjectMonitor(monitorFields, publicLocations, privateLocations) {
  const locationsError = validateLocation(monitorFields, publicLocations, privateLocations);
  // Cast it to ICMPCodec to satisfy typing. During runtime, correct codec will be used to decode.
  const decodedMonitor = _runtime_types.ProjectMonitorCodec.decode(monitorFields);
  if ((0, _Either.isLeft)(decodedMonitor)) {
    return {
      valid: false,
      reason: INVALID_CONFIGURATION_ERROR,
      details: [...(0, _securitysolutionIoTsUtils.formatErrors)(decodedMonitor.left), locationsError].filter(error => error !== '' && error !== undefined).join(' | '),
      payload: monitorFields
    };
  }
  if (locationsError) {
    return {
      valid: false,
      reason: INVALID_CONFIGURATION_ERROR,
      details: locationsError,
      payload: monitorFields
    };
  }
  return {
    valid: true,
    reason: '',
    details: '',
    payload: monitorFields
  };
}
function validateLocation(monitorFields, publicLocations, privateLocations) {
  var _monitorFields$privat2;
  const hasPublicLocationsConfigured = (monitorFields.locations || []).length > 0;
  const hasPrivateLocationsConfigured = (monitorFields.privateLocations || []).length > 0;
  if (hasPublicLocationsConfigured) {
    var _monitorFields$locati;
    let invalidLocation = '';
    const hasValidPublicLocation = (_monitorFields$locati = monitorFields.locations) === null || _monitorFields$locati === void 0 ? void 0 : _monitorFields$locati.some(location => {
      if (publicLocations.length === 0) {
        invalidLocation = location;
        return false;
      }
      return publicLocations.some(supportedLocation => {
        const locationIsValid = supportedLocation.id === location;
        if (!locationIsValid) {
          invalidLocation = location;
        }
        return locationIsValid;
      });
    });
    if (!hasValidPublicLocation) {
      return INVALID_PUBLIC_LOCATION_ERROR(invalidLocation);
    }
  }
  if (hasPrivateLocationsConfigured) {
    var _monitorFields$privat;
    let invalidLocation = '';
    const hasValidPrivateLocation = (_monitorFields$privat = monitorFields.privateLocations) === null || _monitorFields$privat === void 0 ? void 0 : _monitorFields$privat.some(location => {
      if (privateLocations.length === 0) {
        invalidLocation = location;
        return false;
      }
      return privateLocations.some(supportedLocation => {
        const locationIsValid = supportedLocation.label === location;
        if (!locationIsValid) {
          invalidLocation = location;
        }
        return locationIsValid;
      });
    });
    if (!hasValidPrivateLocation) {
      return INVALID_PRIVATE_LOCATION_ERROR(invalidLocation);
    }
  }
  const hasEmptyLocations = monitorFields.locations && monitorFields.locations.length === 0 && ((_monitorFields$privat2 = monitorFields.privateLocations) !== null && _monitorFields$privat2 !== void 0 ? _monitorFields$privat2 : []).length === 0;
  if (hasEmptyLocations) {
    return EMPTY_LOCATION_ERROR;
  }
}
const INVALID_CONFIGURATION_ERROR = _i18n.i18n.translate('xpack.synthetics.server.monitors.invalidConfigurationError', {
  defaultMessage: "Couldn't save or update monitor because of an invalid configuration."
});
const INVALID_PAYLOAD_ERROR = _i18n.i18n.translate('xpack.synthetics.server.monitors.invalidPayloadError', {
  defaultMessage: 'Payload is not a valid monitor object'
});
const INVALID_TYPE_ERROR = _i18n.i18n.translate('xpack.synthetics.server.monitors.invalidTypeError', {
  defaultMessage: 'Monitor type is invalid'
});
const INVALID_SCHEDULE_ERROR = _i18n.i18n.translate('xpack.synthetics.server.monitors.invalidScheduleError', {
  defaultMessage: 'Monitor schedule is invalid'
});
const INVALID_SCHEDULE_DETAILS = schedule => _i18n.i18n.translate('xpack.synthetics.server.monitors.invalidScheduleDetails', {
  defaultMessage: 'Invalid schedule {schedule} minutes supplied to monitor configuration. Please use a supported monitor schedule.',
  values: {
    schedule
  }
});
const INVALID_SCHEMA_ERROR = type => _i18n.i18n.translate('xpack.synthetics.server.monitors.invalidSchemaError', {
  defaultMessage: 'Monitor is not a valid monitor of type {type}',
  values: {
    type
  }
});
const EMPTY_LOCATION_ERROR = _i18n.i18n.translate('xpack.synthetics.server.projectMonitors.locationEmptyError', {
  defaultMessage: 'You must add at least one location or private location to this monitor.'
});
const INVALID_PRIVATE_LOCATION_ERROR = location => _i18n.i18n.translate('xpack.synthetics.server.projectMonitors.invalidPrivateLocationError', {
  defaultMessage: 'Invalid private location: "{location}". Remove it or replace it with a valid private location.',
  values: {
    location
  }
});
const INVALID_PUBLIC_LOCATION_ERROR = location => _i18n.i18n.translate('xpack.synthetics.server.projectMonitors.invalidPublicLocationError', {
  defaultMessage: 'Invalid location: "{location}". Remove it or replace it with a valid location.',
  values: {
    location
  }
});