"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validate = exports.validJSONFormat = exports.WHOLE_NUMBERS_ONLY = exports.INCLUDES_VALID_PORT = exports.FLOATS_ONLY = exports.DIGITS_ONLY = void 0;
exports.validateHeaders = validateHeaders;
exports.validateTimeout = void 0;
var _types = require("../types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DIGITS_ONLY = /^[0-9]*$/g;
exports.DIGITS_ONLY = DIGITS_ONLY;
const INCLUDES_VALID_PORT = /[^\:]+:[0-9]{1,5}$/g;
exports.INCLUDES_VALID_PORT = INCLUDES_VALID_PORT;
const WHOLE_NUMBERS_ONLY = /^[0-9]+(.?[0]+)?$/;
exports.WHOLE_NUMBERS_ONLY = WHOLE_NUMBERS_ONLY;
const FLOATS_ONLY = /^[0-9]+(.?[0-9]+)?$/;
exports.FLOATS_ONLY = FLOATS_ONLY;
// returns true if invalid
function validateHeaders(headers) {
  return Object.keys(headers).some(key => {
    if (key) {
      const whiteSpaceRegEx = /[\s]/g;
      return whiteSpaceRegEx.test(key);
    } else {
      return false;
    }
  });
}

// returns true if invalid
const validateTimeout = ({
  scheduleNumber,
  scheduleUnit,
  timeout
}) => {
  let schedule;
  switch (scheduleUnit) {
    case _types.ScheduleUnit.SECONDS:
      schedule = parseFloat(scheduleNumber);
      break;
    case _types.ScheduleUnit.MINUTES:
      schedule = parseFloat(scheduleNumber) * 60;
      break;
    default:
      schedule = parseFloat(scheduleNumber);
  }
  return parseFloat(timeout) > schedule;
};
exports.validateTimeout = validateTimeout;
const validJSONFormat = value => {
  let obj;
  try {
    obj = JSON.parse(value);
    if (!obj || typeof obj !== 'object') {
      return false;
    }
  } catch (e) {
    return false;
  }
  return true;
};

// validation functions return true when invalid
exports.validJSONFormat = validJSONFormat;
const validateCommon = {
  [_types.ConfigKey.SCHEDULE]: ({
    [_types.ConfigKey.SCHEDULE]: value
  }) => {
    const {
      number,
      unit
    } = value;
    const parsedFloat = parseFloat(number);
    return !parsedFloat || !unit || parsedFloat < 1;
  },
  [_types.ConfigKey.TIMEOUT]: ({
    [_types.ConfigKey.MONITOR_TYPE]: monitorType,
    [_types.ConfigKey.TIMEOUT]: timeout,
    [_types.ConfigKey.SCHEDULE]: schedule
  }) => {
    const {
      number,
      unit
    } = schedule;

    // Timeout is not currently supported by browser monitors
    if (monitorType === _types.MonitorTypeEnum.BROWSER) {
      return false;
    }
    return !timeout || parseFloat(timeout) < 0 || validateTimeout({
      timeout,
      scheduleNumber: number,
      scheduleUnit: unit
    });
  }
};
const validateHTTP = {
  [_types.ConfigKey.RESPONSE_STATUS_CHECK]: ({
    [_types.ConfigKey.RESPONSE_STATUS_CHECK]: value
  }) => {
    const statusCodes = value;
    return statusCodes.length ? statusCodes.some(code => !`${code}`.match(DIGITS_ONLY)) : false;
  },
  [_types.ConfigKey.RESPONSE_HEADERS_CHECK]: ({
    [_types.ConfigKey.RESPONSE_HEADERS_CHECK]: value
  }) => {
    const headers = value;
    return validateHeaders(headers);
  },
  [_types.ConfigKey.REQUEST_HEADERS_CHECK]: ({
    [_types.ConfigKey.REQUEST_HEADERS_CHECK]: value
  }) => {
    const headers = value;
    return validateHeaders(headers);
  },
  [_types.ConfigKey.MAX_REDIRECTS]: ({
    [_types.ConfigKey.MAX_REDIRECTS]: value
  }) => !!value && !`${value}`.match(DIGITS_ONLY) || parseFloat(value) < 0,
  [_types.ConfigKey.URLS]: ({
    [_types.ConfigKey.URLS]: value
  }) => !value,
  ...validateCommon
};
const validateTCP = {
  [_types.ConfigKey.HOSTS]: ({
    [_types.ConfigKey.HOSTS]: value
  }) => {
    return !value || !`${value}`.match(INCLUDES_VALID_PORT);
  },
  ...validateCommon
};
const validateICMP = {
  [_types.ConfigKey.HOSTS]: ({
    [_types.ConfigKey.HOSTS]: value
  }) => !value,
  [_types.ConfigKey.WAIT]: ({
    [_types.ConfigKey.WAIT]: value
  }) => !!value && !DIGITS_ONLY.test(`${value}`) && parseFloat(value) < 0,
  ...validateCommon
};
const validateThrottleValue = (speed, allowZero) => {
  if (speed === undefined || speed === '') return false;
  const throttleValue = parseFloat(speed);
  return isNaN(throttleValue) || (allowZero ? throttleValue < 0 : throttleValue <= 0);
};
const validateBrowser = {
  ...validateCommon,
  [_types.ConfigKey.SOURCE_INLINE]: ({
    [_types.ConfigKey.SOURCE_INLINE]: inlineScript
  }) => !inlineScript,
  [_types.ConfigKey.THROTTLING_CONFIG]: ({
    throttling
  }) => {
    if (!throttling || throttling.value === null) return true;
    const {
      download,
      upload,
      latency
    } = throttling.value;
    return validateThrottleValue(String(download)) || validateThrottleValue(String(upload)) || validateThrottleValue(String(latency), true);
  },
  [_types.ConfigKey.PLAYWRIGHT_OPTIONS]: ({
    [_types.ConfigKey.PLAYWRIGHT_OPTIONS]: playwrightOptions
  }) => playwrightOptions ? !validJSONFormat(playwrightOptions) : false,
  [_types.ConfigKey.PARAMS]: ({
    [_types.ConfigKey.PARAMS]: params
  }) => params ? !validJSONFormat(params) : false
};
const validate = {
  [_types.MonitorTypeEnum.HTTP]: validateHTTP,
  [_types.MonitorTypeEnum.TCP]: validateTCP,
  [_types.MonitorTypeEnum.ICMP]: validateICMP,
  [_types.MonitorTypeEnum.BROWSER]: validateBrowser
};
exports.validate = validate;