"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createExternalService = void 0;
var _axios = _interopRequireDefault(require("axios"));
var _i18n = require("@kbn/i18n");
var _axios_utils = require("@kbn/actions-plugin/server/lib/axios_utils");
var _pipeable = require("fp-ts/lib/pipeable");
var _Option = require("fp-ts/lib/Option");
var _translations = require("./translations");
var _lib = require("../../../common/slack_api/lib");
var _constants = require("../../../common/slack_api/constants");
var _http_response_retry_header = require("../lib/http_response_retry_header");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const buildSlackExecutorErrorResponse = ({
  slackApiError,
  logger
}) => {
  if (!slackApiError.response) {
    return (0, _lib.serviceErrorResult)(_constants.SLACK_API_CONNECTOR_ID, slackApiError.message);
  }
  const {
    status,
    statusText,
    headers
  } = slackApiError.response;

  // special handling for 5xx
  if (status >= 500) {
    return (0, _lib.retryResult)(_constants.SLACK_API_CONNECTOR_ID, slackApiError.message);
  }

  // special handling for rate limiting
  if (status === 429) {
    return (0, _pipeable.pipe)((0, _http_response_retry_header.getRetryAfterIntervalFromHeaders)(headers), (0, _Option.map)(retry => (0, _lib.retryResultSeconds)(_constants.SLACK_API_CONNECTOR_ID, slackApiError.message, retry)), (0, _Option.getOrElse)(() => (0, _lib.retryResult)(_constants.SLACK_API_CONNECTOR_ID, slackApiError.message)));
  }
  const errorMessage = _i18n.i18n.translate('xpack.stackConnectors.slack.unexpectedHttpResponseErrorMessage', {
    defaultMessage: 'unexpected http response from slack: {httpStatus} {httpStatusText}',
    values: {
      httpStatus: status,
      httpStatusText: statusText
    }
  });
  logger.error(`error on ${_constants.SLACK_API_CONNECTOR_ID} slack action: ${errorMessage}`);
  return (0, _lib.errorResult)(_constants.SLACK_API_CONNECTOR_ID, errorMessage);
};
const buildSlackExecutorSuccessResponse = ({
  slackApiResponseData
}) => {
  if (!slackApiResponseData) {
    const errMessage = _i18n.i18n.translate('xpack.stackConnectors.slack.unexpectedNullResponseErrorMessage', {
      defaultMessage: 'unexpected null response from slack'
    });
    return (0, _lib.errorResult)(_constants.SLACK_API_CONNECTOR_ID, errMessage);
  }
  if (!slackApiResponseData.ok) {
    return (0, _lib.serviceErrorResult)(_constants.SLACK_API_CONNECTOR_ID, slackApiResponseData.error);
  }
  return (0, _lib.successResult)(_constants.SLACK_API_CONNECTOR_ID, slackApiResponseData);
};
const createExternalService = ({
  config,
  secrets
}, logger, configurationUtilities) => {
  const {
    token
  } = secrets;
  const {
    allowedChannels
  } = config || {
    allowedChannels: []
  };
  const allowedChannelIds = allowedChannels === null || allowedChannels === void 0 ? void 0 : allowedChannels.map(ac => ac.id);
  if (!token) {
    throw Error(`[Action][${_translations.SLACK_CONNECTOR_NAME}]: Wrong configuration.`);
  }
  const axiosInstance = _axios.default.create();
  const headers = {
    Authorization: `Bearer ${token}`,
    'Content-type': 'application/json; charset=UTF-8'
  };
  const validChannelId = async channelId => {
    try {
      const validChannel = () => {
        return (0, _axios_utils.request)({
          axios: axiosInstance,
          configurationUtilities,
          logger,
          method: 'get',
          headers,
          url: `${_constants.SLACK_URL}conversations.info?channel=${channelId}`
        });
      };
      if (channelId.length === 0) {
        return buildSlackExecutorErrorResponse({
          slackApiError: new Error('The channel id is empty'),
          logger
        });
      }
      const result = await validChannel();
      return buildSlackExecutorSuccessResponse({
        slackApiResponseData: result.data
      });
    } catch (error) {
      return buildSlackExecutorErrorResponse({
        slackApiError: error,
        logger
      });
    }
  };
  const postMessage = async ({
    channels,
    channelIds = [],
    text
  }) => {
    try {
      if (channelIds.length > 0 && allowedChannelIds && allowedChannelIds.length > 0 && !channelIds.every(cId => allowedChannelIds.includes(cId))) {
        return buildSlackExecutorErrorResponse({
          slackApiError: {
            message: `One of channel ids "${channelIds.join()}" is not included in the allowed channels list "${allowedChannelIds.join()}"`
          },
          logger
        });
      }

      // For now, we only allow one channel but we wanted
      // to have a array in case we need to allow multiple channels
      // in one actions
      let channelToUse = channelIds.length > 0 ? channelIds[0] : '';
      if (channelToUse.length === 0 && channels && channels.length > 0 && channels[0].length > 0) {
        channelToUse = channels[0];
      }
      if (channelToUse.length === 0) {
        return buildSlackExecutorErrorResponse({
          slackApiError: {
            message: `The channel is empty"`
          },
          logger
        });
      }
      const result = await (0, _axios_utils.request)({
        axios: axiosInstance,
        method: 'post',
        url: `${_constants.SLACK_URL}chat.postMessage`,
        logger,
        data: {
          channel: channelToUse,
          text
        },
        headers,
        configurationUtilities
      });
      return buildSlackExecutorSuccessResponse({
        slackApiResponseData: result.data
      });
    } catch (error) {
      return buildSlackExecutorErrorResponse({
        slackApiError: error,
        logger
      });
    }
  };
  return {
    validChannelId,
    postMessage
  };
};
exports.createExternalService = createExternalService;