"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ShareToSpaceFlyoutInternal = void 0;
require("./share_to_space_flyout_internal.scss");
var _eui = require("@elastic/eui");
var _react = _interopRequireWildcard(require("react"));
var _i18n = require("@kbn/i18n");
var _i18nReact = require("@kbn/i18n-react");
var _alias_table = require("./alias_table");
var _relatives_footer = require("./relatives_footer");
var _share_to_space_form = require("./share_to_space_form");
var _constants = require("../../../common/constants");
var _constants2 = require("../../constants");
var _copy_saved_objects_to_space = require("../../copy_saved_objects_to_space");
var _spaces_context = require("../../spaces_context");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// No need to wrap LazyCopyToSpaceFlyout in an error boundary, because the ShareToSpaceFlyoutInternal component itself is only ever used in
// a lazy-loaded fashion with an error boundary.
const LazyCopyToSpaceFlyout = /*#__PURE__*/(0, _react.lazy)(() => (0, _copy_saved_objects_to_space.getCopyToSpaceFlyoutComponent)().then(component => ({
  default: component
})));
const ALL_SPACES_TARGET = _i18n.i18n.translate('xpack.spaces.shareToSpace.allSpacesTarget', {
  defaultMessage: 'all spaces'
});
function getSpacesTargetString(spaces) {
  if (spaces.includes(_constants.ALL_SPACES_ID)) {
    return ALL_SPACES_TARGET;
  }
  return _i18n.i18n.translate('xpack.spaces.shareToSpace.spacesTarget', {
    defaultMessage: '{spacesCount, plural, one {# space} other {# spaces}}',
    values: {
      spacesCount: spaces.length
    }
  });
}
const arraysAreEqual = (a, b) => a.every(x => b.includes(x)) && b.every(x => a.includes(x));
function createDefaultChangeSpacesHandler(object, spacesManager, toastNotifications) {
  return async (objects, spacesToAdd, spacesToRemove) => {
    const {
      title
    } = object;
    const objectsToUpdate = objects.map(({
      type,
      id
    }) => ({
      type,
      id
    })); // only use 'type' and 'id' fields
    const relativesCount = objects.length - 1;
    const toastTitle = _i18n.i18n.translate('xpack.spaces.shareToSpace.shareSuccessTitle', {
      values: {
        objectNoun: object.noun
      },
      defaultMessage: 'Updated {objectNoun}',
      description: `Object noun can be plural or singular, examples: "Updated objects", "Updated job"`
    });

    // If removing spaces and there are referenced objects ("related objects" in UI),
    // only remove spaces from the target object.
    if (spacesToRemove.length > 0 && objectsToUpdate.length > 1) {
      const indexOfTarget = objectsToUpdate.findIndex(element => element.id === object.id);
      if (indexOfTarget >= 0) {
        objectsToUpdate.splice(indexOfTarget, 1);
      }
      const updateTarget = spacesManager.updateSavedObjectsSpaces([{
        type: object.type,
        id: object.id
      }], spacesToAdd, spacesToRemove);

      // Only if there are also spaces being added, affect any referenced/related objects
      const updateRelated = spacesToAdd.length > 0 ? spacesManager.updateSavedObjectsSpaces(objectsToUpdate, spacesToAdd, []) : undefined;
      await Promise.all([updateTarget, updateRelated]);
    } else {
      await spacesManager.updateSavedObjectsSpaces(objectsToUpdate, spacesToAdd, spacesToRemove);
    }
    const isSharedToAllSpaces = spacesToAdd.includes(_constants.ALL_SPACES_ID);
    let toastText;
    if (spacesToAdd.length > 0 && spacesToRemove.length > 0 && !isSharedToAllSpaces) {
      toastText = _i18n.i18n.translate('xpack.spaces.shareToSpace.shareSuccessAddRemoveText', {
        defaultMessage: `'{object}' {relativesCount, plural, =0 {was} =1 {and {relativesCount} related object were} other {and {relativesCount} related objects were}} added to {spacesTargetAdd}. '{object}' was removed from {spacesTargetRemove}.`,
        values: {
          object: title,
          relativesCount,
          spacesTargetAdd: getSpacesTargetString(spacesToAdd),
          spacesTargetRemove: getSpacesTargetString(spacesToRemove)
        },
        description: `Uses output of xpack.spaces.shareToSpace.spacesTarget or xpack.spaces.shareToSpace.allSpacesTarget as 'spacesTarget...' inputs. Example strings: "'Finance dashboard' was added to 1 space. 'Finance dashboard' was removed from 2 spaces.", "'Finance dashboard' and 2 related objects were added to 3 spaces. 'Finance dashboard' was removed from all spaces."`
      });
    } else if (spacesToAdd.length > 0) {
      toastText = _i18n.i18n.translate('xpack.spaces.shareToSpace.shareSuccessAddText', {
        defaultMessage: `'{object}' {relativesCount, plural, =0 {was} =1 {and {relativesCount} related object were} other {and {relativesCount} related objects were}} added to {spacesTarget}.`,
        values: {
          object: title,
          relativesCount,
          spacesTarget: getSpacesTargetString(spacesToAdd)
        },
        description: `Uses output of xpack.spaces.shareToSpace.spacesTarget or xpack.spaces.shareToSpace.allSpacesTarget as 'spacesTarget' input. Example strings: "'Finance dashboard' was added to 1 space.", "'Finance dashboard' and 2 related objects were added to all spaces."`
      });
    } else {
      toastText = _i18n.i18n.translate('xpack.spaces.shareToSpace.shareSuccessRemoveText', {
        defaultMessage: `'{object}' was removed from {spacesTarget}.`,
        values: {
          object: title,
          spacesTarget: getSpacesTargetString(spacesToRemove)
        },
        description: `Uses output of xpack.spaces.shareToSpace.spacesTarget or xpack.spaces.shareToSpace.allSpacesTarget as 'spacesTarget' input. Example string: "'Finance dashboard' was removed from 1 space.", "'Finance dashboard' was removed from all spaces."`
      });
    }
    toastNotifications.addSuccess({
      title: toastTitle,
      text: toastText
    });
  };
}
const ShareToSpaceFlyoutInternal = props => {
  const {
    spacesManager,
    spacesDataPromise,
    services
  } = (0, _spaces_context.useSpaces)();
  const {
    notifications
  } = services;
  const toastNotifications = notifications.toasts;
  const {
    savedObjectTarget: object
  } = props;
  const savedObjectTarget = (0, _react.useMemo)(() => ({
    type: object.type,
    id: object.id,
    namespaces: object.namespaces,
    icon: object.icon,
    title: object.title || `${object.type} [id=${object.id}]`,
    noun: object.noun || _constants2.DEFAULT_OBJECT_NOUN
  }), [object]);
  const {
    flyoutIcon,
    flyoutTitle = _i18n.i18n.translate('xpack.spaces.shareToSpace.flyoutTitle', {
      defaultMessage: 'Share {objectNoun} to spaces',
      values: {
        objectNoun: savedObjectTarget.noun
      }
    }),
    enableCreateCopyCallout = false,
    enableCreateNewSpaceLink = false,
    behaviorContext,
    changeSpacesHandler = createDefaultChangeSpacesHandler(savedObjectTarget, spacesManager, toastNotifications),
    onUpdate = () => null,
    onClose = () => null
  } = props;
  const enableSpaceAgnosticBehavior = behaviorContext === 'outside-space';
  const [shareOptions, setShareOptions] = (0, _react.useState)({
    selectedSpaceIds: [],
    initiallySelectedSpaceIds: []
  });
  const [canShareToAllSpaces, setCanShareToAllSpaces] = (0, _react.useState)(false);
  const [showMakeCopy, setShowMakeCopy] = (0, _react.useState)(false);
  const [{
    isLoading,
    spaces,
    referenceGraph,
    aliasTargets,
    prohibitedSpaces
  }, setSpacesState] = (0, _react.useState)({
    isLoading: true,
    spaces: [],
    referenceGraph: [],
    aliasTargets: [],
    prohibitedSpaces: new Set()
  });
  (0, _react.useEffect)(() => {
    const {
      type,
      id
    } = savedObjectTarget;
    const getShareableReferences = spacesManager.getShareableReferences([{
      type,
      id
    }]);
    const getPermissions = spacesManager.getShareSavedObjectPermissions(type);
    Promise.all([spacesDataPromise, getShareableReferences, getPermissions]).then(([spacesData, shareableReferences, permissions]) => {
      const activeSpaceId = !enableSpaceAgnosticBehavior && spacesData.activeSpaceId;
      const selectedSpaceIds = savedObjectTarget.namespaces.filter(spaceId => spaceId !== activeSpaceId);
      setShareOptions({
        selectedSpaceIds,
        initiallySelectedSpaceIds: selectedSpaceIds
      });
      setCanShareToAllSpaces(permissions.shareToAllSpaces);
      setSpacesState({
        isLoading: false,
        spaces: [...spacesData.spacesMap].map(([, spaceTarget]) => spaceTarget),
        referenceGraph: shareableReferences.objects,
        aliasTargets: shareableReferences.objects.reduce((acc, x) => {
          for (const space of (_x$spacesWithMatching = x.spacesWithMatchingAliases) !== null && _x$spacesWithMatching !== void 0 ? _x$spacesWithMatching : []) {
            var _x$spacesWithMatching;
            if (space !== _constants.UNKNOWN_SPACE) {
              const spaceExists = spacesData.spacesMap.has(space);
              // If the user does not have privileges to view all spaces, they will be redacted; we cannot attempt to disable aliases for redacted spaces.
              acc.push({
                targetSpace: space,
                targetType: x.type,
                sourceId: x.id,
                spaceExists
              });
            }
          }
          return acc;
        }, []),
        prohibitedSpaces: shareableReferences.objects.reduce((acc, x) => {
          // Whenever we detect that a space contains an object with a matching origin, *and* the list of currently selected spaces does
          // not include it, then it is prohibited. That means the user cannot share the object to those spaces.
          for (const space of (_x$spacesWithMatching2 = x.spacesWithMatchingOrigins) !== null && _x$spacesWithMatching2 !== void 0 ? _x$spacesWithMatching2 : []) {
            var _x$spacesWithMatching2;
            if (space !== _constants.UNKNOWN_SPACE && !selectedSpaceIds.includes(space) && space !== activeSpaceId) {
              acc.add(space);
            }
          }
          return acc;
        }, new Set())
      });
    }).catch(e => {
      toastNotifications.addError(e, {
        title: _i18n.i18n.translate('xpack.spaces.shareToSpace.spacesLoadErrorTitle', {
          defaultMessage: 'Error loading available spaces'
        })
      });
    });
  }, [savedObjectTarget, spacesManager, spacesDataPromise, toastNotifications, enableSpaceAgnosticBehavior]);
  const getSelectionChanges = () => {
    if (!spaces.length) {
      return {
        isSelectionChanged: false,
        spacesToAdd: [],
        spacesToRemove: [],
        aliasesToDisable: []
      };
    }
    const activeSpaceId = !enableSpaceAgnosticBehavior && spaces.find(space => space.isActiveSpace).id;
    const initialSelection = savedObjectTarget.namespaces.filter(spaceId => spaceId !== activeSpaceId && spaceId !== _constants.UNKNOWN_SPACE);
    const {
      selectedSpaceIds
    } = shareOptions;
    const filteredSelection = selectedSpaceIds.filter(x => x !== _constants.UNKNOWN_SPACE);
    const initiallySharedToAllSpaces = initialSelection.includes(_constants.ALL_SPACES_ID);
    const selectionIncludesAllSpaces = filteredSelection.includes(_constants.ALL_SPACES_ID);
    const isSharedToAllSpaces = !initiallySharedToAllSpaces && selectionIncludesAllSpaces;
    const isUnsharedFromAllSpaces = initiallySharedToAllSpaces && !selectionIncludesAllSpaces;
    const selectedSpacesChanged = !selectionIncludesAllSpaces && !arraysAreEqual(initialSelection, filteredSelection);
    const isSelectionChanged = isSharedToAllSpaces || isUnsharedFromAllSpaces || !isSharedToAllSpaces && !isUnsharedFromAllSpaces && selectedSpacesChanged;
    const selectedSpacesToAdd = filteredSelection.filter(spaceId => !initialSelection.includes(spaceId));
    const selectedSpacesToRemove = initialSelection.filter(spaceId => !filteredSelection.includes(spaceId));
    const activeSpaceArray = activeSpaceId ? [activeSpaceId] : []; // if we have an active space, it is automatically selected
    const spacesToAdd = isSharedToAllSpaces ? [_constants.ALL_SPACES_ID] : isUnsharedFromAllSpaces ? [...activeSpaceArray, ...selectedSpacesToAdd] : selectedSpacesToAdd;
    const spacesToAddSet = new Set(spacesToAdd);
    const spacesToRemove = isUnsharedFromAllSpaces || !isSharedToAllSpaces ? selectedSpacesToRemove : [...activeSpaceArray, ...initialSelection];
    const aliasesToDisable = isSharedToAllSpaces ? aliasTargets : aliasTargets.filter(({
      targetSpace
    }) => spacesToAddSet.has(targetSpace));
    return {
      isSelectionChanged,
      spacesToAdd,
      spacesToRemove,
      aliasesToDisable
    };
  };
  const {
    isSelectionChanged,
    spacesToAdd,
    spacesToRemove,
    aliasesToDisable
  } = getSelectionChanges();
  const [showAliasesToDisable, setShowAliasesToDisable] = (0, _react.useState)(false);
  const [shareInProgress, setShareInProgress] = (0, _react.useState)(false);
  async function startShare() {
    setShareInProgress(true);
    try {
      if (aliasesToDisable.length) {
        const aliases = aliasesToDisable.map(({
          spaceExists,
          ...alias
        }) => alias); // only use 'targetSpace', 'targetType', and 'sourceId' fields
        await spacesManager.disableLegacyUrlAliases(aliases);
      }
      await changeSpacesHandler(referenceGraph, spacesToAdd, spacesToRemove);
      const updatedObjects = referenceGraph.map(({
        type,
        id
      }) => ({
        type,
        id
      })); // only use 'type' and 'id' fields
      onUpdate(updatedObjects);
      onClose();
    } catch (e) {
      setShareInProgress(false);
      toastNotifications.addError(e, {
        title: _i18n.i18n.translate('xpack.spaces.shareToSpace.shareErrorTitle', {
          values: {
            objectNoun: savedObjectTarget.noun
          },
          defaultMessage: 'Error updating {objectNoun}',
          description: `Object noun can be plural or singular, examples: "Failed to update objects", "Failed to update job"`
        }),
        toastMessage: _i18n.i18n.translate('xpack.spaces.shareToSpace.shareErrorText', {
          defaultMessage: `Unable to update '{object}' {relativesCount, plural, =0 {} =1 {or {relativesCount} related object} other {or one or more of {relativesCount} related objects}}.`,
          values: {
            object: savedObjectTarget.title,
            relativesCount: spacesToAdd.length > 0 ? referenceGraph.length - 1 : 0
          },
          description: `Uses output of xpack.spaces.shareToSpace.spacesTarget or xpack.spaces.shareToSpace.allSpacesTarget as 'spacesTarget...' inputs. Example strings: "'Finance dashboard' was added to 1 space. 'Finance dashboard' was removed from 2 spaces.", "'Finance dashboard' and 2 related objects were added to 3 spaces. 'Finance dashboard' was removed from all spaces."`
        })
      });
    }
  }
  const getFlyoutBody = () => {
    // Step 1: loading assets for main form
    if (isLoading) {
      return /*#__PURE__*/_react.default.createElement(_eui.EuiLoadingSpinner, null);
    }
    if (!showAliasesToDisable) {
      // If the object has not been shared yet (e.g., it currently exists in exactly one space), and there is at least one space that we could
      // share this object to, we want to display a callout to the user that explains the ramifications of shared objects. They might actually
      // want to make a copy instead, so this callout contains a link that opens the Copy flyout.
      const showCreateCopyCallout = enableCreateCopyCallout && spaces.length > 1 && savedObjectTarget.namespaces.length === 1 && !arraysAreEqual(savedObjectTarget.namespaces, [_constants.ALL_SPACES_ID]);
      // Step 2: Share has not been initiated yet; User must fill out form to continue.
      return /*#__PURE__*/_react.default.createElement(_share_to_space_form.ShareToSpaceForm, {
        spaces: spaces,
        objectNoun: savedObjectTarget.noun,
        shareOptions: shareOptions,
        onUpdate: setShareOptions,
        showCreateCopyCallout: showCreateCopyCallout,
        canShareToAllSpaces: canShareToAllSpaces,
        makeCopy: () => setShowMakeCopy(true),
        enableCreateNewSpaceLink: enableCreateNewSpaceLink,
        enableSpaceAgnosticBehavior: enableSpaceAgnosticBehavior,
        prohibitedSpaces: prohibitedSpaces
      });
    }
    return /*#__PURE__*/_react.default.createElement(_alias_table.AliasTable, {
      spaces: spaces,
      aliasesToDisable: aliasesToDisable
    });
  };
  const getFlyoutFooter = () => {
    const filteredAliasesToDisable = aliasesToDisable.filter(({
      spaceExists
    }) => spaceExists);
    const showContinueButton = filteredAliasesToDisable.length && !showAliasesToDisable;
    return /*#__PURE__*/_react.default.createElement(_react.default.Fragment, null, /*#__PURE__*/_react.default.createElement(_relatives_footer.RelativesFooter, {
      savedObjectTarget: savedObjectTarget,
      referenceGraph: referenceGraph,
      isDisabled: isStartShareButtonDisabled
    }), /*#__PURE__*/_react.default.createElement(_eui.EuiFlexGroup, {
      justifyContent: "spaceBetween",
      responsive: false
    }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
      grow: false
    }, /*#__PURE__*/_react.default.createElement(_eui.EuiButtonEmpty, {
      onClick: () => onClose(),
      "data-test-subj": "sts-cancel-button",
      disabled: shareInProgress
    }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.spaces.shareToSpace.cancelButton",
      defaultMessage: "Cancel"
    }))), /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
      grow: false
    }, showContinueButton ? /*#__PURE__*/_react.default.createElement(_eui.EuiButton, {
      fill: true,
      onClick: () => setShowAliasesToDisable(true),
      "data-test-subj": "sts-continue-button",
      disabled: isStartShareButtonDisabled
    }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.spaces.shareToSpace.continueButton",
      defaultMessage: "Continue"
    })) : /*#__PURE__*/_react.default.createElement(_eui.EuiButton, {
      fill: true,
      onClick: () => startShare(),
      "data-test-subj": "sts-save-button",
      disabled: isStartShareButtonDisabled
    }, /*#__PURE__*/_react.default.createElement(_i18nReact.FormattedMessage, {
      id: "xpack.spaces.shareToSpace.saveButton",
      defaultMessage: "Save & close"
    })))));
  };
  if (showMakeCopy) {
    return /*#__PURE__*/_react.default.createElement(_react.Suspense, {
      fallback: /*#__PURE__*/_react.default.createElement(_eui.EuiLoadingSpinner, null)
    }, /*#__PURE__*/_react.default.createElement(LazyCopyToSpaceFlyout, {
      onClose: onClose,
      savedObjectTarget: savedObjectTarget
    }));
  }
  const isStartShareButtonDisabled = !isSelectionChanged || shareInProgress || enableSpaceAgnosticBehavior && !shareOptions.selectedSpaceIds.length; // the object must exist in at least one space, or all spaces

  return /*#__PURE__*/_react.default.createElement(_eui.EuiFlyout, {
    onClose: onClose,
    maxWidth: 500,
    "data-test-subj": "share-to-space-flyout"
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlyoutHeader, {
    hasBorder: true
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexGroup, {
    alignItems: "center",
    gutterSize: "m",
    responsive: false
  }, flyoutIcon && /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
    grow: false
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiIcon, {
    size: "l",
    type: flyoutIcon
  })), /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, null, /*#__PURE__*/_react.default.createElement(_eui.EuiTitle, {
    size: "m"
  }, /*#__PURE__*/_react.default.createElement("h2", null, flyoutTitle))))), /*#__PURE__*/_react.default.createElement(_eui.EuiFlexGroup, {
    direction: "column",
    gutterSize: "none",
    className: "spcShareToSpace__flyoutBodyWrapper eui-yScroll",
    responsive: false
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
    grow: false
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiFlexGroup, {
    alignItems: "center",
    gutterSize: "m",
    responsive: false
  }, savedObjectTarget.icon && /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, {
    grow: false
  }, /*#__PURE__*/_react.default.createElement(_eui.EuiIcon, {
    type: savedObjectTarget.icon
  })), /*#__PURE__*/_react.default.createElement(_eui.EuiFlexItem, null, /*#__PURE__*/_react.default.createElement(_eui.EuiText, null, /*#__PURE__*/_react.default.createElement("p", null, savedObjectTarget.title))))), /*#__PURE__*/_react.default.createElement(_eui.EuiSpacer, {
    size: "m"
  }), getFlyoutBody()), /*#__PURE__*/_react.default.createElement(_eui.EuiFlyoutFooter, null, getFlyoutFooter()));
};
exports.ShareToSpaceFlyoutInternal = ShareToSpaceFlyoutInternal;