"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerRoutes = registerRoutes;
var _elasticsearch = require("@elastic/elasticsearch");
var _boom = require("@hapi/boom");
var t = _interopRequireWildcard(require("io-ts"));
var _lodash = require("lodash");
var _decode_request_params = require("./decode_request_params");
var _parse_endpoint = require("./parse_endpoint");
var _route_validation_object = require("./route_validation_object");
function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }
function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const CLIENT_CLOSED_REQUEST = {
  statusCode: 499,
  body: {
    message: 'Client closed request'
  }
};
function registerRoutes({
  core,
  repository,
  logger,
  dependencies
}) {
  const routes = Object.values(repository);
  const router = core.http.createRouter();
  routes.forEach(route => {
    const {
      params,
      endpoint,
      options,
      handler
    } = route;
    const {
      method,
      pathname,
      version
    } = (0, _parse_endpoint.parseEndpoint)(endpoint);
    const wrappedHandler = async (context, request, response) => {
      try {
        const runtimeType = params || t.strict({});
        const validatedParams = (0, _decode_request_params.decodeRequestParams)((0, _lodash.pick)(request, 'params', 'body', 'query'), runtimeType);
        const {
          aborted,
          data
        } = await Promise.race([handler({
          request,
          context,
          params: validatedParams,
          logger,
          ...dependencies
        }).then(value => {
          return {
            aborted: false,
            data: value
          };
        }), request.events.aborted$.toPromise().then(() => {
          return {
            aborted: true,
            data: undefined
          };
        })]);
        if (aborted) {
          return response.custom(CLIENT_CLOSED_REQUEST);
        }
        const body = data || {};
        return response.ok({
          body
        });
      } catch (error) {
        logger.error(error);
        const opts = {
          statusCode: 500,
          body: {
            message: error.message,
            attributes: {
              data: {}
            }
          }
        };
        if (error instanceof _elasticsearch.errors.RequestAbortedError) {
          return response.custom((0, _lodash.merge)(opts, CLIENT_CLOSED_REQUEST));
        }
        if ((0, _boom.isBoom)(error)) {
          opts.statusCode = error.output.statusCode;
          opts.body.attributes.data = error === null || error === void 0 ? void 0 : error.data;
        }
        return response.custom(opts);
      }
    };
    logger.debug(`Registering endpoint ${endpoint}`);
    if (!version) {
      router[method]({
        path: pathname,
        options,
        validate: _route_validation_object.routeValidationObject
      }, wrappedHandler);
    } else {
      router.versioned[method]({
        path: pathname,
        access: pathname.startsWith('/internal/') ? 'internal' : 'public',
        options
      }).addVersion({
        version,
        validate: {
          request: _route_validation_object.routeValidationObject
        }
      }, wrappedHandler);
    }
  });
}