"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createConfig = exports.configSchema = exports.config = void 0;
var _configSchema = require("@kbn/config-schema");
var _config = require("../common/config");
var _experimental_features = require("../common/experimental_features");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const configSchema = _configSchema.schema.object({
  enabled: _configSchema.schema.boolean({
    defaultValue: false
  }),
  productTypes: _config.productTypes,
  /**
   * For internal use. A list of string values (comma delimited) that will enable experimental
   * type of functionality that is not yet released. Valid values for this settings need to
   * be defined in:
   * `x-pack/plugins/security_solution_serverless/common/experimental_features.ts`
   * under the `allowedExperimentalValues` object
   *
   * @example
   * xpack.securitySolutionServerless.enableExperimental:
   *   - someCrazyServerlessFeature
   *   - someEvenCrazierServerlessFeature
   */
  enableExperimental: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
    defaultValue: () => []
  })
});
exports.configSchema = configSchema;
const config = {
  exposeToBrowser: {
    enableExperimental: true,
    productTypes: true
  },
  schema: configSchema,
  deprecations: ({
    renameFromRoot
  }) => [renameFromRoot('xpack.serverless.security.productTypes', 'xpack.securitySolutionServerless.productTypes', {
    silent: true,
    level: 'warning'
  })]
};
exports.config = config;
const createConfig = (context, securitySolution) => {
  const {
    enableExperimental,
    ...pluginConfig
  } = context.config.get();
  const logger = context.logger.get('config');
  const {
    invalid,
    duplicated,
    features: experimentalFeatures
  } = (0, _experimental_features.parseExperimentalConfigValue)(enableExperimental, securitySolution.experimentalFeatures);
  if (invalid.length) {
    logger.warn(`Unsupported "xpack.securitySolutionServerless.enableExperimental" values detected.
The following configuration values are not supported and should be removed from the configuration:

    xpack.securitySolutionServerless.enableExperimental:
${invalid.map(key => `      - ${key}`).join('\n')}
`);
  }
  if (duplicated.length) {
    logger.warn(`Duplicated "xpack.securitySolutionServerless.enableExperimental" values detected.
The following configuration values are should only be defined using the generic "xpack.securitySolution.enableExperimental":

    xpack.securitySolutionServerless.enableExperimental:
${duplicated.map(key => `      - ${key}`).join('\n')}
`);
  }
  return {
    ...pluginConfig,
    experimentalFeatures
  };
};
exports.createConfig = createConfig;