"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createProjectNavLinks$ = void 0;
var _rxjs = require("rxjs");
var _securitySolutionNavigation = require("@kbn/security-solution-navigation");
var _links = require("@kbn/security-solution-navigation/links");
var _assets_links = require("./sections/assets_links");
var _ml_links = require("./sections/ml_links");
var _project_settings_links = require("./sections/project_settings_links");
var _dev_tools_links = require("./sections/dev_tools_links");
var _discover_links = require("./sections/discover_links");
var _util = require("./util");
var _investigations_links = require("./sections/investigations_links");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createProjectNavLinks$ = (securityNavLinks$, core, cloud) => {
  const {
    chrome
  } = core;
  return (0, _rxjs.combineLatest)([securityNavLinks$, chrome.navLinks.getNavLinks$()]).pipe((0, _rxjs.debounceTime)(100),
  // avoid multiple calls in a short period of time
  (0, _rxjs.skipWhile)(([securityNavLinks, chromeNavLinks]) => securityNavLinks.length === 0 || chromeNavLinks.length === 0 // skip if not initialized
  ), (0, _rxjs.map)(([securityNavLinks]) => processNavLinks(securityNavLinks, chrome.navLinks, cloud)));
};

/**
 * Takes the security nav links and the chrome nav links and generates the project nav links
 * containing Security internal nav links and the external nav links (ML, Dev Tools, Project Settings, etc.)
 */
exports.createProjectNavLinks$ = createProjectNavLinks$;
const processNavLinks = (securityNavLinks, chromeNavLinks, cloud) => {
  const projectNavLinks = [...securityNavLinks];

  // Discover. just pushing it
  projectNavLinks.push(_discover_links.discoverNavLink);

  // Investigations. injecting external sub-links and categories definition to the landing
  const investigationsLinkIndex = projectNavLinks.findIndex(({
    id
  }) => id === _securitySolutionNavigation.SecurityPageName.investigations);
  if (investigationsLinkIndex !== -1) {
    var _investigationNavLink;
    const investigationNavLink = projectNavLinks[investigationsLinkIndex];
    projectNavLinks[investigationsLinkIndex] = {
      ...investigationNavLink,
      links: [...((_investigationNavLink = investigationNavLink.links) !== null && _investigationNavLink !== void 0 ? _investigationNavLink : []), ..._investigations_links.investigationsNavLinks]
    };
  }

  // ML. injecting external sub-links and categories definition to the landing
  const mlLinkIndex = projectNavLinks.findIndex(({
    id
  }) => id === _securitySolutionNavigation.SecurityPageName.mlLanding);
  if (mlLinkIndex !== -1) {
    projectNavLinks[mlLinkIndex] = {
      ...projectNavLinks[mlLinkIndex],
      categories: _ml_links.mlNavCategories,
      links: _ml_links.mlNavLinks
    };
  }

  // Assets, adding fleet external sub-links
  const assetsLinkIndex = projectNavLinks.findIndex(({
    id
  }) => id === _securitySolutionNavigation.SecurityPageName.assets);
  if (assetsLinkIndex !== -1) {
    var _assetsNavLink$links;
    const assetsNavLink = projectNavLinks[assetsLinkIndex];
    projectNavLinks[assetsLinkIndex] = {
      ...assetsNavLink,
      links: [..._assets_links.assetsNavLinks, ...((_assetsNavLink$links = assetsNavLink.links) !== null && _assetsNavLink$links !== void 0 ? _assetsNavLink$links : [])] // adds fleet to the existing (endpoints and cloud) links
    };
  }

  // Dev Tools. just pushing it
  projectNavLinks.push(_dev_tools_links.devToolsNavLink);
  projectNavLinks.push(..._project_settings_links.projectSettingsNavLinks);
  return processCloudLinks(filterDisabled(projectNavLinks, chromeNavLinks), cloud);
};

/**
 * Filters out the disabled external kibana nav links from the project nav links.
 * Internal Security links are already filtered by the security_solution plugin appLinks.
 */
const filterDisabled = (projectNavLinks, chromeNavLinks) => {
  return projectNavLinks.reduce((filteredNavLinks, navLink) => {
    const {
      id,
      links
    } = navLink;
    if (!(0, _links.isSecurityId)(id) && !(0, _util.isCloudLink)(id)) {
      const navLinkId = (0, _util.getNavLinkIdFromProjectPageName)(id);
      if (!chromeNavLinks.has(navLinkId)) {
        return filteredNavLinks;
      }
    }
    if (links) {
      filteredNavLinks.push({
        ...navLink,
        links: filterDisabled(links, chromeNavLinks)
      });
    } else {
      filteredNavLinks.push(navLink);
    }
    return filteredNavLinks;
  }, []);
};
const processCloudLinks = (links, cloud) => {
  return links.map(link => {
    const extraProps = {};
    if ((0, _util.isCloudLink)(link.id)) {
      const externalUrl = (0, _util.getCloudUrl)((0, _util.getCloudLinkKey)(link.id), cloud);
      extraProps.externalUrl = externalUrl || '#'; // fallback to # if empty, should only happen in dev
    }

    if (link.links) {
      extraProps.links = processCloudLinks(link.links, cloud);
    }
    return {
      ...link,
      ...extraProps
    };
  });
};