"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getRiskEngineMetrics = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const allEnititiesByTypeAggregationBody = {
  size: 0,
  aggs: {
    user_name: {
      value_count: {
        field: 'user.name'
      }
    },
    host_name: {
      value_count: {
        field: 'host.name'
      }
    }
  }
};
const getEntitiesAggregationData = async ({
  esClient,
  index,
  logger,
  hostMetricField,
  userMetricField,
  lastDay
}) => {
  try {
    var _riskScoreAggsRespons, _riskScoreAggsRespons2, _riskScoreAggsRespons3, _riskScoreAggsRespons4;
    const bodyRequest = {
      ...allEnititiesByTypeAggregationBody
    };
    if (lastDay) {
      bodyRequest.query = {
        range: {
          '@timestamp': {
            gte: 'now-24h',
            lt: 'now'
          }
        }
      };
    }
    const riskScoreAggsResponse = await esClient.search({
      index,
      body: bodyRequest
    });
    return {
      [userMetricField]: riskScoreAggsResponse === null || riskScoreAggsResponse === void 0 ? void 0 : (_riskScoreAggsRespons = riskScoreAggsResponse.aggregations) === null || _riskScoreAggsRespons === void 0 ? void 0 : (_riskScoreAggsRespons2 = _riskScoreAggsRespons.user_name) === null || _riskScoreAggsRespons2 === void 0 ? void 0 : _riskScoreAggsRespons2.value,
      [hostMetricField]: riskScoreAggsResponse === null || riskScoreAggsResponse === void 0 ? void 0 : (_riskScoreAggsRespons3 = riskScoreAggsResponse.aggregations) === null || _riskScoreAggsRespons3 === void 0 ? void 0 : (_riskScoreAggsRespons4 = _riskScoreAggsRespons3.host_name) === null || _riskScoreAggsRespons4 === void 0 ? void 0 : _riskScoreAggsRespons4.value
    };
  } catch (err) {
    logger.error(`Error while fetching risk score  metrics for ${hostMetricField} and ${userMetricField}: ${err}`);
    return {};
  }
};
const getIndexSize = async ({
  esClient,
  index,
  logger,
  metricField
}) => {
  try {
    var _riskScoreIndexStats$, _riskScoreIndexStats$2, _riskScoreIndexStats$3, _riskScoreIndexStats$4;
    const riskScoreIndexStats = await esClient.indices.stats({
      index
    });
    const sizeInMb = ((_riskScoreIndexStats$ = riskScoreIndexStats === null || riskScoreIndexStats === void 0 ? void 0 : (_riskScoreIndexStats$2 = riskScoreIndexStats._all) === null || _riskScoreIndexStats$2 === void 0 ? void 0 : (_riskScoreIndexStats$3 = _riskScoreIndexStats$2.primaries) === null || _riskScoreIndexStats$3 === void 0 ? void 0 : (_riskScoreIndexStats$4 = _riskScoreIndexStats$3.store) === null || _riskScoreIndexStats$4 === void 0 ? void 0 : _riskScoreIndexStats$4.size_in_bytes) !== null && _riskScoreIndexStats$ !== void 0 ? _riskScoreIndexStats$ : 0) / 1e6;
    return {
      [metricField]: sizeInMb
    };
  } catch (err) {
    logger.error(`Error while fetching risk score  metrics for ${metricField}: ${err}`);
    return {};
  }
};
const getRiskEngineMetrics = async ({
  esClient,
  logger,
  riskEngineIndexPatterns
}) => {
  logger.info('Fetch risk engine metrics');
  try {
    const riskEngineIndexes = await esClient.indices.get({
      index: `${riskEngineIndexPatterns.all}`
    });
    const isRiskEngineExists = Object.keys(riskEngineIndexes).length > 0;
    if (!isRiskEngineExists) {
      return {};
    }
    const results = await Promise.all([getEntitiesAggregationData({
      esClient,
      index: riskEngineIndexPatterns.latest,
      logger,
      lastDay: false,
      hostMetricField: 'unique_host_risk_score_total',
      userMetricField: 'unique_user_risk_score_total'
    }), getEntitiesAggregationData({
      esClient,
      index: riskEngineIndexPatterns.latest,
      logger,
      lastDay: true,
      hostMetricField: 'unique_host_risk_score_day',
      userMetricField: 'unique_user_risk_score_day'
    }), getEntitiesAggregationData({
      esClient,
      index: riskEngineIndexPatterns.all,
      logger,
      lastDay: false,
      hostMetricField: 'all_host_risk_scores_total',
      userMetricField: 'all_user_risk_scores_total'
    }), getEntitiesAggregationData({
      esClient,
      index: riskEngineIndexPatterns.all,
      logger,
      lastDay: true,
      hostMetricField: 'all_host_risk_scores_total_day',
      userMetricField: 'all_user_risk_scores_total_day'
    }), getIndexSize({
      esClient,
      logger,
      index: riskEngineIndexPatterns.all,
      metricField: 'all_risk_scores_index_size'
    }), getIndexSize({
      esClient,
      logger,
      index: riskEngineIndexPatterns.latest,
      metricField: 'unique_risk_scores_index_size'
    })]);
    return results.reduce((acc, curr) => ({
      ...acc,
      ...curr
    }), {});
  } catch (e) {
    logger.error(`Error while fetching risk engine metrics: ${e.message}`);
    return {};
  }
};
exports.getRiskEngineMetrics = getRiskEngineMetrics;