"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findSavedObjectsWithTagReference = exports.createRiskScoreTag = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _i18n = require("@kbn/i18n");
var _utils = require("./utils");
var _saved_objects = require("../../../tags/saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createRiskScoreTag = async ({
  riskScoreEntity,
  logger,
  savedObjectsClient,
  spaceId = 'default'
}) => {
  const tagName = (0, _utils.getRiskScoreTagName)(riskScoreEntity, spaceId);
  const savedObjectTemplate = `${riskScoreEntity}RiskScoreDashboards`;
  try {
    // check the tag does not already exist
    const [existingRiskScoreTag] = await (0, _saved_objects.findTagsByName)({
      savedObjectsClient,
      tagName
    });
    if ((existingRiskScoreTag === null || existingRiskScoreTag === void 0 ? void 0 : existingRiskScoreTag.id) != null) {
      logger.error(`${savedObjectTemplate} already exists`);
      return {
        [savedObjectTemplate]: {
          success: false,
          error: _i18n.i18n.translate('xpack.securitySolution.riskScore.savedObjects.templateAlreadyExistsTitle', {
            values: {
              savedObjectTemplate
            },
            defaultMessage: `Failed to import saved objects: {savedObjectTemplate} were not created as already exist`
          })
        }
      };
    }
  } catch (err) {
    const error = (0, _securitysolutionEsUtils.transformError)(err);
    logger.error(`${savedObjectTemplate} cannot be installed as failed to find the tag: ${tagName} - ${error === null || error === void 0 ? void 0 : error.message}`);
    return {
      [savedObjectTemplate]: {
        success: false,
        error: _i18n.i18n.translate('xpack.securitySolution.riskScore.savedObjects.failedToFindTagTitle', {
          values: {
            savedObjectTemplate,
            tagName
          },
          defaultMessage: `Failed to import saved objects: {savedObjectTemplate} were not created as failed to find the tag: {tagName}`
        })
      }
    };
  }
  try {
    const createTagResponse = await (0, _saved_objects.createTag)({
      savedObjectsClient,
      tagName,
      description: _utils.RISK_SCORE_TAG_DESCRIPTION,
      color: '#6edb7f'
    });
    return {
      [savedObjectTemplate]: {
        success: true,
        error: null,
        body: {
          type: 'tag',
          id: createTagResponse === null || createTagResponse === void 0 ? void 0 : createTagResponse.id,
          name: createTagResponse.attributes.name,
          description: createTagResponse.attributes.description
        }
      }
    };
  } catch (err) {
    const error = (0, _securitysolutionEsUtils.transformError)(err);
    logger.error(`${savedObjectTemplate} cannot be installed as failed to create the tag: ${tagName} - ${error === null || error === void 0 ? void 0 : error.message}`);
    return {
      [savedObjectTemplate]: {
        success: false,
        error: _i18n.i18n.translate('xpack.securitySolution.riskScore.savedObjects.failedToCreateTagTitle', {
          values: {
            savedObjectTemplate,
            tagName
          },
          defaultMessage: `Failed to import saved objects: {savedObjectTemplate} were not created as failed to create the tag: {tagName}`
        })
      }
    };
  }
};
exports.createRiskScoreTag = createRiskScoreTag;
const findSavedObjectsWithTagReference = async ({
  savedObjectsClient,
  savedObjectTypes,
  tagId
}) => {
  const linkedSavedObjects = await savedObjectsClient.find({
    type: savedObjectTypes,
    hasReference: {
      type: 'tag',
      id: tagId
    }
  });
  return linkedSavedObjects === null || linkedSavedObjects === void 0 ? void 0 : linkedSavedObjects.saved_objects;
};
exports.findSavedObjectsWithTagReference = findSavedObjectsWithTagReference;