"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.riskEngineInitRoute = void 0;
var _utils = require("@kbn/lists-plugin/server/routes/utils");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _constants = require("../../../../../common/constants");
var _translations = require("./translations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const riskEngineInitRoute = (router, getStartServices) => {
  router.versioned.post({
    access: 'internal',
    path: _constants.RISK_ENGINE_INIT_URL,
    options: {
      tags: ['access:securitySolution', `access:${_constants.APP_ID}-entity-analytics`]
    }
  }).addVersion({
    version: '1',
    validate: {}
  }, async (context, request, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    const securitySolution = await context.securitySolution;
    const [_, {
      taskManager
    }] = await getStartServices();
    const riskEngineDataClient = securitySolution.getRiskEngineDataClient();
    const riskScoreDataClient = securitySolution.getRiskScoreDataClient();
    const spaceId = securitySolution.getSpaceId();
    try {
      if (!taskManager) {
        return siemResponse.error({
          statusCode: 400,
          body: _translations.TASK_MANAGER_UNAVAILABLE_ERROR
        });
      }
      const initResult = await riskEngineDataClient.init({
        taskManager,
        namespace: spaceId,
        riskScoreDataClient
      });
      const initResultResponse = {
        risk_engine_enabled: initResult.riskEngineEnabled,
        risk_engine_resources_installed: initResult.riskEngineResourcesInstalled,
        risk_engine_configuration_created: initResult.riskEngineConfigurationCreated,
        legacy_risk_engine_disabled: initResult.legacyRiskEngineDisabled,
        errors: initResult.errors
      };
      if (!initResult.riskEngineEnabled || !initResult.riskEngineResourcesInstalled || !initResult.riskEngineConfigurationCreated) {
        return siemResponse.error({
          statusCode: 400,
          body: {
            message: initResultResponse.errors.join('\n'),
            full_error: initResultResponse
          },
          bypassErrorFormat: true
        });
      }
      return response.ok({
        body: {
          result: initResultResponse
        }
      });
    } catch (e) {
      const error = (0, _securitysolutionEsUtils.transformError)(e);
      return siemResponse.error({
        statusCode: error.statusCode,
        body: {
          message: error.message,
          full_error: JSON.stringify(e)
        },
        bypassErrorFormat: true
      });
    }
  });
};
exports.riskEngineInitRoute = riskEngineInitRoute;