"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AssetCriticalityDataClient = void 0;
var _common = require("@kbn/alerting-plugin/common");
var _create_or_update_index = require("../utils/create_or_update_index");
var _asset_criticality = require("../../../../common/asset_criticality");
var _configurations = require("./configurations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createId = ({
  idField,
  idValue
}) => `${idField}:${idValue}`;
class AssetCriticalityDataClient {
  constructor(options) {
    this.options = options;
  }
  /**
   * It will create idex for asset criticality,
   * or update mappings if index exists
   */
  async init() {
    await (0, _create_or_update_index.createOrUpdateIndex)({
      esClient: this.options.esClient,
      logger: this.options.logger,
      options: {
        index: this.getIndex(),
        mappings: (0, _common.mappingFromFieldMap)(_configurations.assetCriticalityFieldMap, 'strict')
      }
    });
  }
  getIndex() {
    return (0, _asset_criticality.getAssetCriticalityIndex)(this.options.namespace);
  }
  async doesIndexExist() {
    try {
      const result = await this.options.esClient.indices.exists({
        index: this.getIndex()
      });
      return result;
    } catch (e) {
      return false;
    }
  }
  async getStatus() {
    const isAssetCriticalityResourcesInstalled = await this.doesIndexExist();
    return {
      isAssetCriticalityResourcesInstalled
    };
  }
  async get(idParts) {
    const id = createId(idParts);
    try {
      const body = await this.options.esClient.get({
        id,
        index: this.getIndex()
      });
      return body._source;
    } catch (err) {
      if (err.statusCode === 404) {
        return undefined;
      } else {
        throw err;
      }
    }
  }
  async upsert(record) {
    const id = createId(record);
    const doc = {
      id_field: record.idField,
      id_value: record.idValue,
      criticality_level: record.criticalityLevel,
      '@timestamp': new Date().toISOString()
    };
    await this.options.esClient.update({
      id,
      index: this.getIndex(),
      body: {
        doc,
        doc_as_upsert: true
      }
    });
    return doc;
  }
  async delete(idParts) {
    await this.options.esClient.delete({
      id: createId(idParts),
      index: this.getIndex()
    });
  }
}
exports.AssetCriticalityDataClient = AssetCriticalityDataClient;