"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getClusterHealthRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _route_validation = require("../../../../../../utils/build_validation/route_validation");
var _utils = require("../../../../routes/utils");
var _rule_monitoring = require("../../../../../../../common/api/detection_engine/rule_monitoring");
var _health_timings = require("../health_timings");
var _get_cluster_health_request = require("./get_cluster_health_request");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Get health overview of the whole cluster. Scope: all detection rules in all Kibana spaces.
 * Returns:
 * - health state at the moment of the API call
 * - health stats over a specified period of time ("health interval")
 * - health stats history within the same interval in the form of a histogram
 *   (the same stats are calculated over each of the discreet sub-intervals of the whole interval)
 */
const getClusterHealthRoute = router => {
  router.versioned.get({
    access: 'internal',
    path: _rule_monitoring.GET_CLUSTER_HEALTH_URL,
    options: {
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: '1',
    validate: {}
  }, async (context, request, response) => {
    return handleClusterHealthRequest({
      response,
      resolveParameters: () => (0, _get_cluster_health_request.validateGetClusterHealthRequest)({}),
      resolveDependencies: async () => {
        const ctx = await context.resolve(['securitySolution']);
        const healthClient = ctx.securitySolution.getDetectionEngineHealthClient();
        return {
          healthClient
        };
      }
    });
  });
  router.versioned.post({
    access: 'internal',
    path: _rule_monitoring.GET_CLUSTER_HEALTH_URL,
    options: {
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: (0, _route_validation.buildRouteValidation)(_rule_monitoring.GetClusterHealthRequestBody)
      }
    }
  }, async (context, request, response) => {
    return handleClusterHealthRequest({
      response,
      resolveParameters: () => (0, _get_cluster_health_request.validateGetClusterHealthRequest)(request.body),
      resolveDependencies: async () => {
        const ctx = await context.resolve(['securitySolution']);
        const healthClient = ctx.securitySolution.getDetectionEngineHealthClient();
        return {
          healthClient
        };
      }
    });
  });
};
exports.getClusterHealthRoute = getClusterHealthRoute;
const handleClusterHealthRequest = async args => {
  const {
    response,
    resolveParameters,
    resolveDependencies
  } = args;
  const siemResponse = (0, _utils.buildSiemResponse)(response);
  try {
    const params = resolveParameters();
    const {
      healthClient
    } = await resolveDependencies();
    const clusterHealthParameters = {
      interval: params.interval
    };
    const clusterHealth = await healthClient.calculateClusterHealth(clusterHealthParameters);
    const responseBody = {
      timings: (0, _health_timings.calculateHealthTimings)(params.requestReceivedAt),
      parameters: clusterHealthParameters,
      health: {
        ...clusterHealth,
        debug: params.debug ? clusterHealth.debug : undefined
      }
    };
    return response.ok({
      body: responseBody
    });
  } catch (err) {
    const error = (0, _securitysolutionEsUtils.transformError)(err);
    return siemResponse.error({
      body: error.message,
      statusCode: error.statusCode
    });
  }
};