"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.importRuleActionConnectors = void 0;
var _stream = require("stream");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const NO_ACTION_RESULT = {
  success: true,
  errors: [],
  successCount: 0,
  warnings: []
};
const importRuleActionConnectors = async ({
  actionConnectors,
  actionsClient,
  actionsImporter,
  rules,
  overwrite
}) => {
  try {
    const connectorIdToRuleIdsMap = (0, _utils.getActionConnectorRules)(rules);
    const referencedConnectorIds = await filterOutPreconfiguredConnectors(actionsClient, Object.keys(connectorIdToRuleIdsMap));
    if (!referencedConnectorIds.length) {
      return NO_ACTION_RESULT;
    }
    if (overwrite && !actionConnectors.length) {
      return (0, _utils.handleActionsHaveNoConnectors)(referencedConnectorIds, connectorIdToRuleIdsMap);
    }
    let actionConnectorsToImport = actionConnectors;
    if (!overwrite) {
      const newIdsToAdd = await (0, _utils.filterExistingActionConnectors)(actionsClient, referencedConnectorIds);
      const foundMissingConnectors = (0, _utils.checkIfActionsHaveMissingConnectors)(actionConnectors, newIdsToAdd, connectorIdToRuleIdsMap);
      if (foundMissingConnectors) return foundMissingConnectors;
      // filter out existing connectors
      actionConnectorsToImport = actionConnectors.filter(({
        id
      }) => newIdsToAdd.includes(id));
    }
    if (!actionConnectorsToImport.length) {
      return NO_ACTION_RESULT;
    }
    const readStream = _stream.Readable.from(actionConnectorsToImport);
    const {
      success,
      successCount,
      successResults,
      warnings,
      errors
    } = await actionsImporter.import({
      readStream,
      overwrite,
      createNewCopies: false
    });
    /*
      // When a connector is exported from one namespace and imported to another, it does not result in an error, but instead a new object is created with
      // new destination id and id will have the old  origin id, so in order to be able to use the newly generated Connectors id, this util is used to swap the old id with the
      // new destination Id
      */
    let rulesWithMigratedActions;
    if (successResults !== null && successResults !== void 0 && successResults.some(res => res.destinationId)) rulesWithMigratedActions = (0, _utils.updateRuleActionsWithMigratedResults)(rules, successResults);
    return {
      success,
      successCount,
      errors: errors ? (0, _utils.mapSOErrorToRuleError)(errors) : [],
      warnings: warnings || [],
      rulesWithMigratedActions
    };
  } catch (error) {
    return (0, _utils.returnErroredImportResult)(error);
  }
};
exports.importRuleActionConnectors = importRuleActionConnectors;
async function fetchPreconfiguredActionConnectors(actionsClient) {
  const knownConnectors = await actionsClient.getAll();
  return knownConnectors.filter(c => c.isPreconfigured);
}
async function filterOutPreconfiguredConnectors(actionsClient, connectorsIds) {
  if (connectorsIds.length === 0) {
    return [];
  }
  const preconfiguredActionConnectors = await fetchPreconfiguredActionConnectors(actionsClient);
  const preconfiguredActionConnectorIds = new Set(preconfiguredActionConnectors.map(c => c.id));
  return connectorsIds.filter(id => !preconfiguredActionConnectorIds.has(id));
}