"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.calculateRuleFieldsDiff = void 0;
var _utility_types = require("../../../../../../../common/utility_types");
var _invariant = require("../../../../../../../common/utils/invariant");
var _three_way_diff = require("../../../../../../../common/api/detection_engine/prebuilt_rules/model/diff/three_way_diff/three_way_diff");
var _diff_calculation_helpers = require("./diff_calculation_helpers");
var _simple_diff_algorithm = require("./algorithms/simple_diff_algorithm");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const BASE_TYPE_ERROR = `Base version can't be of different rule type`;
const TARGET_TYPE_ERROR = `Target version can't be of different rule type`;

/**
 * Calculates a three-way diff per each top-level rule field.
 * Returns an object which keys are equal to rule's field names and values are
 * three-way diffs calculated for those fields.
 */
const calculateRuleFieldsDiff = ruleVersions => {
  const commonFieldsDiff = calculateCommonFieldsDiff(ruleVersions);
  // eslint-disable-next-line @typescript-eslint/naming-convention
  const {
    base_version,
    current_version,
    target_version
  } = ruleVersions;
  const hasBaseVersion = base_version !== _three_way_diff.MissingVersion;
  const isRuleTypeDifferentInTargetVersion = current_version.type !== target_version.type;
  const isRuleTypeDifferentInBaseVersion = hasBaseVersion ? current_version.type !== base_version.type : false;
  if (isRuleTypeDifferentInTargetVersion || isRuleTypeDifferentInBaseVersion) {
    // If rule type has been changed by Elastic in the target version (can happen)
    // or by user in the current version (should never happen), we can't calculate the diff
    // only for fields of a single rule type, and need to calculate it for all fields
    // of all the rule types we have.
    // TODO: Try to get rid of "as" casting
    return calculateAllFieldsDiff({
      base_version: base_version,
      current_version: current_version,
      target_version: target_version
    });
  }
  switch (current_version.type) {
    case 'query':
      {
        if (hasBaseVersion) {
          (0, _invariant.invariant)(base_version.type === 'query', BASE_TYPE_ERROR);
        }
        (0, _invariant.invariant)(target_version.type === 'query', TARGET_TYPE_ERROR);
        return {
          ...commonFieldsDiff,
          ...calculateCustomQueryFieldsDiff({
            base_version,
            current_version,
            target_version
          })
        };
      }
    case 'saved_query':
      {
        if (hasBaseVersion) {
          (0, _invariant.invariant)(base_version.type === 'saved_query', BASE_TYPE_ERROR);
        }
        (0, _invariant.invariant)(target_version.type === 'saved_query', TARGET_TYPE_ERROR);
        return {
          ...commonFieldsDiff,
          ...calculateSavedQueryFieldsDiff({
            base_version,
            current_version,
            target_version
          })
        };
      }
    case 'eql':
      {
        if (hasBaseVersion) {
          (0, _invariant.invariant)(base_version.type === 'eql', BASE_TYPE_ERROR);
        }
        (0, _invariant.invariant)(target_version.type === 'eql', TARGET_TYPE_ERROR);
        return {
          ...commonFieldsDiff,
          ...calculateEqlFieldsDiff({
            base_version,
            current_version,
            target_version
          })
        };
      }
    case 'threat_match':
      {
        if (hasBaseVersion) {
          (0, _invariant.invariant)(base_version.type === 'threat_match', BASE_TYPE_ERROR);
        }
        (0, _invariant.invariant)(target_version.type === 'threat_match', TARGET_TYPE_ERROR);
        return {
          ...commonFieldsDiff,
          ...calculateThreatMatchFieldsDiff({
            base_version,
            current_version,
            target_version
          })
        };
      }
    case 'threshold':
      {
        if (hasBaseVersion) {
          (0, _invariant.invariant)(base_version.type === 'threshold', BASE_TYPE_ERROR);
        }
        (0, _invariant.invariant)(target_version.type === 'threshold', TARGET_TYPE_ERROR);
        return {
          ...commonFieldsDiff,
          ...calculateThresholdFieldsDiff({
            base_version,
            current_version,
            target_version
          })
        };
      }
    case 'machine_learning':
      {
        if (hasBaseVersion) {
          (0, _invariant.invariant)(base_version.type === 'machine_learning', BASE_TYPE_ERROR);
        }
        (0, _invariant.invariant)(target_version.type === 'machine_learning', TARGET_TYPE_ERROR);
        return {
          ...commonFieldsDiff,
          ...calculateMachineLearningFieldsDiff({
            base_version,
            current_version,
            target_version
          })
        };
      }
    case 'new_terms':
      {
        if (hasBaseVersion) {
          (0, _invariant.invariant)(base_version.type === 'new_terms', BASE_TYPE_ERROR);
        }
        (0, _invariant.invariant)(target_version.type === 'new_terms', TARGET_TYPE_ERROR);
        return {
          ...commonFieldsDiff,
          ...calculateNewTermsFieldsDiff({
            base_version,
            current_version,
            target_version
          })
        };
      }
    case 'esql':
      {
        if (hasBaseVersion) {
          (0, _invariant.invariant)(base_version.type === 'esql', BASE_TYPE_ERROR);
        }
        (0, _invariant.invariant)(target_version.type === 'esql', TARGET_TYPE_ERROR);
        return {
          ...commonFieldsDiff,
          ...calculateEsqlFieldsDiff({
            base_version,
            current_version,
            target_version
          })
        };
      }
    default:
      {
        return (0, _utility_types.assertUnreachable)(current_version, 'Unhandled rule type');
      }
  }
};
exports.calculateRuleFieldsDiff = calculateRuleFieldsDiff;
const calculateCommonFieldsDiff = ruleVersions => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, commonFieldsDiffAlgorithms);
};
const commonFieldsDiffAlgorithms = {
  rule_id: _simple_diff_algorithm.simpleDiffAlgorithm,
  version: _simple_diff_algorithm.simpleDiffAlgorithm,
  meta: _simple_diff_algorithm.simpleDiffAlgorithm,
  name: _simple_diff_algorithm.simpleDiffAlgorithm,
  tags: _simple_diff_algorithm.simpleDiffAlgorithm,
  description: _simple_diff_algorithm.simpleDiffAlgorithm,
  severity: _simple_diff_algorithm.simpleDiffAlgorithm,
  severity_mapping: _simple_diff_algorithm.simpleDiffAlgorithm,
  risk_score: _simple_diff_algorithm.simpleDiffAlgorithm,
  risk_score_mapping: _simple_diff_algorithm.simpleDiffAlgorithm,
  references: _simple_diff_algorithm.simpleDiffAlgorithm,
  false_positives: _simple_diff_algorithm.simpleDiffAlgorithm,
  threat: _simple_diff_algorithm.simpleDiffAlgorithm,
  note: _simple_diff_algorithm.simpleDiffAlgorithm,
  setup: _simple_diff_algorithm.simpleDiffAlgorithm,
  related_integrations: _simple_diff_algorithm.simpleDiffAlgorithm,
  required_fields: _simple_diff_algorithm.simpleDiffAlgorithm,
  author: _simple_diff_algorithm.simpleDiffAlgorithm,
  license: _simple_diff_algorithm.simpleDiffAlgorithm,
  rule_schedule: _simple_diff_algorithm.simpleDiffAlgorithm,
  actions: _simple_diff_algorithm.simpleDiffAlgorithm,
  throttle: _simple_diff_algorithm.simpleDiffAlgorithm,
  exceptions_list: _simple_diff_algorithm.simpleDiffAlgorithm,
  max_signals: _simple_diff_algorithm.simpleDiffAlgorithm,
  rule_name_override: _simple_diff_algorithm.simpleDiffAlgorithm,
  timestamp_override: _simple_diff_algorithm.simpleDiffAlgorithm,
  timeline_template: _simple_diff_algorithm.simpleDiffAlgorithm,
  building_block: _simple_diff_algorithm.simpleDiffAlgorithm
};
const calculateCustomQueryFieldsDiff = ruleVersions => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, customQueryFieldsDiffAlgorithms);
};
const customQueryFieldsDiffAlgorithms = {
  type: _simple_diff_algorithm.simpleDiffAlgorithm,
  kql_query: _simple_diff_algorithm.simpleDiffAlgorithm,
  data_source: _simple_diff_algorithm.simpleDiffAlgorithm,
  alert_suppression: _simple_diff_algorithm.simpleDiffAlgorithm
};
const calculateSavedQueryFieldsDiff = ruleVersions => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, savedQueryFieldsDiffAlgorithms);
};
const savedQueryFieldsDiffAlgorithms = {
  type: _simple_diff_algorithm.simpleDiffAlgorithm,
  kql_query: _simple_diff_algorithm.simpleDiffAlgorithm,
  data_source: _simple_diff_algorithm.simpleDiffAlgorithm,
  alert_suppression: _simple_diff_algorithm.simpleDiffAlgorithm
};
const calculateEqlFieldsDiff = ruleVersions => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, eqlFieldsDiffAlgorithms);
};
const eqlFieldsDiffAlgorithms = {
  type: _simple_diff_algorithm.simpleDiffAlgorithm,
  eql_query: _simple_diff_algorithm.simpleDiffAlgorithm,
  data_source: _simple_diff_algorithm.simpleDiffAlgorithm,
  event_category_override: _simple_diff_algorithm.simpleDiffAlgorithm,
  timestamp_field: _simple_diff_algorithm.simpleDiffAlgorithm,
  tiebreaker_field: _simple_diff_algorithm.simpleDiffAlgorithm
};
const calculateEsqlFieldsDiff = ruleVersions => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, esqlFieldsDiffAlgorithms);
};
const esqlFieldsDiffAlgorithms = {
  type: _simple_diff_algorithm.simpleDiffAlgorithm,
  esql_query: _simple_diff_algorithm.simpleDiffAlgorithm
};
const calculateThreatMatchFieldsDiff = ruleVersions => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, threatMatchFieldsDiffAlgorithms);
};
const threatMatchFieldsDiffAlgorithms = {
  type: _simple_diff_algorithm.simpleDiffAlgorithm,
  kql_query: _simple_diff_algorithm.simpleDiffAlgorithm,
  data_source: _simple_diff_algorithm.simpleDiffAlgorithm,
  threat_query: _simple_diff_algorithm.simpleDiffAlgorithm,
  threat_index: _simple_diff_algorithm.simpleDiffAlgorithm,
  threat_mapping: _simple_diff_algorithm.simpleDiffAlgorithm,
  threat_indicator_path: _simple_diff_algorithm.simpleDiffAlgorithm,
  concurrent_searches: _simple_diff_algorithm.simpleDiffAlgorithm,
  items_per_search: _simple_diff_algorithm.simpleDiffAlgorithm
};
const calculateThresholdFieldsDiff = ruleVersions => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, thresholdFieldsDiffAlgorithms);
};
const thresholdFieldsDiffAlgorithms = {
  type: _simple_diff_algorithm.simpleDiffAlgorithm,
  kql_query: _simple_diff_algorithm.simpleDiffAlgorithm,
  data_source: _simple_diff_algorithm.simpleDiffAlgorithm,
  threshold: _simple_diff_algorithm.simpleDiffAlgorithm
};
const calculateMachineLearningFieldsDiff = ruleVersions => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, machineLearningFieldsDiffAlgorithms);
};
const machineLearningFieldsDiffAlgorithms = {
  type: _simple_diff_algorithm.simpleDiffAlgorithm,
  machine_learning_job_id: _simple_diff_algorithm.simpleDiffAlgorithm,
  anomaly_threshold: _simple_diff_algorithm.simpleDiffAlgorithm
};
const calculateNewTermsFieldsDiff = ruleVersions => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, newTermsFieldsDiffAlgorithms);
};
const newTermsFieldsDiffAlgorithms = {
  type: _simple_diff_algorithm.simpleDiffAlgorithm,
  kql_query: _simple_diff_algorithm.simpleDiffAlgorithm,
  data_source: _simple_diff_algorithm.simpleDiffAlgorithm,
  new_terms_fields: _simple_diff_algorithm.simpleDiffAlgorithm,
  history_window_start: _simple_diff_algorithm.simpleDiffAlgorithm
};
const calculateAllFieldsDiff = ruleVersions => {
  return (0, _diff_calculation_helpers.calculateFieldsDiffFor)(ruleVersions, allFieldsDiffAlgorithms);
};
const allFieldsDiffAlgorithms = {
  ...commonFieldsDiffAlgorithms,
  ...customQueryFieldsDiffAlgorithms,
  ...savedQueryFieldsDiffAlgorithms,
  ...eqlFieldsDiffAlgorithms,
  ...threatMatchFieldsDiffAlgorithms,
  ...thresholdFieldsDiffAlgorithms,
  ...machineLearningFieldsDiffAlgorithms,
  ...newTermsFieldsDiffAlgorithms,
  type: _simple_diff_algorithm.simpleDiffAlgorithm
};