"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.performRuleInstallationRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _prebuilt_rules = require("../../../../../../common/api/detection_engine/prebuilt_rules");
var _route_validation = require("../../../../../utils/build_validation/route_validation");
var _utils = require("../../../routes/utils");
var _rule_converters = require("../../../rule_management/normalization/rule_converters");
var _aggregate_prebuilt_rule_errors = require("../../logic/aggregate_prebuilt_rule_errors");
var _ensure_latest_rules_package_installed = require("../../logic/ensure_latest_rules_package_installed");
var _prebuilt_rule_assets_client = require("../../logic/rule_assets/prebuilt_rule_assets_client");
var _create_prebuilt_rules = require("../../logic/rule_objects/create_prebuilt_rules");
var _prebuilt_rule_objects_client = require("../../logic/rule_objects/prebuilt_rule_objects_client");
var _fetch_rule_versions_triad = require("../../logic/rule_versions/fetch_rule_versions_triad");
var _get_version_buckets = require("../../model/rule_versions/get_version_buckets");
var _perform_timelines_installation = require("../../logic/perform_timelines_installation");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const performRuleInstallationRoute = router => {
  router.versioned.post({
    access: 'internal',
    path: _prebuilt_rules.PERFORM_RULE_INSTALLATION_URL,
    options: {
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: (0, _route_validation.buildRouteValidation)(_prebuilt_rules.PerformRuleInstallationRequestBody)
      }
    }
  }, async (context, request, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    try {
      const ctx = await context.resolve(['core', 'alerting', 'securitySolution']);
      const config = ctx.securitySolution.getConfig();
      const soClient = ctx.core.savedObjects.client;
      const rulesClient = ctx.alerting.getRulesClient();
      const ruleAssetsClient = (0, _prebuilt_rule_assets_client.createPrebuiltRuleAssetsClient)(soClient);
      const ruleObjectsClient = (0, _prebuilt_rule_objects_client.createPrebuiltRuleObjectsClient)(rulesClient);
      const exceptionsListClient = ctx.securitySolution.getExceptionListClient();
      const {
        mode
      } = request.body;

      // This will create the endpoint list if it does not exist yet
      await (exceptionsListClient === null || exceptionsListClient === void 0 ? void 0 : exceptionsListClient.createEndpointList());

      // If this API is used directly without hitting any detection engine
      // pages first, the rules package might be missing.
      await (0, _ensure_latest_rules_package_installed.ensureLatestRulesPackageInstalled)(ruleAssetsClient, config, ctx.securitySolution);
      const fetchErrors = [];
      const skippedRules = [];
      const ruleVersionsMap = await (0, _fetch_rule_versions_triad.fetchRuleVersionsTriad)({
        ruleAssetsClient,
        ruleObjectsClient,
        versionSpecifiers: mode === 'ALL_RULES' ? undefined : request.body.rules
      });
      const {
        currentRules,
        installableRules
      } = (0, _get_version_buckets.getVersionBuckets)(ruleVersionsMap);

      // Perform all the checks we can before we start the upgrade process
      if (mode === 'SPECIFIC_RULES') {
        const currentRuleIds = new Set(currentRules.map(rule => rule.rule_id));
        const installableRuleIds = new Set(installableRules.map(rule => rule.rule_id));
        request.body.rules.forEach(rule => {
          // Check that the requested rule is not installed yet
          if (currentRuleIds.has(rule.rule_id)) {
            skippedRules.push({
              rule_id: rule.rule_id,
              reason: _prebuilt_rules.SkipRuleInstallReason.ALREADY_INSTALLED
            });
            return;
          }

          // Check that the requested rule is installable
          if (!installableRuleIds.has(rule.rule_id)) {
            fetchErrors.push({
              error: new Error(`Rule with ID "${rule.rule_id}" and version "${rule.version}" not found`),
              item: rule
            });
          }
        });
      }
      const {
        results: installedRules,
        errors: installationErrors
      } = await (0, _create_prebuilt_rules.createPrebuiltRules)(rulesClient, installableRules);
      const ruleErrors = [...fetchErrors, ...installationErrors];
      const {
        error: timelineInstallationError
      } = await (0, _perform_timelines_installation.performTimelinesInstallation)(ctx.securitySolution);
      const allErrors = (0, _aggregate_prebuilt_rule_errors.aggregatePrebuiltRuleErrors)(ruleErrors);
      if (timelineInstallationError) {
        allErrors.push({
          message: timelineInstallationError,
          rules: []
        });
      }
      const body = {
        summary: {
          total: installedRules.length + skippedRules.length + ruleErrors.length,
          succeeded: installedRules.length,
          skipped: skippedRules.length,
          failed: ruleErrors.length
        },
        results: {
          created: installedRules.map(({
            result
          }) => (0, _rule_converters.internalRuleToAPIResponse)(result)),
          skipped: skippedRules
        },
        errors: allErrors
      };
      return response.ok({
        body
      });
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return siemResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.performRuleInstallationRoute = performRuleInstallationRoute;