"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.resolveTimeline = exports.persistTimeline = exports.persistFavorite = exports.installPrepackedTimelines = exports.importTimelines = exports.getTimelineTemplate = exports.getTimeline = exports.getDraftTimeline = exports.getAllTimelines = exports.exportSelectedTimeline = exports.deleteTimelinesByIds = exports.copyTimeline = exports.cleanDraftTimeline = void 0;
var _Either = require("fp-ts/lib/Either");
var _function = require("fp-ts/lib/function");
var _pipeable = require("fp-ts/lib/pipeable");
var _lodash = require("lodash");
var _common = require("@kbn/cases-plugin/common");
var _timeline = require("../../../common/api/timeline");
var _constants = require("../../../common/constants");
var _kibana = require("../../common/lib/kibana");
var _toasters = require("../../common/components/toasters");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createToasterPlainError = message => new _toasters.ToasterError([message]);
const decodeTimelineResponse = respTimeline => (0, _pipeable.pipe)(_timeline.TimelineResponseType.decode(respTimeline), (0, _Either.fold)((0, _common.throwErrors)(createToasterPlainError), _function.identity));
const decodeSingleTimelineResponse = respTimeline => (0, _pipeable.pipe)(_timeline.SingleTimelineResponseType.decode(respTimeline), (0, _Either.fold)((0, _common.throwErrors)(createToasterPlainError), _function.identity));
const decodeResolvedSingleTimelineResponse = respTimeline => (0, _pipeable.pipe)(_timeline.ResolvedSingleTimelineResponseType.decode(respTimeline), (0, _Either.fold)((0, _common.throwErrors)(createToasterPlainError), _function.identity));
const decodeAllTimelinesResponse = respTimeline => (0, _pipeable.pipe)(_timeline.allTimelinesResponse.decode(respTimeline), (0, _Either.fold)((0, _common.throwErrors)(createToasterPlainError), _function.identity));
const decodeTimelineErrorResponse = respTimeline => (0, _pipeable.pipe)(_timeline.TimelineErrorResponseType.decode(respTimeline), (0, _Either.fold)((0, _common.throwErrors)(createToasterPlainError), _function.identity));
const decodePrepackedTimelineResponse = respTimeline => (0, _pipeable.pipe)(_timeline.importTimelineResultSchema.decode(respTimeline), (0, _Either.fold)((0, _common.throwErrors)(createToasterPlainError), _function.identity));
const decodeResponseFavoriteTimeline = respTimeline => (0, _pipeable.pipe)(_timeline.responseFavoriteTimeline.decode(respTimeline), (0, _Either.fold)((0, _common.throwErrors)(createToasterPlainError), _function.identity));
const postTimeline = async ({
  timeline
}) => {
  let requestBody;
  try {
    requestBody = JSON.stringify({
      timeline
    });
  } catch (err) {
    return Promise.reject(new Error(`Failed to stringify query: ${JSON.stringify(err)}`));
  }
  const response = await _kibana.KibanaServices.get().http.post(_constants.TIMELINE_URL, {
    method: 'POST',
    body: requestBody,
    version: '2023-10-31'
  });
  return decodeTimelineResponse(response);
};
const patchTimeline = async ({
  timelineId,
  timeline,
  version,
  savedSearch
}) => {
  let response = null;
  let requestBody = null;
  try {
    requestBody = JSON.stringify({
      timeline,
      timelineId,
      version
    });
  } catch (err) {
    return Promise.reject(new Error(`Failed to stringify query: ${JSON.stringify(err)}`));
  }
  try {
    if (timeline.savedSearchId && savedSearch) {
      const {
        savedSearch: savedSearchService
      } = _kibana.KibanaServices.get();
      await savedSearchService.save(savedSearch, {
        onTitleDuplicate: () => ({}),
        copyOnSave: false
      });
    }
  } catch (e) {
    return Promise.reject(new Error(`Failed to copy saved search: ${timeline.savedSearchId}`));
  }
  try {
    response = await _kibana.KibanaServices.get().http.patch(_constants.TIMELINE_URL, {
      method: 'PATCH',
      body: requestBody,
      version: '2023-10-31'
    });
  } catch (err) {
    // For Future developer
    // We are not rejecting our promise here because we had issue with our RXJS epic
    // the issue we were not able to pass the right object to it so we did manage the error in the success
    return Promise.resolve(decodeTimelineErrorResponse(err.body));
  }
  return decodeTimelineResponse(response);
};

/**
 * Creates a copy of the timeline with the given id. It will also apply changes to the original timeline
 * which are passed as `timeline` here.
 */
const copyTimeline = async ({
  timelineId,
  timeline,
  savedSearch
}) => {
  let response = null;
  let requestBody = null;
  let newSavedSearchId = null;
  try {
    if (timeline.savedSearchId && savedSearch) {
      const {
        savedSearch: savedSearchService
      } = _kibana.KibanaServices.get();
      const savedSearchCopy = {
        ...savedSearch
      };
      // delete the id and change the title to make sure we can copy the saved search
      delete savedSearchCopy.id;
      newSavedSearchId = await savedSearchService.save(savedSearchCopy, {
        onTitleDuplicate: () => ({}),
        copyOnSave: false
      });
    }
  } catch (e) {
    return Promise.reject(new Error(`Failed to copy saved search: ${timeline.savedSearchId}`));
  }
  try {
    requestBody = JSON.stringify({
      timeline: {
        ...timeline,
        savedSearchId: newSavedSearchId || timeline.savedSearchId
      },
      timelineIdToCopy: timelineId
    });
  } catch (err) {
    return Promise.reject(new Error(`Failed to stringify query: ${JSON.stringify(err)}`));
  }
  try {
    response = await _kibana.KibanaServices.get().http.post(_constants.TIMELINE_COPY_URL, {
      method: 'POST',
      body: requestBody,
      version: '1'
    });
  } catch (err) {
    // For Future developer
    // We are not rejecting our promise here because we had issue with our RXJS epic
    // the issue we were not able to pass the right object to it so we did manage the error in the success
    return Promise.resolve(decodeTimelineErrorResponse(err.body));
  }
  return decodeTimelineResponse(response);
};
exports.copyTimeline = copyTimeline;
const persistTimeline = async ({
  timelineId,
  timeline,
  version,
  savedSearch
}) => {
  try {
    if ((0, _lodash.isEmpty)(timelineId) && timeline.status === _timeline.TimelineStatus.draft && timeline) {
      var _timeline$templateTim, _timeline$templateTim2, _draftTimeline$data$p, _draftTimeline$data$p2, _draftTimeline$data$p3;
      const temp = await cleanDraftTimeline({
        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        timelineType: timeline.timelineType,
        templateTimelineId: (_timeline$templateTim = timeline.templateTimelineId) !== null && _timeline$templateTim !== void 0 ? _timeline$templateTim : undefined,
        templateTimelineVersion: (_timeline$templateTim2 = timeline.templateTimelineVersion) !== null && _timeline$templateTim2 !== void 0 ? _timeline$templateTim2 : undefined
      });
      const draftTimeline = decodeTimelineResponse(temp);
      const templateTimelineInfo = timeline.timelineType === _timeline.TimelineType.template ? {
        templateTimelineId: (_draftTimeline$data$p = draftTimeline.data.persistTimeline.timeline.templateTimelineId) !== null && _draftTimeline$data$p !== void 0 ? _draftTimeline$data$p : timeline.templateTimelineId,
        templateTimelineVersion: (_draftTimeline$data$p2 = draftTimeline.data.persistTimeline.timeline.templateTimelineVersion) !== null && _draftTimeline$data$p2 !== void 0 ? _draftTimeline$data$p2 : timeline.templateTimelineVersion
      } : {};
      return patchTimeline({
        timelineId: draftTimeline.data.persistTimeline.timeline.savedObjectId,
        timeline: {
          ...timeline,
          ...templateTimelineInfo
        },
        version: (_draftTimeline$data$p3 = draftTimeline.data.persistTimeline.timeline.version) !== null && _draftTimeline$data$p3 !== void 0 ? _draftTimeline$data$p3 : '',
        savedSearch
      });
    }
    if ((0, _lodash.isEmpty)(timelineId)) {
      return postTimeline({
        timeline
      });
    }
    return patchTimeline({
      timelineId: timelineId !== null && timelineId !== void 0 ? timelineId : '-1',
      timeline,
      version: version !== null && version !== void 0 ? version : '',
      savedSearch
    });
  } catch (err) {
    if (err.status_code === 403 || err.body.status_code === 403) {
      return Promise.resolve({
        data: {
          persistTimeline: {
            code: 403,
            message: err.message || err.body.message,
            timeline: {
              ...timeline,
              savedObjectId: '',
              version: ''
            }
          }
        }
      });
    }
    return Promise.resolve(err);
  }
};
exports.persistTimeline = persistTimeline;
const importTimelines = async ({
  fileToImport,
  signal
}) => {
  const formData = new FormData();
  formData.append('file', fileToImport);
  return _kibana.KibanaServices.get().http.fetch(`${_constants.TIMELINE_IMPORT_URL}`, {
    method: 'POST',
    headers: {
      'Content-Type': undefined
    },
    body: formData,
    signal,
    version: '2023-10-31'
  });
};
exports.importTimelines = importTimelines;
const exportSelectedTimeline = ({
  filename = `timelines_export.ndjson`,
  ids = [],
  signal
}) => {
  let requestBody;
  try {
    requestBody = ids.length > 0 ? JSON.stringify({
      ids
    }) : undefined;
  } catch (err) {
    return Promise.reject(new Error(`Failed to stringify query: ${JSON.stringify(err)}`));
  }
  return _kibana.KibanaServices.get().http.fetch(`${_constants.TIMELINE_EXPORT_URL}`, {
    method: 'POST',
    body: requestBody,
    query: {
      file_name: filename
    },
    signal,
    version: '2023-10-31'
  });
};
exports.exportSelectedTimeline = exportSelectedTimeline;
const getDraftTimeline = async ({
  timelineType
}) => {
  const response = await _kibana.KibanaServices.get().http.get(_constants.TIMELINE_DRAFT_URL, {
    query: {
      timelineType
    },
    version: '2023-10-31'
  });
  return decodeTimelineResponse(response);
};
exports.getDraftTimeline = getDraftTimeline;
const cleanDraftTimeline = async ({
  timelineType,
  templateTimelineId,
  templateTimelineVersion
}) => {
  let requestBody;
  const templateTimelineInfo = timelineType === _timeline.TimelineType.template ? {
    templateTimelineId,
    templateTimelineVersion
  } : {};
  try {
    requestBody = JSON.stringify({
      timelineType,
      ...templateTimelineInfo
    });
  } catch (err) {
    return Promise.reject(new Error(`Failed to stringify query: ${JSON.stringify(err)}`));
  }
  const response = await _kibana.KibanaServices.get().http.post(_constants.TIMELINE_DRAFT_URL, {
    body: requestBody,
    version: '2023-10-31'
  });
  return decodeTimelineResponse(response);
};
exports.cleanDraftTimeline = cleanDraftTimeline;
const installPrepackedTimelines = async () => {
  const response = await _kibana.KibanaServices.get().http.post(_constants.TIMELINE_PREPACKAGED_URL, {
    version: '2023-10-31'
  });
  return decodePrepackedTimelineResponse(response);
};
exports.installPrepackedTimelines = installPrepackedTimelines;
const getTimeline = async id => {
  const response = await _kibana.KibanaServices.get().http.get(_constants.TIMELINE_URL, {
    query: {
      id
    },
    version: '2023-10-31'
  });
  return decodeSingleTimelineResponse(response);
};
exports.getTimeline = getTimeline;
const resolveTimeline = async id => {
  const response = await _kibana.KibanaServices.get().http.get(_constants.TIMELINE_RESOLVE_URL, {
    query: {
      id
    },
    version: '2023-10-31'
  });
  return decodeResolvedSingleTimelineResponse(response);
};
exports.resolveTimeline = resolveTimeline;
const getTimelineTemplate = async templateTimelineId => {
  const response = await _kibana.KibanaServices.get().http.get(_constants.TIMELINE_URL, {
    query: {
      template_timeline_id: templateTimelineId
    },
    version: '2023-10-31'
  });
  return decodeSingleTimelineResponse(response);
};
exports.getTimelineTemplate = getTimelineTemplate;
const getAllTimelines = async (args, abortSignal) => {
  var _args$pageInfo, _args$pageInfo2, _args$sort, _args$sort2, _args$sort3, _args$sort4;
  const response = await _kibana.KibanaServices.get().http.fetch(_constants.TIMELINES_URL, {
    method: 'GET',
    query: {
      ...(args.onlyUserFavorite ? {
        only_user_favorite: args.onlyUserFavorite
      } : {}),
      ...(args !== null && args !== void 0 && (_args$pageInfo = args.pageInfo) !== null && _args$pageInfo !== void 0 && _args$pageInfo.pageSize ? {
        page_size: args.pageInfo.pageSize
      } : {}),
      ...(args !== null && args !== void 0 && (_args$pageInfo2 = args.pageInfo) !== null && _args$pageInfo2 !== void 0 && _args$pageInfo2.pageIndex ? {
        page_index: args.pageInfo.pageIndex
      } : {}),
      ...(args.search ? {
        search: args.search
      } : {}),
      ...(args !== null && args !== void 0 && (_args$sort = args.sort) !== null && _args$sort !== void 0 && _args$sort.sortField ? {
        sort_field: args === null || args === void 0 ? void 0 : (_args$sort2 = args.sort) === null || _args$sort2 === void 0 ? void 0 : _args$sort2.sortField
      } : {}),
      ...(args !== null && args !== void 0 && (_args$sort3 = args.sort) !== null && _args$sort3 !== void 0 && _args$sort3.sortOrder ? {
        sort_order: args === null || args === void 0 ? void 0 : (_args$sort4 = args.sort) === null || _args$sort4 === void 0 ? void 0 : _args$sort4.sortOrder
      } : {}),
      ...(args.status ? {
        status: args.status
      } : {}),
      ...(args.timelineType ? {
        timeline_type: args.timelineType
      } : {})
    },
    signal: abortSignal,
    version: '2023-10-31'
  });
  return decodeAllTimelinesResponse(response);
};
exports.getAllTimelines = getAllTimelines;
const persistFavorite = async ({
  timelineId,
  templateTimelineId,
  templateTimelineVersion,
  timelineType
}) => {
  let requestBody;
  try {
    requestBody = JSON.stringify({
      timelineId,
      templateTimelineId,
      templateTimelineVersion,
      timelineType
    });
  } catch (err) {
    return Promise.reject(new Error(`Failed to stringify query: ${JSON.stringify(err)}`));
  }
  const response = await _kibana.KibanaServices.get().http.patch(_constants.TIMELINE_FAVORITE_URL, {
    method: 'PATCH',
    body: requestBody,
    version: '2023-10-31'
  });
  return decodeResponseFavoriteTimeline(response);
};
exports.persistFavorite = persistFavorite;
const deleteTimelinesByIds = async (savedObjectIds, searchIds) => {
  let requestBody;
  try {
    if (searchIds) {
      requestBody = JSON.stringify({
        savedObjectIds,
        searchIds
      });
    } else {
      requestBody = JSON.stringify({
        savedObjectIds
      });
    }
  } catch (err) {
    return Promise.reject(new Error(`Failed to stringify query: ${JSON.stringify(err)}`));
  }
  const response = await _kibana.KibanaServices.get().http.delete(_constants.TIMELINE_URL, {
    method: 'DELETE',
    body: requestBody,
    version: '2023-10-31'
  });
  return response;
};
exports.deleteTimelinesByIds = deleteTimelinesByIds;